from ..messages import Message

import logging

logger = logging.getLogger(__name__)


class Study:
    def __init__(self, env, study_id, responsible):
        self.study_id = study_id
        self.env = env
        self.start_date = None

        if study_id in self.env.studies:
            raise ValueError(f"Study with ID {study_id} already exists.")

        self.responsible = responsible
        self.env.studies[study_id] = self

        # lifecycle flags
        if not hasattr(self, "ongoing"):
            self.ongoing = False
        if not hasattr(self, "approved"):
            self.approved = False
        self.completed = False
        self.analysed = False
        self.results = None

        # simpy events
        self.started_event = env.event()  # fires when run actually begins
        self.completed_event = env.event()  # fires when completion happens

        if self.responsible is not None:
            self.env.event().succeed(
                {
                    "type": "Study designed",
                    "Details": {
                        "study_id": self.study_id,
                        "time": self.env.now,
                        "study_type": self.__class__.__name__,
                    },
                }
            )
        env.process(self.waiting_approval())

    def waiting_approval(self):
        """
        Wait for study approval before starting the study.
        """
        reminder = (
            f"Study {self.study_id} is awaiting approval by the clinical trial Sponsor."
        )
        if not self.approved:
            self.send_reminder(reminder)
        wait_time = 0
        # Waiting for approval by Sponsor
        while not self.approved:
            yield self.env.timeout(1)
            wait_time += 1
            if wait_time % (1) == 0:
                self.send_reminder(reminder)
        self.env.process(self.waiting_start())

    def waiting_start(self):
        """
        Wait for the start of the study
        """
        reminder = f"Study {self.study_id} is approved and is waiting to be monitored by an Investigator before starting."
        if not self.ongoing:
            self.send_reminder(reminder)
        wait_time = 0
        # Waiting for the study to start
        while not self.ongoing:
            yield self.env.timeout(1)
            wait_time += 1
            if wait_time % (1) == 0:
                self.send_reminder(reminder)

    def send_reminder(self, reminder):
        """
        Send a reminder for to the study's responsible.
        """

        logger.info("reminder ")
        if self.responsible is None:
            logger.info(reminder)
        else:
            logger.info(
                "Study %s: reminder dispatched to %s. The reminder is: %s",
                self.study_id,
                getattr(self.responsible, "org_role", "unknown"),
                reminder,
            )
            self.responsible.receive_message(
                Message(
                    env=self.env,
                    content=reminder,
                    sender=self.study_id,
                    recipient=self.responsible.actor_id,
                    comm_type="async",
                )
            )

    def run_study(
        self,
    ):
        """
        Start running the study
        """
        raise NotImplementedError("Subclasses must implement this method.")

    def display_start(self):
        """
        Message to be sent when the study is starting
        """
        raise NotImplementedError("Subclasses must implement this method.")

    def display_design(self):
        """
        Message to be sent when the study is designed
        """
        raise NotImplementedError("Subclasses must implement this method.")

    def completion(self):
        """
        Handle when a study is completed
        """
        self.completed = True
        logger.info("Study %s completed at t=%.1f", self.study_id, self.env.now)
        self.send_results()

        for patient in getattr(self, "patient_list", []):
            try:
                patient.simulated = False
            except Exception:
                pass

        if self.responsible is not None:
            self.env.event().succeed(
                {
                    "type": "Study completed",
                    "Details": {
                        "study_id": self.study_id,
                        "time": self.env.now,
                        "study_type": self.__class__.__name__,
                    },
                }
            )

    def send_results(self):
        """
        Send the results of the study to the responsible.
        """

        if self.responsible is None:
            logger.info(self.display_results())
        else:
            logger.info(
                "Study %s: results dispatched to %s",
                self.study_id,
                getattr(self.responsible, "org_role", "unknown"),
            )
            logger.info("Results: %s", self.display_results())
            self.responsible.receive_message(
                Message(
                    env=self.env,
                    content=self.display_results(),
                    sender=self.study_id,
                    recipient=self.responsible.actor_id,
                    comm_type="async",
                )
            )
