import logging
from dataclasses import dataclass

_LLM_USAGE_LOGGER = logging.getLogger("LLM.usage")


@dataclass
class _Counters:
    prompt: int = 0
    completion: int = 0
    calls: int = 0


class LLMUsageTracker:
    """
    Minimal, cheap counters for policy LLM usage (no per-call I/O).
    """

    def __init__(self, name: str, actor_id: int = 0):
        self.name = name
        self.actor_id = actor_id
        self._c = _Counters()

    def log(self, messages, response, context: str = ""):
        try:
            usage = getattr(response, "usage", None)
            pt = int(getattr(usage, "prompt_tokens", 0) or 0) if usage else 0
            ct = int(getattr(usage, "completion_tokens", 0) or 0) if usage else 0
            if pt == 0:
                pt = 0
                for m in messages or []:
                    c = m.get("content") if isinstance(m, dict) else None
                    if isinstance(c, str):
                        pt += (len(c) + 3) // 4
            if ct == 0:
                ctext = ""
                try:
                    ctext = getattr(response.choices[0].message, "content", "") or ""
                except Exception:
                    pass
                if ctext:
                    ct = (len(ctext) + 3) // 4
            self._c.prompt += pt
            self._c.completion += ct
            self._c.calls += 1
        except Exception:
            pass

    def emit_totals(self):
        try:
            total = self._c.prompt + self._c.completion
            _LLM_USAGE_LOGGER.info(
                'ACTOR totals: {"actor_id":%d,"role":"%s","calls":%d,'
                '"prompt_tokens":%d,"completion_tokens":%d,"total_tokens":%d}',
                int(self.actor_id),
                f"Policy:{self.name}",
                int(self._c.calls),
                int(self._c.prompt),
                int(self._c.completion),
                int(total),
            )
        except Exception:
            pass
