import time
from typing import Sequence
import ray

from collections import namedtuple

from expground.types import Dict, List, AgentID, PolicyID, Any, Tuple
from expground.logger import Log
from expground.gt.payoff_matrix import PayoffMatrix
from expground.gt.meta_solver import MetaSolver


Identifier = namedtuple("Identifier", "combinations, key")


@ray.remote
class PayoffServer:
    def __init__(
        self,
        agents: List[AgentID],
        meta_solver: str,
        mode: str,
        rectifier_type: int = 0,
        web_client_params: Dict[str, Any] = None,
    ) -> None:
        agents = tuple(sorted(agents))

        self.agents = agents
        self.payoff_matrix = PayoffMatrix(agents=agents, rectifier_type=rectifier_type)
        self.solver = MetaSolver.from_type(meta_solver)
        self.identifier_cache: List[Any] = []
        self.client = None
        self.mode = mode  # could be one of: full_sync, semi_sync, async

    def get_payoff_matrix(self) -> PayoffMatrix:
        return self.payoff_matrix

    def set_payoff_matrix(self, payoff_matrix: PayoffMatrix):
        self.payoff_matrix = payoff_matrix
        self.identifier_cache = []

    def expand(self, supports: Dict[AgentID, Tuple[PolicyID]]):
        # expand payoff matrix with given supports, and generate identifiers
        # identifiers have three types: 1) fully sync; 2) semi-sync; and 3) async
        # - fully sync: all agents' new supports will be linked to one identifier
        # - semi-sync: supports to a certain agent will be linked to a same identifier
        # - async: one support combiation, one identifier.
        Log.debug(
            "one worker requests for expanding payoff matrix with supports: {}".format(
                supports
            )
        )

        self.payoff_matrix.expand(supports)

    def replace_or_expand(
        self, new_brs: Dict[AgentID, Any], replaced: Dict[AgentID, Any] = None
    ):
        if replaced is None:
            self.expand(new_brs)
        else:
            # replace old brs with new brs
            self.payoff_matrix.replace(new_brs, replaced)

    def get_identifier(self):
        identifier = self.identifier_cache.pop()
        Log.debug(
            "one worker requests for one identifier from payoff manager: {}".format(
                identifier
            )
        )
        return identifier

    def gen_simulations(self, split: bool = False):
        return self.payoff_matrix.gen_simulations(split)

    def update_item(self, comb_and_result_tups):
        self.payoff_matrix.update_payoff_and_simulation_status(comb_and_result_tups)

    def get_policy_distribution(self, supports: Dict[AgentID, Sequence[PolicyID]]):
        sub_matrix = self.payoff_matrix.get_sub_matrix(supports)

        eq = self.solver.solve(sub_matrix)
        policy_dist_dict = {k: dict(zip(supports[k], v)) for k, v in eq.items()}

        Log.debug(
            "a worker retrieves a dict of policy distribution: {}".format(
                policy_dist_dict
            )
        )

        return policy_dist_dict

    def get_equilibrium(
        self, population: Dict[AgentID, List[PolicyID]]
    ) -> Dict[AgentID, Dict[PolicyID, float]]:
        """Compute equilibrium with given population.

        Args:
            population (Dict[AgentID, List[PolicyID]]): A population dict.
        """

        sub_matrix = self.payoff_matrix.get_sub_matrix(population)
        eq_ndarray_dict = self.solver.solve(sub_matrix)
        equilibrium = {
            k: dict(
                zip(
                    population[k],
                    v,
                )
            )
            for k, v in eq_ndarray_dict.items()
        }
        return equilibrium
