import time

from expground.logger.client import logging_client_set, Client, ClientSet
from expground.logger.server import LoggingServerProcess


# logging in local mode
import logging
import os

from colorlog import ColoredFormatter

from expground import settings

Log = logging.getLogger("expground")
Log.setLevel(settings.LOG_LEVEL)
Log.handlers = []  # No duplicated handlers
Log.propagate = False  # workaround for duplicated logs in ipython
log_level = settings.LOG_LEVEL

stream_handler = logging.StreamHandler()
stream_handler.setLevel(log_level)

stream_formatter = ColoredFormatter(
    "%(log_color)s[%(asctime)s][%(process)05d][%(levelname)s] %(message)s",
    datefmt=None,
    reset=True,
    log_colors={
        "DEBUG": "cyan",
        "INFO": "white,bold",
        "INFOV": "cyan,bold",
        "WARNING": "yellow",
        "ERROR": "red,bold",
        "CRITICAL": "red,bg_white",
    },
    secondary_log_colors={},
    style="%",
)
stream_handler.setFormatter(stream_formatter)
Log.addHandler(stream_handler)


def init_file_logger(experiment_dir_: str):
    """Initialize a file logger with given experiment directory.

    Args:
        experiment_dir_ (str): Local experiment logging directory.
    """

    file_handler = logging.FileHandler(os.path.join(experiment_dir_, "sf_log.txt"))
    file_handler.setLevel(log_level)
    file_formatter = logging.Formatter(
        fmt="[%(asctime)s][%(process)05d] %(message)s", datefmt=None, style="%"
    )
    file_handler.setFormatter(file_formatter)
    Log.addHandler(file_handler)


def monitor(enable_time: bool = False, enable_returns: bool = True, prefix: str = ""):
    def decorator(func):
        def wrap(*args, **kwargs):
            Log.debug(f"entering func: {func.__name__}")
            # _________ enter _________
            mess = prefix
            start = time.time()
            rets = func(*args, **kwargs)
            end = time.time()
            if enable_time:
                mess += f"time consumption = {end - start}s "
            if enable_returns:
                mess += str(rets)
            # _________ exit __________
            Log.debug(mess)
            return rets

        return wrap

    return decorator
