import importlib
import pytest
import random

from expground.envs.utils import from_payoff_tables


@pytest.mark.parametrize(
    "module_path,cls,configs",
    [
        (
            "expground.envs.matrix",
            "creator",
            {
                "id": "symmetric",
                "scenario_config": {
                    "num_players": 2,
                    "payoff_type": 0,
                    "dim": 3,
                    "max_cycles": 2,
                },
            },
        )
    ],
)
def test_matrix_game_convert(module_path, cls, configs):
    Creator = getattr(importlib.import_module(module_path), cls)
    env = Creator(**configs)
    env.seed(seed=1)
    env.reset()
    payoff_matrix = env.unwrapped.payoff_matrix

    row = payoff_matrix["player_0"]
    column = payoff_matrix["player_1"]

    game = from_payoff_tables(row, column)

    # Quick test: inspect top-left utility values:
    print(
        "Values for joint action ({},{}) is {},{}".format(
            game.row_action_name(0),
            game.col_action_name(0),
            game.player_utility(0, 0, 0),
            game.player_utility(1, 0, 0),
        )
    )

    state = game.new_initial_state()

    # Print the initial state
    print("State:")
    print(str(state))

    assert state.is_simultaneous_node()

    # Simultaneous node: sample actions for all players.
    chosen_actions = [
        random.choice(state.legal_actions(pid)) for pid in range(game.num_players())
    ]
    print(
        "Chosen actions: ",
        [
            state.action_to_string(pid, action)
            for pid, action in enumerate(chosen_actions)
        ],
    )
    state.apply_actions(chosen_actions)

    assert state.is_terminal()

    # Game is now done. Print utilities for each player
    returns = state.returns()
    for pid in range(game.num_players()):
        print("Utility for player {} is {}".format(pid, returns[pid]))
