from typing import List, Dict
import pathlib
import json

from path_learning.utils import ROOT_DIR
from path_learning.utils.log import LOGDIR
from path_learning.utils.result import ExperimentSetResult
from .analysis_method import ANALYSIS_METHODS
from path_learning.utils.log import get_timestring
from analysis.utils import ANALYSIS_LOGDIR


def pick_analysis(name):
    try:
        return ANALYSIS_METHODS[name]
    except KeyError:
        raise KeyError(f"Unknown analysis method '{name}'. Must be one of {list(ANALYSIS_METHODS.keys())}")


class Analyzer:
    def __init__(self, experiment_name: str, **kwargs):
        self.kwargs: Dict = kwargs
        self.exp_set_name: str = experiment_name
        self.name: str = kwargs["name"]
        self.analysis_methods: List[Dict] = kwargs["methods"]
        self.logdir = None

    def analyze(self):
        for exp_set in self.get_experiment_sets():
            timestamp = get_timestring()
            self.logdir = self.create_analyzer_logdir(timestamp, exp_set)
            self.save_kwargs()
            for seed in exp_set.seeds:
                for method in self.analysis_methods:
                    logdir = self.create_method_logdir(seed, method)
                    constructor = pick_analysis(method["name"])
                    analysis = constructor(exp_set, seed, logdir, **method["kwargs"])
                    analysis.run()

    def save_kwargs(self) -> None:
        with open(str(self.logdir / "analyzer_kwargs.json"), "w") as fp:
            json.dump(self.kwargs, fp, indent=4)

    @staticmethod
    def create_analyzer_logdir(timestamp: str, exp_set: ExperimentSetResult) -> pathlib.Path:
        logdir = ANALYSIS_LOGDIR / f"{timestamp}_{exp_set.name}"
        logdir.mkdir(parents=True, exist_ok=False)
        return logdir

    def create_method_logdir(self, seed: int, method: Dict) -> pathlib.Path:
        method_logdir: pathlib.Path = self.logdir / f"seed_{seed}" / f"{method['name']}"
        method_logdir.mkdir(exist_ok=False, parents=True)
        return method_logdir

    def get_experiment_sets(self, check_complete_bool: bool = True) -> List[ExperimentSetResult]:
        exp_sets = []
        start_path: pathlib.Path = ROOT_DIR / LOGDIR
        # Assumes only directories are in /logs directory
        exp_set_dirs = [child for child in start_path.iterdir() if child.is_dir() and self.exp_set_name in child.stem]

        for exp_set_dir in exp_set_dirs:
            experiment_set = ExperimentSetResult(exp_set_dir, check_complete_bool=check_complete_bool)
            if experiment_set.name == self.exp_set_name:
                exp_sets.append(experiment_set)

        hashes = {idx: hash((tuple(exp_set.seeds), exp_set.name, tuple(exp_set.experiments))) for idx, exp_set in enumerate(exp_sets)}
        assert len(set(hashes.values())) == len(hashes), f"Found multiple runs with the same name and the same seeds."

        return exp_sets
