import json
import pathlib
import logging
import argparse

from analysis.analyzer import Analyzer
from analysis.digester import AnalyzerDigestResult
from analysis.plotter import Plotter
from analysis.utils import ANALYSIS_LOGDIR
from analysis.gather_results import AnalyzerResult
from path_learning.utils import init_logging, project_path


def run_analysis(experiment_name: str, analysis_config: pathlib.Path):
    assert experiment_name is not None, "Choose a valid experiment name"
    with open(str(analysis_config), "r") as analysis_fp:
        config = json.load(analysis_fp)

    analyzer = Analyzer(experiment_name, **config)
    analyzer.analyze()


def run_digest_analysis(experiment_name: str, analysis_config: pathlib.Path):
    assert experiment_name is not None, "Choose a valid experiment name"
    with open(str(analysis_config), "r") as analysis_fp:
        config = json.load(analysis_fp)

    analyzer = AnalyzerDigestResult(experiment_name, **config)
    analyzer.analyze()


def upload_results(experiment_names: str, plot_conf: pathlib.Path):
    result_dirs = []
    experiment_name_list = experiment_names.split(",")
    configs = {}
    for experiment_name in experiment_name_list:
        logging.info(f"uploading all results for experiment {experiment_name}")
        with open(str(plot_conf), "r") as analysis_fp:
            configs[experiment_name] = json.load(analysis_fp)
        result_dirs.extend([result_dir for result_dir in ANALYSIS_LOGDIR.iterdir() if result_dir.is_dir()
                       and experiment_name in result_dir.stem])

    result_df = None
    for result_dir in list(set(result_dirs)):
        result = AnalyzerResult(result_dir)
        if result_df is None:
            result_df = result.get_dataframe()
        else:
            result_df = result_df.append(result.get_dataframe(), ignore_index=True)
    print(result_df.shape)
    plotter = Plotter(result_df, experiment_name_list, **configs)
    plotter.plot()


if __name__ == '__main__':
    init_logging()

    parser = argparse.ArgumentParser(description='Run or Upload Analysis')
    parser.add_argument('--upload', "-u")
    parser.add_argument('--run', "-r")
    parser.add_argument('--digest', "-d")

    args = parser.parse_args()

    if args.run:
        analysis_conf = project_path("analysis/configs/invariance_analysis_config.json")
        logging.info(f"Using analysis config {analysis_conf}")

        logging.warning(f"Analyzing experiment {args.run} ...")
        run_analysis(args.run, analysis_conf)
    elif args.upload:
        plot_conf = project_path("analysis/configs/plotting.json")
        upload_results(args.upload, plot_conf)
    elif args.digest:
        digest_conf = project_path("analysis/digest_configs/test_digest_analysis.json")
        logging.info(f"Using digest analysis config {digest_conf}")

        logging.warning(f"Analyzing digest from experiment {args.digest} ...")
        run_digest_analysis(args.digest, digest_conf)
    else:
        ValueError("Choose valid analysis setting from 'run' (-r), "
                   "'upload' (-u), and 'digest' (-d).")





