"""
    General checker to check completion of subtasks.
    Also checks coverage of the task and success of the task.
"""

class BaseChecker:
    def __init__(
            self,
            subtasks,
            conditional_subtasks,
            independent_subtasks,
            coverage,
            interact_objects,
            interact_receptacles
            ) -> None:

        self.subtasks = subtasks

        # subtasks that are conditional on some other subtasks/state
        self.conditional_subtasks = conditional_subtasks

        # subtasks that are not conditional on any other subtasks/state
        # basically depend on the object in the inventory (either before or after the action)
        self.independent_subtasks = independent_subtasks

        # objects or receptacles
        self.coverage = coverage

        # objects to be interacted with
        self.interact_objects = interact_objects

        # receptacles to be used
        self.interact_receptacles = interact_receptacles

        # all actions in conditional subtask
        self.init_conditional_acts()

        self.subtasks_completed = []
        self.coverage_completed = []
        # to store the subtasks completed with object number (@coela)
        self.subtasks_completed_numerated = []

    def split_action(self, action: str):
        """Split action string into action and object"""
        act = action.split("(")[0]
        object = action.split("(")[1].split(")")[0]
        return act, object

    def denumerate_action(self, action: str):
        """
        Remove object number from object_id
        Example: NavigateTo(Bread_1) -> NavigateTo(Bread)
        """
        act, object = self.split_action(action)
        object_name = self.denumerate_object(object)
        return f"{act}({object_name})"

    def denumerate_object(self, object: str) -> str:
        """
        Remove object number from object_id
        Example: Bread_1 -> Bread
        """
        if "_" in object:
            object_name, object_id = object.split("_")
            return object_name
        else:
            return object

    def init_conditional_acts(self):
        conditional_acts = set()
        for action in self.conditional_subtasks:
            act = action.split("(")[0]
            conditional_acts.add(act)
        self.conditional_acts = list(conditional_acts)

    def give_credit_for_navigate(self, action, success, inventory_object):
        act, object = self.split_action(action)
        if act not in ["NavigateTo"]:
            navigate_action = f"NavigateTo({object})"
            self.check_subtask(navigate_action, success, inventory_object)

    def check_subtask(
        self,
        action,
        success,
        inventory_object,  # object in inventory after the action
    ):
        if action in ["Done", "Idle"]:
            return None

        if "SendMessage" in action:
            return None

        denum_act = self.denumerate_action(action)

        # first check for independent subtasks
        # if the action is in independent subtasks and not in subtasks_completed
        # and the action was successful, then add it to subtasks_completed
        if (
            denum_act in self.independent_subtasks
            and denum_act not in self.subtasks_completed
            and success
        ):
            self.subtasks_completed.append(self.denumerate_action(action))
            self.subtasks_completed_numerated.append(action)

            # Give credit for NavigateTo(object) if action(object) is successful
            # Since this means the agent just happened to already be close enough
            self.give_credit_for_navigate(action, success, inventory_object)

        elif self.conditional_subtasks is not None:
            # elif we need to check for conditional subtasks
            # first check for NavigateTo(Fridge, object_in_inventory)
            # TODO: @nsidn98 check this reasoning:
            # since we are checking after put object is executed,
            # we need to check in previous inventory
            for receptacle in self.interact_receptacles:
                for act in self.conditional_acts:
                    if (
                        denum_act in [f"{act}({receptacle})"]
                        and success
                        and self.denumerate_object(inventory_object) in self.interact_objects
                    ):
                        subtask_attempted = (
                            f"{act}({receptacle}, {self.denumerate_object(inventory_object)})"
                        )
                        if subtask_attempted not in self.subtasks_completed:
                            self.subtasks_completed.append(subtask_attempted)
                            act_receptacle = action.split(")")[0]
                            action_object = f"{act_receptacle}, {inventory_object})"
                            self.subtasks_completed_numerated.append(action_object)

                            # Give credit for NavigateTo(interact_receptacle, object_in_inventory) if action(interact_receptacle, object_in_inventory) is successful
                            # Since this means the agent just happened to already be close enough
                            self.give_credit_for_navigate(action, success, inventory_object)

    def get_transport_rate(self):
        return len(self.subtasks_completed) / len(self.subtasks)

    def check_coverage(self, action: str):
        for object in self.coverage:
            if object in action and object not in self.coverage_completed:
                self.coverage_completed.append(object)

    def perform_metric_check(self, action, success, inventory_object):
        self.check_subtask(action, success, inventory_object)
        self.check_coverage(action)

    def get_coverage(self):
        return len(self.coverage_completed) / len(self.coverage)

    def check_success(self):
        return len(self.subtasks_completed) == len(self.subtasks)
