import argparse
import inspect
import importlib
import re
import os, sys
from pathlib import Path

# set parent directory to address relative imports
directory = Path(os.getcwd()).absolute()
sys.path.append(
    str(directory)
)  # note: no ".parent" addition is needed for python (.py) files

from AI2Thor.env_new import AI2ThorEnv
from thortils.navigation import find_navigation_plan, get_shortest_path_to_object
from AI2Thor.baselines.utils import Logger, AutoConfig

"""
Initialization for the environments (task and room dependent):

This whole thing is just a way to automate the process of writing init functions...
"""

# default config
auto = AutoConfig()
auto.set_task(0)
auto.set_floorplan(0)
config = auto.config()

def all_objects():
    return env.controller.last_event.metadata["objects"]

def compare(a,b):
    # bigger name, name
    # compare string for objects (true if 'equal')
    a,b=a.lower().strip(), b.lower().strip()
    return b in a

def get_id(name, ignore=False, exclusive=True):
    """
    Gets the id for object with name 'name'
    - name - name of object (e.g. 'tomato')
    - ignore - if True, give a list of all the ids with the indicated name. 
           If False, assert that there is only 1 applicable id and output that
    - exclusive - if True, be stricter about name (butterknife becomes different than knife)
                    if you want all knives (regardless) set this to False and ignore to True

    e.g. get_id('knife', ignore=True, exclusive=True) -> will return list of 'knife' objects only (NO 'butterknife')
    """
    ids=[]
    for o in all_objects():
        if compare(o['objectId'].split('|')[0], name):
            # symmetric process (remove extraneous things)
            if exclusive and not compare(name, o['objectId'].split('|')[0]):
                continue
            ids.append(o['objectId'])
    assert len(ids)>0, f'object with name {name} not found'
    if ignore==False:
        assert len(ids)==1, f'more than 1 object with name {name} found (get_id): {ids}. If this is ok, change to \'ignore=True\''
        return ids[0]
    return ids

def get_size():
    """
    Get the x,y,z size of the current scene
    output is dictionary with keys 'x','y','z'
    """
    return env.event.metadata['sceneBounds']['size']

def get_object(obj_name, ignore=False, exclusive=True):
    """
    Get the object metadata for obj with name 'obj_name', NOT with id 'obj_name'
        e.g. get_object('apple')
    See docs above for ignore & exclusive meaning
    """
    objs=[]
    for o in all_objects():
        if compare(o['objectId'].split('|')[0], obj_name):
            if exclusive and not compare(obj_name, o['objectId'].split('|')[0]):
                continue
            objs.append(o)
    if ignore==False:
        assert len(objs)==1, f'more than 1 object with name {name} found (get_object): {ids}. If this is ok, change to \'ignore=True\''
        return objs[0] 
    return objs

def get_position(obj_name, ignore=False, exclusive=True):
    """
    Get position of object, see docs above for ignore & exclusive meaning
    """
    return get_object(obj_name,ignore=ignore, exclusive=exclusive)['position']

def offset_position(obj_name,pos):
    """
    Given object w/ name obj_name, add the offset (x,y,z) from array 'pos'
    and return the offset-ed position as a dict

    x,y,z correspondance in the plot:
    -x : horizontal motion
    -y : out of plane motion (height of object)
    -z : vertical motion

    When setting position, one can put the height 'y' above a table and it'll fall down (no need to get exact)
    """
    x,y,z=pos
    obj_position=get_position(obj_name)
    x+=obj_position['x']
    y+=obj_position['y']
    z+=obj_position['z']
    return dict(zip(obj_position.keys(), [x,y,z]))


def abs_offset(prop):
    """
    Given proportion array (inputs from [0,1]), 
    get corresponding delta change in absolute distances
    e.g. abs_offset([.1, .1, .1]) -> array of 10% of size of room in x,y,z
    """
    sizes=get_size()
    return [v*prop[i] for i,v in enumerate(sizes.values())]

# helper function
def fill_array(v):
    """
    Fill x,y,z array with value 'v'
    """
    return [v]*3

def overhead():
    """ Get overhead view """
    env.controller.step(action='ToggleMapView')
    
# extract number from string ('FloorPlan201' -> 201)
extract_number=lambda s: re.findall(r'\d+',s)[0]



def write_meta(file, fn_name, place_objects=None, actions=None, forceAction=True):
    """
    This is the main function that creates the python file with the auto-initialized code.
    arguments:
        -file - name of file to output it in (without .py)
        -fn_name - name of the function within that file (happens to be same as file name)
        -place_objects (dictionary (w/ keys) object_id and (w/ values) position dict(output from offset_position(...))
            -e.g. {'Apple|1|1|1' : [1.23, 2.21, .54]}
        -actions (dict (w/ keys) action name and (w/ values) obj id *list* to apply that action)
            -e.g. {'SliceObject': ['Apple|1|1|1', 'Tomato|1|1|1'], 'OpenObject': ['Fridge|1|1|1']}
        -forceAction - boolean value whether to force action or not
    """
    path=fn_name
    path+='.py'
    h,t=os.path.split(path) # -> 'init_maker/fns/{deeper}/', 'fn_name.py'
    subdirs=h.split('/') # -> ['init_maker', 'fns', {deeper}]

    with open(path, 'w') as f:
        s=f'def {t[:-3]}(self, env, controller):\n'
        auto='# initialization function - autogenerated\n'
        s+="""
    \"""Pre-initialize the environment for the task.

    Args:
        event: env.event object
        controller: ai2thor.controller object

    Returns:
        event: env.event object
    \"""\n
        """
        s+=auto # because why not

        if place_objects is not None:
            for (_id, pos) in (place_objects.items()):
                s+="""
    event=controller.step(
    action=\'PlaceObjectAtPoint\',
    objectId=\'{_id}\',
    position={pos}
    )
                """.format(_id=_id, pos=pos)

        if actions is not None:
            for actn, _ids in (actions.items()):
                for _id in _ids:
                    s+="""
    event=controller.step(
    action=\'{actn}\',
    objectId=\'{_id}\',
    forceAction={forceAction}
    )
                    """.format(actn=actn, _id=_id, forceAction=forceAction)

        s+="""
    return event
        """
        f.write(s)

# run
parser=argparse.ArgumentParser()
# arguments to parse 
# - scene (put the name of floorplan)
# - fn_n (put the name of the function to create preinit for: NOTE without the _r1 appendage, will choose correct floorplan automatically)
# - render (add --render at the end to show plot of overhead view of what the preinit fn did)
parser.add_argument('--scene', type=str, default='FloorPlan1')
parser.add_argument('--fn_n', type=str, default='preinit_sink')
parser.add_argument('--render', action='store_true')
args=parser.parse_args()

config.scene=args.scene
config.scene_name=args.scene
print('initializing environment...')
# initialise the environment
env = AI2ThorEnv(config)
# reset the environment with a new task - task preinit doesn't matter
d = env.reset(task='put bread, lettuce, tomato in fridge')
print('initialization done.')


# @change - create your own preinit_{name}_r{floorplan_number}
#   make sure the last number is {floorplan_number}
def preinit_sink_r1():
    # @change - put the names of the objects being moved here
    names=['butterknife', 'bowl', 'mug']

    # ---- don't change ----
    fn_name=inspect.currentframe().f_code.co_name
    deeper=dpr(fn_name)
    fn_name=f'init_maker/fns/{deeper}/'+fn_name
    # ---- don't change ----

    # @change - establish the offset proportions (in this case all zeros, no change needed)
    #       must be in same order as names array
    off=[\
        fill_array(0), fill_array(0), fill_array(0)
        ]
    po=dict([(get_id(n), offset_position(n, abs_offset(off))) for off,n in zip(off, names)])

    # @change - add actions instead of 'None' below if needed - see fn below
    # actions are applied to an array of id(s) (see 'write_meta' for further documentation), 
    #           so put a list of ids there (use get_id function to get 1 id or a list of id w/ the same name)
    actions={'CloseObject':get_id('cabinet', ignore=True)} # this will close all the cabinets, as get_id will return list!

    # ---- don't change ----
    write_meta(fn_name=fn_name, file=fn_name, place_objects=po, actions=actions)
    # ---- don't change ----

# Add any extra functions here
##
##
##

# @change
# Add mapping dictionary from key_word->tsk_directory
# this is so that the code knows which directory each function goes - the key_word should be unique to the function
tsk_dict={'sink' : '1_put_knife_bowl_mug_sink'}

if not os.path.exists('init_maker/fns'):
    os.makedirs('init_maker/fns')

for t in tsk_dict.values():
    p=f'init_maker/fns/{t}'
    # create task_dir if it doesn't exist
    if not os.path.exists(p):
        os.makedirs(p)

# get which task folder for which function in this file (helper fn)
def dpr(fn_n):
    # unique words that are in each function (position here aligns directly with folder name above)
    for key_word, tsk_dir in tsk_dict.items():
        if key_word in fn_n:
            return tsk_dict[key_word]
    print(f'function with name {fn_n} not found')

# folder deeper to find function
fn_n=args.fn_n
r=f'_r{extract_number(args.scene)}'
deeper=dpr(fn_n)
fn=fn_n+r
# run preinit function on this file
globals()[fn]()
m=importlib.import_module(f'init_maker.fns.{deeper}.{fn}')
getattr(m,fn)(None, env, env.controller)
print(f'function {fn} being run')
# setup env w/ overhead view
overhead()

# render if necessary
if args.render:
    import cv2
    import matplotlib.pyplot as plt
    img=env.controller.last_event.cv2img
    plt.imshow(cv2.cvtColor(img, cv2.COLOR_BGR2RGB))
    plt.show()
    

