import torch, numpy as np
import open3d as o3d

def estimate_voxel_size_from_points_tensor(points: torch.Tensor,
                                           nb_neighbors=20, std_ratio=2.0,
                                           cand_steps=25, elbow_ratio_range=(0.5, 2.5)):
    """
    points: [N,3] torch.Tensor, 单位应为米（或至少统一单位）
    返回：最近邻中位距、肘部估计、建议体素大小、扫描曲线等
    """
    assert points.ndim == 2 and points.shape[1] == 3, "points 需为 [N,3]"
    # 去 NaN/Inf 和重复点（可选）
    p = points
    mask = np.isfinite(p).all(axis=1)
    p = p[mask]
    # 去重复（可选）
    if len(p) > 0:
        p = np.unique(p, axis=0)

    assert len(p) > 100, f"点太少：{len(p)}"

    # Open3D point cloud
    pcd = o3d.geometry.PointCloud(o3d.utility.Vector3dVector(p))

    # 统计滤波（去离群）
    pcd, ind = pcd.remove_statistical_outlier(nb_neighbors=nb_neighbors, std_ratio=std_ratio)
    pts = np.asarray(pcd.points)
    assert len(pts) > 100, "滤波后点太少，调小 std_ratio 或 nb_neighbors 试试"

    # 最近邻距离 —— 注意跳过自身：用 k=2 取第2近
    kdt = o3d.geometry.KDTreeFlann(pcd)
    dists = np.empty(len(pts), dtype=np.float32)
    for i, x in enumerate(pts):
        _, idx, dist2 = kdt.search_knn_vector_3d(x, 2)
        dists[i] = np.sqrt(dist2[1]) if len(idx) > 1 else np.nan
    dists = dists[np.isfinite(dists)]

    # 剪掉上下1%离群点，取中位数作为“典型点间距”
    lo, hi = np.percentile(dists, [1, 99])
    core = dists[(dists >= lo) & (dists <= hi)]
    d_med = float(np.median(core))

    # 扫描候选体素尺寸，统计“占据体素数”（越密→体素多；出现“肘部”）
    v_min = max(1e-9, elbow_ratio_range[0] * d_med)
    v_max = elbow_ratio_range[1] * d_med
    cand = np.linspace(v_min, v_max, cand_steps)

    # 用 Open3D 的 voxel_down_sample 计算占据体素的代表点数
    occ = []
    for v in cand:
        ds = pcd.voxel_down_sample(v)
        occ.append(len(ds.points))
    occ = np.asarray(occ, dtype=np.int64)

    # 简单“肘部检测”：二阶差分绝对值最大的点
    d1 = np.gradient(occ.astype(np.float64))
    d2 = np.gradient(d1)
    elbow_idx = int(np.argmax(np.abs(d2)))
    v_elbow = float(cand[elbow_idx])

    # 建议：用肘部估计；若想更保守（稍密一点），可用 0.9×v_elbow
    suggested = v_elbow

    return {
        "NN_median_spacing": d_med,          # 最近邻中位距（米）
        "voxel_size_elbow": v_elbow,         # 肘部估计的体素边长（米）
        "suggested_voxel_size": suggested,   # 建议体素（米）
        "scan_candidates": cand,             # 扫描的候选体素（米）
        "scan_occupied_counts": occ          # 各候选体素下的占据体素数
    }

# ================== 用法示例 ==================
import torch
for i in range(30):
    print("================== Region", i, "================")
    file_path = f"../openscene/data/matterport_3d/test/UwV83HsGsw3_region{i}.pth"
    points, colors, instance_labels = torch.load(file_path, map_location='cpu', weights_only=False)

    res = estimate_voxel_size_from_points_tensor(points)
    print({k: (v if not isinstance(v, np.ndarray) else f"array(len={len(v)})")
        for k, v in res.items()})

