"""
Viewpopint class to represent a viewpoint in a HOV-SGraph.
"""

import json
import os
from collections import defaultdict
from typing import Any, List

import numpy as np
import open3d as o3d

from hovsg.utils.clip_utils import get_img_feats, get_text_feats_multiple_templates

class Viewpoint:
    """
    Class to represent a viewpoint in a building.
    :param vp_id: Unique identifier for the viewpoint
    :param floor_id: Identifier of the floor this vp belongs to
    :param name: Name of the vp (e.g., "Living Room", "Bedroom")
    """
    def __init__(self, vp_id, floor_id, name=None):
        self.vp_id = vp_id  # Unique identifier for the vp
        self.name = name  # Name of the vp (e.g., "Living Room", "Bedroom")
        self.floor_id = floor_id  # Identifier of the floor this vp belongs to
        self.objects = []  # List of objects that can be observed  in the vp
        self.embeddings = []  # List of tensors of embeddings of the room
        self.pcd = None  # Point cloud of the room
        self.vp_pos = None  # Position of the viewpoint
        self.vp_images = []  # observed images from the viewpoint

    def add_object(self, objectt):
        """
        Method to add objects to the room
        :param objectt: Object object to be added to the room
        """
        self.objects.append(objectt)  # Method to add objects to the room

    def set_txt_embeddings(self, text):
        self.embeddings.append(get_text_feats_multiple_templates(text))

    def infer_room_type_from_view_embedding(
        self,
        default_room_types: List[str],
        clip_model: Any,
        clip_feat_dim: int,
    ) -> str:
        """Use the embeddings stored inside the room to infer room type. We should already
            save k views CLIP embeddings for each room. We match the k embeddings with room
            types' textual CLIP embeddings to get a room label for each of the k views. Then
            we count which room type has the most votes and return that.

        Args:
            default_room_types (List[str]): the output room type should only be a room type from the list.
            clip_model (Any): when the generate_method is set to "embedding", a clip model needs to be
                                provided to the method.
            clip_feat_dim (int): when the generate_method is set to "embedding", the clip features dimension
                                    needs to be provided to this method

        Returns:
            str: a room type from the default_room_types list
        """
        if len(self.embeddings) == 0:
            print("empty embeddings")
            return "unknown room type"
        text_feats = get_text_feats_multiple_templates(default_room_types, clip_model, clip_feat_dim)
        embeddings = np.array(self.embeddings)
        sim_mat = np.dot(embeddings, text_feats.T)
        # sim_mat = compute_similarity(embeddings, text_feats)
        # print(sim_mat)
        col_ids = np.argmax(sim_mat, axis=1)
        votes = [default_room_types[i] for i in col_ids]
        # print(f"the votes are: {votes}")
        unique, counts = np.unique(col_ids, return_counts=True)
        unique_id = np.argmax(counts)
        type_id = unique[unique_id]
        self.name = default_room_types[type_id]
        # print(f"The room view ids are {self.represent_images}")
        # print(f"The room type is {default_room_types[type_id]}")
        return default_room_types[type_id]

    def save(self, path):
        """
        Save the vp in folder as ply for the point cloud
        and json for the metadata
        """
        # save the point cloud
        o3d.io.write_point_cloud(os.path.join(path, str(self.vp_id) + ".ply"), self.pcd)
        # save the metadata
        metadata = {
            "vp_id": self.vp_id,
            "name": self.name,
            "floor_id": self.floor_id,
            "objects": [obj.object_id for obj in self.objects],
            "vp_pos": self.vp_pos.tolist(),
            "embeddings": [i.tolist() for i in self.embeddings],
            "vp_images": self.vp_images,
        }
        with open(os.path.join(path, str(self.vp_id) + ".json"), "w") as outfile:
            json.dump(metadata, outfile)

    def load(self, path):
        """
        Load the room from folder as ply for the point cloud
        and json for the metadata
        """
        # load the point cloud
        self.pcd = o3d.io.read_point_cloud(os.path.join(path, str(self.vp_id) + ".ply"))
        # load the metadata
        with open(path + "/" + str(self.vp_id) + ".json") as json_file:
            metadata = json.load(json_file)
            self.vp_id = metadata["vp_id"]
            self.name = metadata["name"]
            self.floor_id = metadata["floor_id"]
            self.objects = metadata["objects"]
            self.vp_pos = metadata["vp_pos"]
            self.embeddings = [np.asarray(i) for i in metadata["embeddings"]]
            self.vp_images = metadata["vp_images"]

    def __str__(self):
        return f"Viewpoint ID: {self.vp_id}, Position:{self.vp_pos}, Name: {self.name}, Floor ID: {self.floor_id}, Objects: {len(self.objects)}"
