import math
import torch

Rastrigin = lambda x1,x2: 20 + (x1**2-10*math.cos(2*math.Pi*x1)) + (x2**2-10*math.cos(2*math.Pi*x2))
Rosenbrock = lambda x1,x2: (8 * abs(x1**2 - x2) + (1 - x1)**2)
Himmelblau = lambda x1,x2: (x1**2+x2-11)**2+(x1+x2**2-7)**2
McCormick = lambda x1,x2: math.sin(x1+x2)+(x1-x2)**2-1.5*x1+2.5*x2+1
StyblinskiTang = lambda x1,x2: 0.5*(x1**4-16*x1**2+5*x1+x2**4-16*x2**2+5*x2)

class SimpleReality():

    def __init__(self, device, local_change = 1.0, noise = 0.0, version = 0, a=0.4):

        self.device = device

        self.local_change = local_change

        if version==0:
            self.f2 = lambda t,x1,x2: self.local_change * math.sin(math.sin(30*t)*x1+math.cos(30*t)*x2+1) + McCormick(x1,x2)
        if version==1: # seasonal
            self.f2 = lambda t,x1,x2: 0.1*McCormick(
                0.5*math.cos(0.5*math.sin(self.local_change*t/10))*x1-0.5*math.sin(0.5*math.sin(self.local_change*t/10))*x2,
                0.5*math.sin(0.5*math.sin(self.local_change*t/10))*x1+0.5*math.cos(0.5*math.sin(self.local_change*t/10))*x2)-1
        if version==2: # drift
            self.f2 = lambda t,x1,x2: 0.001*((2+math.sin(a*t))*t+1)*(Rosenbrock(x1,x2)-25+0.1*t)

        self.noise_mean = torch.tensor(0.0, device=self.device)
        self.noise_std = torch.tensor(noise, device=self.device)

        self.reality = lambda t,x1,x2: torch.tensor([self.f2(t,x1,x2)], device=self.device)
        self.num_outputs = 1
        self.num_inputs = 2

        self.xmin = torch.tensor( [-4.0, -4.0], device=self.device)
        self.xmax = torch.tensor( [4.0, 4.0], device=self.device)
        self.xdim = self.xmin.shape[0]
        self.xdiff = self.xmax-self.xmin
        self.safe_min = torch.tensor([-0.5, -1.0], device=self.device)
        self.safe_max = torch.tensor([0.5, 1.0], device=self.device)

        self.current_time = -1.0

        self.xdata = None 
        self.ydata = None
        self.fdata = None

    def next_point(self, x1, x2):
        self.current_time = self.current_time + 1.0

        newx = torch.tensor([[self.current_time, x1, x2]], device=self.device)
        if self.xdata == None:
            self.xdata = newx
        else:
            self.xdata = torch.cat((self.xdata,newx), axis=0)

        newf = self.reality(self.current_time, x1, x2).unsqueeze(1)
        if self.fdata == None:
            self.fdata = newf
        else:
            self.fdata = torch.cat((self.fdata,newf), axis=0)
        newy = newf + torch.normal(mean=self.noise_mean, std=self.noise_std, size=(1,self.num_outputs), device=self.device)
        if self.ydata == None:
            self.ydata = newy
        else:
            self.ydata = torch.cat((self.ydata,newy), axis=0)

        return newy
    
class RailPressureDynamicReality():

    def __init__(self, device, noise=0.02, local_change=0.0):

        self.device = device

        self.noise_mean = torch.tensor(0.0, device=self.device)
        self.noise_std = torch.tensor(noise, device=self.device)

        self.f = lambda t,x1,x2,x3,x4,x5,x6,x7,x8: RailPressure(torch.tensor([x1,x2,x3,x4,x5,x6,x8,0.7,0.7,0.7] ,device=self.device))

        self.reality = lambda t,x1,x2: torch.tensor([[self.f(t,x1,x1,x1,x1,x2,x2,x2,x2)]], device=self.device)
        self.num_outputs = 1
        self.num_inputs = 2

        self.xmin = torch.tensor( [-1.560584088, -1.762384787 ], device=self.device)
        self.xmax = torch.tensor( [2.174939070, 1.004642456 ], device=self.device)
        self.safe_min = torch.tensor([-0.2, -1.1], device=self.device)
        self.safe_max = torch.tensor([+0.2, -0.7], device=self.device)
        self.xdim = self.xmin.shape[0]
        self.xdiff = self.xmax-self.xmin

        self.current_time = -1.0

        self.xdata = None
        self.ydata = None
        self.fdata = None

    def next_point(self, x1, x2):
        self.current_time = self.current_time + 1.0

        newx = [self.current_time, x1, x1, x1, x1, x2, x2, x2, x2]
        if self.xdata != None and self.xdata.shape[0]>0:
            newx[2] = self.xdata[-1,1]
            newx[3] = self.xdata[-1,2]
            newx[4] = self.xdata[-1,3]
            newx[6] = self.xdata[-1,5]
            newx[7] = self.xdata[-1,6]
            newx[8] = self.xdata[-1,7]
        newx = torch.tensor([newx], device=self.device)
        if self.xdata == None:
            self.xdata = newx
        else:
            self.xdata = torch.cat((self.xdata,newx), axis=0)

        newf = torch.tensor([[self.f(self.current_time, self.xdata[-1,1], self.xdata[-1,2], self.xdata[-1,3], self.xdata[-1,4], self.xdata[-1,5], self.xdata[-1,6], self.xdata[-1,7], self.xdata[-1,8])]], device=self.device)
        if (newf.shape)==1:
            newf = newf.unsqueeze(1)
        if self.fdata == None:
            self.fdata = newf
        else:
            self.fdata = torch.cat((self.fdata,newf), axis=0)
        newy = newf + torch.normal(mean=self.noise_mean, std=self.noise_std, size=(1,self.num_outputs), device=self.device)
        if self.ydata == None:
            self.ydata = newy
        else:
            self.ydata = torch.cat((self.ydata,newy), axis=0)

        return newy
    
    def remove_last_point(self):
        self.current_time = self.current_time - 1.0
        self.xdata = self.xdata[:-1,:]
        self.fdata = self.fdata[:-1,:]
        self.ydata = self.ydata[:-1,:]

def RailPressureStatic(xIn):
    x = torch.tensor([xIn[0], xIn[0], xIn[0], xIn[0], xIn[1], xIn[1], xIn[1], xIn[2], xIn[2], xIn[2]], device=xIn.device)
    return RailPressure(x)

#  xIn = [nmot(k);
#         nmot(k-1);
#         nmot(k-2);
#         nmot(k-3);
#         vgeshdp(k);
#         vgeshdp(k-1);
#         vgeshdp(k-3);
#         ti(k);
#         ti(k-2);
#         ti(k-3)];
#  nmot:    engine speed (1/min)
#  vgeshdp: fuel pump actuation (mm³)
#  ti:      injection time (ms)
#  y:       predicted pressure at time step k, should be at most 18
def RailPressure(x):
    y0 = RailPressureInternal(x)
    # y=1-torch.exp((y0-18.0)/10.0) # version from Zimmer et al. 2018, we work in orignal scaled coordinates instead
    y = y0
    return y

def RailPressureInternal(xIn):
    y = 0
    x = torch.zeros(10, device=xIn.device)
    # x[0] = (xIn[0] - 2253.3056464309)/803.10036183397
    # x[1] = (xIn[1] - 2253.3096911602)/803.0960165043
    # x[2] = (xIn[2] - 2253.3137973002)/803.09162233565
    # x[3] = (xIn[3] - 2253.3179648508)/803.08718748759
    # x[4] = (xIn[4] - 38.215389287748)/21.683961765866
    # x[5] = (xIn[5] - 38.215412400486)/21.68392095213
    # x[6] = (xIn[6] - 38.215460803251)/21.683841783813
    # x[7] = (xIn[7] - 0.70942737910475)/0.53560619117677
    # x[8] = (xIn[8] - 0.70941240606961)/0.53561824817519
    # x[9] = (xIn[9] - 0.70940475206843)/0.5356235081677
    x = xIn
    fterm = x[0]* -0.81905999667818 + x[1]* -0.019794720460163 + x[2]* 0.27304841361022 + x[3]* -0.73207404858355 + x[4]* -3.7029124975669 + x[5]* -0.046790795114252 + x[6]* 2.2697846410235 + x[7]* 0.25841712144651 + x[8]* -0.28806898337401 + x[9]* 0.3373995160564
    y=y+ 0.031692908952064 * torch.cos(fterm) + 0.068402277004024 * torch.sin(fterm)
    fterm = x[0]* 0.33752150846961 + x[1]* 0.22665058943887 + x[2]* 0.0036308221642449 + x[3]* 1.0767220185346 + x[4]* 3.2393608709891 + x[5]* 0.19726557402639 + x[6]* -4.0903224279171 + x[7]* 0.33875023498879 + x[8]* -0.027486115217298 + x[9]* 1.7015867079337
    y=y+ 0.0024380870537466 * torch.cos(fterm) + 0.026928849707532 * torch.sin(fterm)
    fterm = x[0]* 0.095036088014746 + x[1]* 0.22714724983666 + x[2]* -0.252352037575 + x[3]* -0.18437125084529 + x[4]* -0.96603866774145 + x[5]* -0.025618018257457 + x[6]* -3.2018494389618 + x[7]* 0.33880919065233 + x[8]* 0.15883552468638 + x[9]* 1.7038404546954
    y=y+ 0.03593989016575 * torch.cos(fterm) + -0.061165600194329 * torch.sin(fterm)
    fterm = x[0]* -0.30750892970496 + x[1]* -0.23891396825039 + x[2]* -0.44176279406099 + x[3]* 0.98871809597707 + x[4]* -5.0299200351586 + x[5]* 0.20496686667996 + x[6]* 1.0113286004103 + x[7]* -0.89964906559001 + x[8]* -1.6251295055196 + x[9]* -1.1594042648733
    y=y+ 0.016985328578553 * torch.cos(fterm) + -0.062906411043536 * torch.sin(fterm)
    fterm = x[0]* 0.54473897047707 + x[1]* 0.19648071480935 + x[2]* -0.80191433634961 + x[3]* -0.31176920243484 + x[4]* 0.48317144030955 + x[5]* 0.34539198746364 + x[6]* 6.3642693877505 + x[7]* 1.6686848402237 + x[8]* 0.50810563186527 + x[9]* 1.6627586231958
    y=y+ -0.010472575984009 * torch.cos(fterm) + -0.034856454748811 * torch.sin(fterm)
    fterm = x[0]* -0.90932965603186 + x[1]* 0.0040861403775149 + x[2]* -0.18367251852817 + x[3]* -0.8384362134349 + x[4]* 1.1095932478839 + x[5]* 0.33627852300982 + x[6]* 0.80578324978802 + x[7]* -0.11488487559319 + x[8]* -0.70000840599637 + x[9]* 0.50949820156166
    y=y+ 0.08771158424235 * torch.cos(fterm) + -0.056379754715369 * torch.sin(fterm)
    fterm = x[0]* -0.34681438881798 + x[1]* 0.043514921077717 + x[2]* 0.12046630595345 + x[3]* -0.11064217792073 + x[4]* 12.409726448748 + x[5]* 0.12883850899569 + x[6]* -13.632000414936 + x[7]* -0.19971464610846 + x[8]* 0.046228428472665 + x[9]* 0.12515304947418
    y=y+ -0.055589202591183 * torch.cos(fterm) + -0.17750376299084 * torch.sin(fterm)
    fterm = x[0]* 0.5126466445729 + x[1]* 0.12860832423111 + x[2]* -0.21064805610633 + x[3]* -0.50620018378431 + x[4]* 5.236695403519 + x[5]* -0.24441406130094 + x[6]* -4.3045949100653 + x[7]* 0.61624905468049 + x[8]* 2.1138749574221 + x[9]* -0.32086813690263
    y=y+ -0.018944359148021 * torch.cos(fterm) + -0.024568581080264 * torch.sin(fterm)
    fterm = x[0]* -0.90786223620836 + x[1]* 0.25228903489171 + x[2]* -0.53485124507674 + x[3]* -1.8522537551352 + x[4]* 1.5666089210993 + x[5]* -0.24489915142453 + x[6]* -3.8189488996167 + x[7]* 0.54909352731943 + x[8]* 0.9272179256372 + x[9]* 1.8009119972273
    y=y+ 0.018516761200688 * torch.cos(fterm) + 0.031299127609342 * torch.sin(fterm)
    fterm = x[0]* 0.84723127607948 + x[1]* 0.050280388584138 + x[2]* 0.16693223406574 + x[3]* -0.60517770461978 + x[4]* -0.79132263098257 + x[5]* 0.084043948902495 + x[6]* 0.55685759728236 + x[7]* 0.34486591580833 + x[8]* 1.4416809309088 + x[9]* -0.53320421692272
    y=y+ -3.4990059374596 * torch.cos(fterm) + 0.68253876032375 * torch.sin(fterm)
    fterm = x[0]* -0.19681773285744 + x[1]* 0.058254908022228 + x[2]* -0.42707928242563 + x[3]* 0.10819564756371 + x[4]* 0.04704308876312 + x[5]* 0.086127140665846 + x[6]* -0.024740149959336 + x[7]* 0.93632616781613 + x[8]* -0.035165710684237 + x[9]* 0.6569247940928
    y=y+ -0.17115042174198 * torch.cos(fterm) + -0.28252973516886 * torch.sin(fterm)
    fterm = x[0]* -0.0019037079266071 + x[1]* 0.18580405652159 + x[2]* 0.25080372029813 + x[3]* -0.38404653550615 + x[4]* -2.6288831462221 + x[5]* -0.2747417045966 + x[6]* 2.1320451793595 + x[7]* -1.045444967573 + x[8]* -0.79282650529856 + x[9]* 0.031010778657905
    y=y+ 0.059811477926127 * torch.cos(fterm) + 0.15289618520218 * torch.sin(fterm)
    fterm = x[0]* 0.19814051576479 + x[1]* 0.12473129275844 + x[2]* 0.18289812163033 + x[3]* -0.040240761962992 + x[4]* -3.3405395082573 + x[5]* 0.11291153891139 + x[6]* -2.1744195857812 + x[7]* 0.16544111454727 + x[8]* 1.0327303718767 + x[9]* 1.5923228862532
    y=y+ -0.010118619494568 * torch.cos(fterm) + -0.0074420349679962 * torch.sin(fterm)
    fterm = x[0]* -0.11678811433869 + x[1]* 0.40545196498035 + x[2]* -0.32209724522756 + x[3]* -1.2191020365812 + x[4]* -7.0124770086352 + x[5]* 0.03675927486414 + x[6]* 5.5603653868838 + x[7]* 0.37164881797739 + x[8]* -1.5113345858393 + x[9]* 0.66274581255803
    y=y+ 0.086117006049066 * torch.cos(fterm) + -0.04913820310423 * torch.sin(fterm)
    fterm = x[0]* -0.32593022105524 + x[1]* 0.079875980525919 + x[2]* 0.099870348032266 + x[3]* -0.67291405487035 + x[4]* -0.37666675860489 + x[5]* 0.20938689455095 + x[6]* 9.1755464525272 + x[7]* 2.2122935682262 + x[8]* 0.16197494065924 + x[9]* -0.018643916230819
    y=y+ 0.0022606327417702 * torch.cos(fterm) + 0.023805375609675 * torch.sin(fterm)
    fterm = x[0]* 0.84460551424592 + x[1]* -0.33094975587693 + x[2]* -0.010826144353245 + x[3]* 0.27948211496192 + x[4]* 0.17197033871996 + x[5]* -0.097541935974101 + x[6]* -0.30621259089848 + x[7]* -0.35493284958512 + x[8]* -0.1698519033907 + x[9]* 0.70620227752212
    y=y+ -0.053636933766471 * torch.cos(fterm) + 0.40565361766112 * torch.sin(fterm)
    fterm = x[0]* 0.17876901517902 + x[1]* 0.070199320612956 + x[2]* 0.14344749513824 + x[3]* -0.46963022472564 + x[4]* 6.3748083838038 + x[5]* 0.11207349357088 + x[6]* -4.1269050756988 + x[7]* 0.63174216577317 + x[8]* -0.10275316910193 + x[9]* -0.1838974495232
    y=y+ 0.0091583716548813 * torch.cos(fterm) + 0.064570155156537 * torch.sin(fterm)
    fterm = x[0]* 0.13173338939796 + x[1]* -0.28009532216854 + x[2]* 0.092772965987904 + x[3]* -1.5059798078413 + x[4]* -2.1352598181993 + x[5]* 0.10509416454745 + x[6]* -0.53606803801543 + x[7]* -0.18990485591095 + x[8]* 0.77387432217994 + x[9]* -0.23630436139784
    y=y+ 0.024383694318798 * torch.cos(fterm) + -0.073684695482718 * torch.sin(fterm)
    fterm = x[0]* 0.8804356524727 + x[1]* -0.14927153177112 + x[2]* -0.057756555652038 + x[3]* 1.0075676397375 + x[4]* -2.4857722942484 + x[5]* 0.15649829748824 + x[6]* 2.5030070853474 + x[7]* 1.3304923152072 + x[8]* -0.12114833963005 + x[9]* -0.31746502804656
    y=y+ 0.084057404036463 * torch.cos(fterm) + 0.080342220979079 * torch.sin(fterm)
    fterm = x[0]* -0.45892995128513 + x[1]* -0.087896961270462 + x[2]* -0.28713116616515 + x[3]* -0.37405851179436 + x[4]* -1.6627204298282 + x[5]* -0.10621951132626 + x[6]* -0.19502828709312 + x[7]* -0.088310579759066 + x[8]* -1.2096077751772 + x[9]* 1.2425887574684
    y=y+ -0.10632114607956 * torch.cos(fterm) + 0.028028588569602 * torch.sin(fterm)
    fterm = x[0]* 0.31768351305308 + x[1]* 0.060754198486302 + x[2]* 0.013545521015915 + x[3]* -0.7766269370097 + x[4]* -0.61787733817351 + x[5]* 0.10467080505564 + x[6]* 1.2082028035424 + x[7]* -0.37850368236208 + x[8]* 0.4094129670629 + x[9]* -0.53695066829666
    y=y+ 0.43252746651226 * torch.cos(fterm) + 0.016552122636672 * torch.sin(fterm)
    fterm = x[0]* 0.25758237719377 + x[1]* -0.25356195272768 + x[2]* -0.6786377277186 + x[3]* -1.0718851905902 + x[4]* 0.83322277658286 + x[5]* -0.30758875879175 + x[6]* -0.63470396094589 + x[7]* 1.2643960155815 + x[8]* -1.0684094036204 + x[9]* 0.45116263179746
    y=y+ -0.0074853421942671 * torch.cos(fterm) + -0.068405137339272 * torch.sin(fterm)
    fterm = x[0]* -0.26216784296152 + x[1]* -0.41730575070274 + x[2]* 0.015155508164056 + x[3]* -0.24359833967796 + x[4]* -0.12071617316971 + x[5]* -0.16570655132297 + x[6]* -1.7909030170594 + x[7]* -0.40755201770749 + x[8]* -1.0229730332706 + x[9]* -0.68446950944111
    y=y+ -0.049061333210103 * torch.cos(fterm) + -0.18136933353012 * torch.sin(fterm)
    fterm = x[0]* -0.17677316792512 + x[1]* -0.025469605233643 + x[2]* -0.1910114792626 + x[3]* 0.92943887248222 + x[4]* 0.35620276653333 + x[5]* -0.23245634977757 + x[6]* -1.8916035502678 + x[7]* 0.74548920174067 + x[8]* -0.40759478019872 + x[9]* -1.1381431030634
    y=y+ -0.016909584800276 * torch.cos(fterm) + 0.10722899471275 * torch.sin(fterm)
    fterm = x[0]* -0.43977970353579 + x[1]* -0.15622561448743 + x[2]* 0.64811378783201 + x[3]* -0.14286146577477 + x[4]* 0.66032866159992 + x[5]* -0.29121017753958 + x[6]* -2.2201938034659 + x[7]* 0.22837161242404 + x[8]* 1.0452005681998 + x[9]* 2.7703274410098
    y=y+ 0.047939907064817 * torch.cos(fterm) + 0.058542345464866 * torch.sin(fterm)
    fterm = x[0]* 0.80878100421884 + x[1]* -0.1832152414184 + x[2]* 0.79808714115796 + x[3]* 0.32102197337873 + x[4]* 0.30887569585679 + x[5]* -0.019982765911454 + x[6]* -0.048597787228745 + x[7]* 0.83984658398251 + x[8]* 0.095459155459479 + x[9]* -1.5308996912961
    y=y+ -0.027583794700889 * torch.cos(fterm) + -0.015807980686177 * torch.sin(fterm)
    fterm = x[0]* 0.75135425825118 + x[1]* 0.32907884453582 + x[2]* 0.083193143066692 + x[3]* 0.63183684850071 + x[4]* 1.453284313977 + x[5]* -0.043615950135482 + x[6]* -0.91140517823509 + x[7]* 0.38237529105443 + x[8]* 0.32872090921479 + x[9]* -0.12651777143292
    y=y+ -0.051786986817071 * torch.cos(fterm) + -0.082560931826909 * torch.sin(fterm)
    fterm = x[0]* 0.011668877891523 + x[1]* -0.041078264870397 + x[2]* 0.62851420706784 + x[3]* -0.6894866112561 + x[4]* 4.157527054699 + x[5]* -0.16842535817262 + x[6]* -4.8078863734826 + x[7]* -0.34741726477873 + x[8]* -0.65921610281342 + x[9]* 0.86790778494934
    y=y+ -0.084121464323058 * torch.cos(fterm) + -0.11677906289896 * torch.sin(fterm)
    fterm = x[0]* -0.46558466749701 + x[1]* -0.18491041977008 + x[2]* -0.17118747295036 + x[3]* 0.49679229476403 + x[4]* 2.5697896940838 + x[5]* -0.066126179601844 + x[6]* -0.82627814981748 + x[7]* 0.31777498840923 + x[8]* 0.84316776324651 + x[9]* 0.95729808447415
    y=y+ -0.11881647098682 * torch.cos(fterm) + -0.0028600422974877 * torch.sin(fterm)
    fterm = x[0]* 1.6037567163433 + x[1]* 0.00053758943283798 + x[2]* -0.32091470765556 + x[3]* 0.97084640098594 + x[4]* -2.9941574018056 + x[5]* -0.46712568401221 + x[6]* -1.8116137105656 + x[7]* 0.054856490528935 + x[8]* 0.48579759317903 + x[9]* -0.24774637606093
    y=y+ 0.015369878517688 * torch.cos(fterm) + -0.0014170875474372 * torch.sin(fterm)
    fterm = x[0]* -0.23987681510562 + x[1]* -0.25593431278915 + x[2]* 0.0019787299505214 + x[3]* -0.32541985863337 + x[4]* -2.9560371696221 + x[5]* -0.34906715633717 + x[6]* -3.7363276022986 + x[7]* 0.042673616217396 + x[8]* -0.58697902541369 + x[9]* -0.7759755094146
    y=y+ -0.015786717443258 * torch.cos(fterm) + 0.024975094967218 * torch.sin(fterm)
    fterm = x[0]* -0.22583019727704 + x[1]* -0.077395441495797 + x[2]* -0.90070097121186 + x[3]* -1.3515781775198 + x[4]* 0.96817936231753 + x[5]* 0.016511460088508 + x[6]* -0.80685627363305 + x[7]* -1.4141167726755 + x[8]* -0.14414186371363 + x[9]* -1.1413811739829
    y=y+ 0.03439862320363 * torch.cos(fterm) + -0.046598080481382 * torch.sin(fterm)
    fterm = x[0]* 0.36159338585778 + x[1]* 0.024673785898007 + x[2]* -0.49530279421355 + x[3]* -0.53200062724239 + x[4]* 4.7428418297513 + x[5]* -0.13520534322518 + x[6]* 2.7194971849481 + x[7]* -0.34157058625228 + x[8]* -0.14560613302904 + x[9]* -1.3550265164725
    y=y+ -0.011518035023694 * torch.cos(fterm) + -0.053273940541563 * torch.sin(fterm)
    fterm = x[0]* 0.43925709965552 + x[1]* 0.045770084803915 + x[2]* -0.28623521155044 + x[3]* -0.36182542785672 + x[4]* 1.8435442636642 + x[5]* 0.13764671627122 + x[6]* -1.7369880310385 + x[7]* -0.67718098088346 + x[8]* 0.96181819307071 + x[9]* 0.90188716127572
    y=y+ 0.23288826713576 * torch.cos(fterm) + -0.15672951801364 * torch.sin(fterm)
    fterm = x[0]* 0.064598397316458 + x[1]* -0.096604519115184 + x[2]* -0.39600241189883 + x[3]* 0.82081540685491 + x[4]* 3.7967918536811 + x[5]* -0.32479462323413 + x[6]* -6.8163120661345 + x[7]* -0.66256830456833 + x[8]* -0.72268349097057 + x[9]* -0.60484037951159
    y=y+ -0.0046388898142893 * torch.cos(fterm) + -0.079687036388495 * torch.sin(fterm)
    fterm = x[0]* 0.37445329094819 + x[1]* -0.363050810254 + x[2]* -0.85538049193332 + x[3]* 0.52346497545624 + x[4]* 4.1858898231664 + x[5]* -0.15817295606714 + x[6]* -6.0893988831619 + x[7]* 0.45630187667385 + x[8]* -0.074689397657911 + x[9]* -0.44773066097934
    y=y+ 0.044154822592607 * torch.cos(fterm) + -0.13794930346985 * torch.sin(fterm)
    fterm = x[0]* -0.059317512967557 + x[1]* -0.11022537799313 + x[2]* 0.31400317452795 + x[3]* -0.75274374233922 + x[4]* 1.0373155190017 + x[5]* 0.30714349808628 + x[6]* -3.5554839662073 + x[7]* -1.2221273756119 + x[8]* 0.50864317961329 + x[9]* -0.84515229970508
    y=y+ -0.060686607415732 * torch.cos(fterm) + -0.088015554200766 * torch.sin(fterm)
    fterm = x[0]* -0.29244085118027 + x[1]* -0.019183333270204 + x[2]* -0.077483622043874 + x[3]* -1.0736630167229 + x[4]* 3.04419549621 + x[5]* -0.11645011564443 + x[6]* -3.7448151043279 + x[7]* -1.0090715552592 + x[8]* -2.0064460721785 + x[9]* -1.0847868318571
    y=y+ 0.064300654108802 * torch.cos(fterm) + 0.025756630419533 * torch.sin(fterm)
    fterm = x[0]* 0.20500545105685 + x[1]* 0.042721643850222 + x[2]* -0.28239556672529 + x[3]* 0.029533614880545 + x[4]* -0.50155646564908 + x[5]* -0.17524499316815 + x[6]* 5.6667285838448 + x[7]* 0.24892316862687 + x[8]* -0.57378696284797 + x[9]* 1.6207438609699
    y=y+ 0.053651372132957 * torch.cos(fterm) + -0.019255348978032 * torch.sin(fterm)
    fterm = x[0]* 0.80835434001989 + x[1]* 0.033730865058327 + x[2]* -0.36192000465059 + x[3]* 0.068972098703966 + x[4]* 0.39392332382431 + x[5]* 0.02545045216391 + x[6]* 1.8776375750048 + x[7]* 0.84579939266458 + x[8]* -0.83045785788514 + x[9]* -0.48607999231105
    y=y+ -0.11951754174837 * torch.cos(fterm) + -0.015213137309564 * torch.sin(fterm)
    fterm = x[0]* 0.61621830446052 + x[1]* -0.20213981234134 + x[2]* -0.68587765032013 + x[3]* -0.16699261687744 + x[4]* 0.57880370456389 + x[5]* -0.097281575761222 + x[6]* -0.10123488308088 + x[7]* -0.65262972664406 + x[8]* -0.77807867134149 + x[9]* 0.43874433329463
    y=y+ 0.67897256508403 * torch.cos(fterm) + 0.45842360791261 * torch.sin(fterm)
    fterm = x[0]* 0.69715873877587 + x[1]* -0.066252785620862 + x[2]* 0.27855742844366 + x[3]* -0.72016847115053 + x[4]* 9.004546581274 + x[5]* -0.023021945093869 + x[6]* -8.0358050609837 + x[7]* 0.070269407451441 + x[8]* -0.17145826222352 + x[9]* 0.23371762786191
    y=y+ -0.11558159654724 * torch.cos(fterm) + 0.1280981152627 * torch.sin(fterm)
    fterm = x[0]* 0.061123532802879 + x[1]* 0.31117935807833 + x[2]* -0.042175847334849 + x[3]* 1.0643277928174 + x[4]* 3.571742118105 + x[5]* 0.060501661631117 + x[6]* -3.1048698462113 + x[7]* 0.40682378280127 + x[8]* 0.83377672510518 + x[9]* -1.0219143195115
    y=y+ 0.12748200297507 * torch.cos(fterm) + 0.049135256473724 * torch.sin(fterm)
    fterm = x[0]* -0.44209301626502 + x[1]* 0.054516649430299 + x[2]* -0.71758192434574 + x[3]* 0.69285987658271 + x[4]* -1.9573420448319 + x[5]* 0.37990622043178 + x[6]* -2.4086154903417 + x[7]* -0.31509319788408 + x[8]* 0.91215670154238 + x[9]* -0.40144522153246
    y=y+ -0.034712965813719 * torch.cos(fterm) + 0.046483858820691 * torch.sin(fterm)
    fterm = x[0]* -0.8115922537416 + x[1]* 0.015683113181554 + x[2]* 0.041448036256392 + x[3]* -0.58974637492163 + x[4]* 1.0023351224226 + x[5]* -0.10293406857145 + x[6]* -0.81923159078529 + x[7]* -1.0525319967428 + x[8]* -0.4659667165728 + x[9]* -1.2676270486531
    y=y+ -0.068291348633799 * torch.cos(fterm) + 0.077118109718707 * torch.sin(fterm)
    fterm = x[0]* -0.6870714685788 + x[1]* -0.16954182644662 + x[2]* -0.15518803300695 + x[3]* -0.10882766625271 + x[4]* 0.95457462902051 + x[5]* -0.23744205001861 + x[6]* -0.36616055202298 + x[7]* 0.3541862497669 + x[8]* 0.057071813041939 + x[9]* -0.18828258229724
    y=y+ -0.12766999274979 * torch.cos(fterm) + 0.37594579379313 * torch.sin(fterm)
    fterm = x[0]* -1.2233133350886 + x[1]* 0.2818169111056 + x[2]* 0.68768773694539 + x[3]* 0.30228449129992 + x[4]* 2.9491625036004 + x[5]* 0.021740132213148 + x[6]* -0.78981562725561 + x[7]* -1.1797429142248 + x[8]* -1.3043266779761 + x[9]* 0.27035547879731
    y=y+ -0.0043180756877553 * torch.cos(fterm) + -0.039611362195905 * torch.sin(fterm)
    fterm = x[0]* 0.40396480218111 + x[1]* 0.036748442602898 + x[2]* -0.61937693804835 + x[3]* 0.27402193978315 + x[4]* -1.7842147913316 + x[5]* 0.18969225643528 + x[6]* 2.5576465009336 + x[7]* -0.15063584259683 + x[8]* -2.0517360876039 + x[9]* 1.775081107002
    y=y+ 0.020784074404751 * torch.cos(fterm) + -0.10852650573083 * torch.sin(fterm)
    fterm = x[0]* 0.4028289598915 + x[1]* -0.073175411728252 + x[2]* -0.53458882076387 + x[3]* 0.29728563312547 + x[4]* 0.62698706643598 + x[5]* -0.1625781539619 + x[6]* -0.17510801055693 + x[7]* 0.51199125506705 + x[8]* 0.54716019643085 + x[9]* -1.3632242063645
    y=y+ 0.3427433447271 * torch.cos(fterm) + 0.29556700908644 * torch.sin(fterm)
    fterm = x[0]* 0.24719249279029 + x[1]* 0.53888175734965 + x[2]* 0.82640950274578 + x[3]* 0.77593890535894 + x[4]* -1.4871335430835 + x[5]* 0.063923384775276 + x[6]* -1.4823186298831 + x[7]* 0.45828293578681 + x[8]* 0.61353634209822 + x[9]* -0.030632200465192
    y=y+ -0.079967881764442 * torch.cos(fterm) + -0.039971495234953 * torch.sin(fterm)
    fterm = x[0]* 0.012422692796249 + x[1]* 0.28588048766519 + x[2]* -0.14705908711552 + x[3]* 1.0347973283625 + x[4]* -3.0289662397224 + x[5]* -0.21657919993359 + x[6]* -3.0576659047943 + x[7]* -0.52152583763585 + x[8]* 0.60779451482956 + x[9]* 1.5724457442509
    y=y+ 0.016197643889979 * torch.cos(fterm) + -0.02311509558311 * torch.sin(fterm)
    fterm = x[0]* -0.61141511700201 + x[1]* 0.13290626223625 + x[2]* 0.095219977256908 + x[3]* -0.14657309929045 + x[4]* -4.1277646553299 + x[5]* 0.11068453286229 + x[6]* 2.8747029628855 + x[7]* -0.54215387567982 + x[8]* -0.45116706073738 + x[9]* 1.7741546185254
    y=y+ -0.040138849880116 * torch.cos(fterm) + 0.074962524125227 * torch.sin(fterm)
    fterm = x[0]* -0.39107551291792 + x[1]* 0.12274660608723 + x[2]* -0.60178277343316 + x[3]* 0.6602659147272 + x[4]* 1.5156204231173 + x[5]* -0.010184729966952 + x[6]* -1.8673434534205 + x[7]* 0.28893976397337 + x[8]* -0.77075294468846 + x[9]* -0.19924946508226
    y=y+ -0.20336766861947 * torch.cos(fterm) + 0.33595325035672 * torch.sin(fterm)
    fterm = x[0]* 0.013413509884823 + x[1]* 0.040304867605146 + x[2]* 0.79704844922216 + x[3]* 0.83154542377705 + x[4]* 0.14268370176646 + x[5]* 0.11559116505276 + x[6]* 1.9640395532104 + x[7]* 1.38891577714 + x[8]* 0.34927172570989 + x[9]* 0.46568732037223
    y=y+ 0.069177423378979 * torch.cos(fterm) + -0.061434451343931 * torch.sin(fterm)
    fterm = x[0]* 0.23627975818342 + x[1]* 0.026179118145544 + x[2]* -0.43496460935533 + x[3]* -0.2957285364787 + x[4]* 0.8359053206001 + x[5]* -0.13529122795089 + x[6]* -0.59076809851459 + x[7]* -0.31742811991633 + x[8]* -1.5477420509238 + x[9]* 0.5490837993564
    y=y+ 3.0717135720378 * torch.cos(fterm) + 0.42595342511833 * torch.sin(fterm)
    fterm = x[0]* -0.33577157755192 + x[1]* -0.1890541145897 + x[2]* -0.044115794840124 + x[3]* 0.25836953839076 + x[4]* 2.6503599150371 + x[5]* -0.18131623386889 + x[6]* 3.0719027772353 + x[7]* 0.24558608715437 + x[8]* 0.90474113435491 + x[9]* -1.4636261033049
    y=y+ 0.025265155820038 * torch.cos(fterm) + 0.0082123072262535 * torch.sin(fterm)
    fterm = x[0]* 0.25459319593585 + x[1]* -0.388183457583 + x[2]* -0.57560753449426 + x[3]* 0.076838701081425 + x[4]* 2.100720398664 + x[5]* -0.099082438310629 + x[6]* 0.58133229208003 + x[7]* -0.016242740362897 + x[8]* 1.1969145539546 + x[9]* -2.6154766617621
    y=y+ -0.055206719748503 * torch.cos(fterm) + -0.037797398860482 * torch.sin(fterm)
    fterm = x[0]* -0.25188182923633 + x[1]* -0.18007007905242 + x[2]* -0.36475152983176 + x[3]* -0.087203328561246 + x[4]* 0.48076813999135 + x[5]* -0.17766311279376 + x[6]* 1.688345899356 + x[7]* -0.096706131852119 + x[8]* -0.14158435712413 + x[9]* -0.29487491948705
    y=y+ -0.03430882993683 * torch.cos(fterm) + 0.15894107821953 * torch.sin(fterm)
    fterm = x[0]* 0.40576220232117 + x[1]* 0.080362129537167 + x[2]* 0.27772777647952 + x[3]* 0.4482599293108 + x[4]* 2.2325772254501 + x[5]* -0.13769785324422 + x[6]* -3.8766768620419 + x[7]* -0.041222843424161 + x[8]* -0.27890131495352 + x[9]* 0.19552807997064
    y=y+ 0.040853423247748 * torch.cos(fterm) + 0.026081877370598 * torch.sin(fterm)
    fterm = x[0]* 1.0486970586256 + x[1]* 0.58480202365134 + x[2]* 0.19761511597858 + x[3]* -0.48318636514403 + x[4]* 2.1692622059488 + x[5]* 0.104324832837 + x[6]* 3.0013041490242 + x[7]* -0.0013034081387521 + x[8]* -0.2755002819141 + x[9]* -1.5566669526027
    y=y+ -0.0013882997029764 * torch.cos(fterm) + -0.015511807781405 * torch.sin(fterm)
    fterm = x[0]* -0.31386081002059 + x[1]* 0.1293020499412 + x[2]* 0.11529795460325 + x[3]* -0.24822392526695 + x[4]* -3.3160561798488 + x[5]* 0.44706491421305 + x[6]* -1.8294190352312 + x[7]* -0.053453033705414 + x[8]* -0.49280245287429 + x[9]* 0.19694055980245
    y=y+ -0.060980637003501 * torch.cos(fterm) + 0.14055568329736 * torch.sin(fterm)
    fterm = x[0]* 0.25994500139995 + x[1]* -0.078939580699695 + x[2]* -0.01494390554366 + x[3]* 0.47595755604079 + x[4]* 0.81351882927992 + x[5]* 0.071500271326471 + x[6]* -0.55798198062438 + x[7]* -1.1381046703675 + x[8]* -1.0426145502673 + x[9]* -0.59339919002787
    y=y+ -0.089029068082855 * torch.cos(fterm) + 0.004576124468916 * torch.sin(fterm)
    fterm = x[0]* -0.73157058341543 + x[1]* 0.3700308214291 + x[2]* 0.33500770801749 + x[3]* 0.018024187931452 + x[4]* 1.2205331546415 + x[5]* -0.18169028160815 + x[6]* -4.4857174787933 + x[7]* -0.75684286937228 + x[8]* 0.34485953827102 + x[9]* 0.085304783163633
    y=y+ 0.085377058203667 * torch.cos(fterm) + 0.031528621633092 * torch.sin(fterm)
    fterm = x[0]* -0.15242786643233 + x[1]* 0.0025867522449353 + x[2]* 0.44500713480761 + x[3]* -0.79318481758384 + x[4]* -1.7823408419956 + x[5]* 0.20269196179305 + x[6]* -4.8894343557039 + x[7]* 0.65510375537264 + x[8]* 0.80417195998693 + x[9]* 0.8577866112518
    y=y+ 0.014358954022024 * torch.cos(fterm) + 0.029772391811455 * torch.sin(fterm)
    fterm = x[0]* -0.063928130376962 + x[1]* -0.0046039198715692 + x[2]* 0.67747500772939 + x[3]* 0.43190885676114 + x[4]* 0.91902505388909 + x[5]* 0.20261839326194 + x[6]* -2.7174511895402 + x[7]* 0.62376436068261 + x[8]* -1.5480884497021 + x[9]* -1.9334713339308
    y=y+ 0.00092096943883582 * torch.cos(fterm) + 0.051218065835908 * torch.sin(fterm)
    fterm = x[0]* 0.17447009697427 + x[1]* 0.16264768985038 + x[2]* 0.6126307557983 + x[3]* 0.38821088563286 + x[4]* 1.6460486822933 + x[5]* 0.11904629785382 + x[6]* 4.4526474812372 + x[7]* -1.2874473677698 + x[8]* -0.53353562411878 + x[9]* -0.94954520859861
    y=y+ 0.0041032472616334 * torch.cos(fterm) + 0.026064196886314 * torch.sin(fterm)
    fterm = x[0]* -0.11381273673559 + x[1]* -0.11309789711385 + x[2]* 0.40577937398995 + x[3]* -0.60012174166095 + x[4]* -3.0754731732655 + x[5]* 0.00059941618785972 + x[6]* -2.255890362792 + x[7]* -0.22475924599207 + x[8]* -0.37080176484437 + x[9]* 0.24450911270488
    y=y+ -0.037312990607426 * torch.cos(fterm) + -0.13923789452907 * torch.sin(fterm)
    fterm = x[0]* 0.73311134921818 + x[1]* 0.30884254753799 + x[2]* 0.19912824792272 + x[3]* 0.47867765355674 + x[4]* -9.9376866939528 + x[5]* -0.26458859522512 + x[6]* -3.4585407939256 + x[7]* -0.64018780183579 + x[8]* 0.87941998857369 + x[9]* -1.1364026640412
    y=y+ -0.0080843906837836 * torch.cos(fterm) + 0.0085226826746696 * torch.sin(fterm)
    fterm = x[0]* 0.77106503579505 + x[1]* -0.090102588331579 + x[2]* 0.18881987091193 + x[3]* 0.74168758914324 + x[4]* -4.6126095589984 + x[5]* 0.12580729209152 + x[6]* 0.61363369687333 + x[7]* -0.24884677386098 + x[8]* -0.65843414165576 + x[9]* -2.2102908294502
    y=y+ 0.015989293279952 * torch.cos(fterm) + -0.012531332785732 * torch.sin(fterm)
    fterm = x[0]* -0.32634568173872 + x[1]* 0.041571313604241 + x[2]* -0.76226551693684 + x[3]* -0.059375361674349 + x[4]* 3.5742752522233 + x[5]* -0.14970045182107 + x[6]* 0.97835294350369 + x[7]* 0.10905628141524 + x[8]* 0.23364474393908 + x[9]* 0.94102444425663
    y=y+ -0.0064939405986199 * torch.cos(fterm) + -0.046195210898848 * torch.sin(fterm)
    fterm = x[0]* -0.89739581078345 + x[1]* 0.14111364565511 + x[2]* -0.29049561000264 + x[3]* -0.23732677579395 + x[4]* 0.73573877375747 + x[5]* 0.20717826702202 + x[6]* 1.0510327826059 + x[7]* -1.89027936533 + x[8]* 0.98497827877588 + x[9]* -1.361749264021
    y=y+ 0.040109282095949 * torch.cos(fterm) + 0.034693150867751 * torch.sin(fterm)
    fterm = x[0]* -0.5826919870097 + x[1]* 0.021842676690041 + x[2]* -0.48892035358605 + x[3]* 0.63536266921571 + x[4]* 4.1600250518594 + x[5]* 0.25507730526919 + x[6]* -2.2829257029429 + x[7]* -1.1224303280368 + x[8]* -0.39902675894711 + x[9]* -1.0826638376451
    y=y+ -0.039981510039405 * torch.cos(fterm) + -0.017204146455158 * torch.sin(fterm)
    fterm = x[0]* 0.030230176956927 + x[1]* -0.296730661779 + x[2]* 0.011517022440393 + x[3]* 0.055187508805417 + x[4]* -1.9566404265983 + x[5]* 0.077188842592816 + x[6]* 2.068856413021 + x[7]* 1.1795985503023 + x[8]* 0.68331050991756 + x[9]* 2.7860010644792
    y=y+ -0.058584871580974 * torch.cos(fterm) + -0.017393943656854 * torch.sin(fterm)
    fterm = x[0]* 0.31822709333089 + x[1]* -0.014271464298613 + x[2]* 0.70535523626635 + x[3]* 0.2742032039563 + x[4]* 1.4570613279171 + x[5]* 0.11385902494481 + x[6]* -0.99207722105192 + x[7]* -0.65912129020136 + x[8]* 1.3836058945175 + x[9]* 1.7607664501676
    y=y+ 0.091013521410747 * torch.cos(fterm) + -0.016028891176556 * torch.sin(fterm)
    fterm = x[0]* -0.86802514838381 + x[1]* -0.3354593208007 + x[2]* -0.41552143253572 + x[3]* 0.20824900545061 + x[4]* -3.8462053401617 + x[5]* 0.026213222748612 + x[6]* -0.17350927185547 + x[7]* 1.409010967085 + x[8]* 0.89354106903164 + x[9]* 1.2971490987342
    y=y+ -0.019081474586905 * torch.cos(fterm) + -0.022425834954556 * torch.sin(fterm)
    fterm = x[0]* -0.25793449504941 + x[1]* 0.71037864534253 + x[2]* 0.7267138173557 + x[3]* 1.1998813791212 + x[4]* 3.5053884111367 + x[5]* 0.14584112687278 + x[6]* -2.9387848525964 + x[7]* -0.23244674942071 + x[8]* -1.4816666350978 + x[9]* -0.16874633991965
    y=y+ 0.0095682147175978 * torch.cos(fterm) + 0.011403276459972 * torch.sin(fterm)
    fterm = x[0]* 0.74100143992648 + x[1]* 0.19894178799665 + x[2]* 0.018332770745976 + x[3]* -0.63276674310163 + x[4]* 2.3105586916758 + x[5]* -0.26716792155061 + x[6]* -2.0023305883282 + x[7]* -0.044046540936418 + x[8]* -0.61801259397661 + x[9]* 0.019247008483582
    y=y+ 0.38594893233202 * torch.cos(fterm) + 0.13289299422207 * torch.sin(fterm)
    fterm = x[0]* -0.21210027099482 + x[1]* 0.23213437753485 + x[2]* 0.32295060672409 + x[3]* 0.37943237695541 + x[4]* -0.098498100939365 + x[5]* -0.06472187290408 + x[6]* -3.7431422682593 + x[7]* -1.4681947230847 + x[8]* 1.5739407894498 + x[9]* -1.3693647285006
    y=y+ -0.0057533003864996 * torch.cos(fterm) + -0.053911558965515 * torch.sin(fterm)
    fterm = x[0]* -0.3294642883148 + x[1]* -0.08221291323503 + x[2]* 0.21569623682358 + x[3]* -0.74270301815544 + x[4]* 0.19744550085899 + x[5]* -0.26914920486125 + x[6]* 1.3797277456814 + x[7]* -0.35061993574076 + x[8]* 0.65226836204528 + x[9]* 0.9871671568106
    y=y+ 0.063113280244711 * torch.cos(fterm) + 0.094482739846245 * torch.sin(fterm)
    fterm = x[0]* -0.0024102825865738 + x[1]* -0.31836713541293 + x[2]* -0.034270346647274 + x[3]* 0.52430545902216 + x[4]* -1.6725636427132 + x[5]* 0.05855420600357 + x[6]* -0.33719717795816 + x[7]* 0.53398251160395 + x[8]* 0.96179910045751 + x[9]* 0.762603622521
    y=y+ 0.15792444770157 * torch.cos(fterm) + -0.022371143214056 * torch.sin(fterm)
    fterm = x[0]* 0.75413364712181 + x[1]* -0.45647784695528 + x[2]* 1.0265422834213 + x[3]* -0.68153086045924 + x[4]* -1.2594124711685 + x[5]* -0.13599007237109 + x[6]* -1.7556455791042 + x[7]* -0.19956850595755 + x[8]* -0.95035708496566 + x[9]* -0.73805599735145
    y=y+ -0.081925089440202 * torch.cos(fterm) + 0.017141090759093 * torch.sin(fterm)
    fterm = x[0]* -0.45156317334975 + x[1]* -0.32357922653324 + x[2]* 0.7760892474274 + x[3]* 0.43832634996747 + x[4]* 3.9279769394259 + x[5]* -0.11136671902838 + x[6]* -0.7575971715878 + x[7]* 0.37561305045973 + x[8]* 0.17832902806532 + x[9]* 0.24555793334044
    y=y+ 0.13738424777592 * torch.cos(fterm) + 0.053265857701089 * torch.sin(fterm)
    fterm = x[0]* -0.6906030480701 + x[1]* 0.41163068958208 + x[2]* -0.4027333293875 + x[3]* 0.31605385389028 + x[4]* 0.076494241743703 + x[5]* -0.19652125245129 + x[6]* 6.2120166079051 + x[7]* -0.14757603720768 + x[8]* -1.756674274605 + x[9]* -1.9280803867154
    y=y+ -0.03549032657104 * torch.cos(fterm) + 0.017672908686963 * torch.sin(fterm)
    fterm = x[0]* -0.48235174914102 + x[1]* -0.14505745567104 + x[2]* -0.89285023189899 + x[3]* -0.65796650618568 + x[4]* -1.7818085457027 + x[5]* -0.077077512049044 + x[6]* -3.1546989700182 + x[7]* 0.28593200288056 + x[8]* -1.7965952265115 + x[9]* 0.68987570372033
    y=y+ -0.033383596761382 * torch.cos(fterm) + 0.01708951694821 * torch.sin(fterm)
    fterm = x[0]* -0.43372466387743 + x[1]* 0.0070073398159157 + x[2]* 0.28432167179528 + x[3]* 0.34117314551252 + x[4]* -0.91625966586057 + x[5]* 0.19353827335367 + x[6]* -0.24508433580305 + x[7]* -0.70501247944371 + x[8]* -1.1859299029664 + x[9]* 0.86604144279862
    y=y+ -0.11207783813209 * torch.cos(fterm) + -0.066353923098077 * torch.sin(fterm)
    fterm = x[0]* 1.0530682027985 + x[1]* 0.28636638293607 + x[2]* 0.57150450833751 + x[3]* -0.11144645862663 + x[4]* -3.0725638863648 + x[5]* -0.17342818569404 + x[6]* 1.0228359792419 + x[7]* -0.92256310608875 + x[8]* 2.2013540385711 + x[9]* 1.2926329931788
    y=y+ -0.030808166862244 * torch.cos(fterm) + 0.038739528019412 * torch.sin(fterm)
    fterm = x[0]* 0.21742569239359 + x[1]* -0.30008192867895 + x[2]* 0.0090961284191189 + x[3]* 0.59998484535416 + x[4]* -6.836350335213 + x[5]* -0.24115220856752 + x[6]* -4.492997778579 + x[7]* -0.095675525034475 + x[8]* 0.76763340402437 + x[9]* -0.39440568040737
    y=y+ -0.0060646128345046 * torch.cos(fterm) + 0.0020001498578622 * torch.sin(fterm)
    fterm = x[0]* 0.33592101311812 + x[1]* 0.29300517495018 + x[2]* -0.081008193206153 + x[3]* 0.65788763344974 + x[4]* 0.49443884363231 + x[5]* -0.14546781842356 + x[6]* 2.6823414085615 + x[7]* 0.1016495953276 + x[8]* 0.71731959720856 + x[9]* 0.49214444361749
    y=y+ 0.049365601607823 * torch.cos(fterm) + -0.11114151241192 * torch.sin(fterm)
    fterm = x[0]* -0.12313743363216 + x[1]* 0.054325631391886 + x[2]* 0.44523931940928 + x[3]* -0.29965463301554 + x[4]* -6.5319333906529 + x[5]* 0.10456259009935 + x[6]* 6.4734343584366 + x[7]* -0.26451209770456 + x[8]* -0.032280273582581 + x[9]* 0.10726501135496
    y=y+ -0.18747373692342 * torch.cos(fterm) + 0.13584975865301 * torch.sin(fterm)
    fterm = x[0]* 0.4006160658585 + x[1]* -0.21283083202285 + x[2]* -0.068312192457003 + x[3]* 0.20618038289807 + x[4]* -2.4511429858129 + x[5]* -0.052334463687479 + x[6]* 2.1087287187327 + x[7]* -0.29875820186742 + x[8]* -0.80743257379728 + x[9]* -0.66277883318956
    y=y+ 0.10927821611682 * torch.cos(fterm) + -0.16980173269916 * torch.sin(fterm)
    fterm = x[0]* 0.27701655987021 + x[1]* -0.18859547586744 + x[2]* 0.58822055078665 + x[3]* -0.169705037502 + x[4]* -5.2430251385084 + x[5]* -0.014934294393431 + x[6]* 4.9214388242382 + x[7]* -1.0679270513252 + x[8]* -0.82405027514928 + x[9]* 0.48310683714814
    y=y+ 0.036114118023937 * torch.cos(fterm) + 0.02177097381349 * torch.sin(fterm)
    fterm = x[0]* 0.76812303505839 + x[1]* -0.21549786958375 + x[2]* -0.56064503290645 + x[3]* 0.41941568588048 + x[4]* -4.3821816595169 + x[5]* -0.28204286471956 + x[6]* 0.95775292601595 + x[7]* 0.27313504970283 + x[8]* -0.60848484526538 + x[9]* -1.019094971232
    y=y+ 0.022156600266183 * torch.cos(fterm) + -0.071298248083971 * torch.sin(fterm)
    fterm = x[0]* 0.98327764735988 + x[1]* 0.076033702180366 + x[2]* -0.15793453359845 + x[3]* 0.71950716503931 + x[4]* -0.47354412535569 + x[5]* 0.24127484360951 + x[6]* 0.7659084120113 + x[7]* 0.83200048015893 + x[8]* -0.26268813906648 + x[9]* -1.2054529288702
    y=y+ -0.064087023224232 * torch.cos(fterm) + 0.079679907842945 * torch.sin(fterm)
    fterm = x[0]* 0.8916980909554 + x[1]* 0.076720953877967 + x[2]* -0.45019093325921 + x[3]* 0.36548965680332 + x[4]* 0.21068731421127 + x[5]* 0.44490303586371 + x[6]* 0.56120813986296 + x[7]* -0.011496002719082 + x[8]* -1.0078130376216 + x[9]* 0.44490598310089
    y=y+ -0.15284350315258 * torch.cos(fterm) + -0.070732848135661 * torch.sin(fterm)
    fterm = x[0]* -0.29091377550497 + x[1]* -0.32912199707093 + x[2]* -0.4298165020062 + x[3]* 0.59574409894471 + x[4]* -2.8754419446166 + x[5]* -0.086683256116449 + x[6]* 1.6454379877669 + x[7]* 0.25938854556758 + x[8]* -0.97475354208878 + x[9]* 0.10806979810943
    y=y+ -0.24232239333697 * torch.cos(fterm) + -0.081581950149644 * torch.sin(fterm)
    fterm = x[0]* 0.17893792251594 + x[1]* 0.16954010216603 + x[2]* 0.51201140789787 + x[3]* -1.0523064897854 + x[4]* -1.167040024119 + x[5]* -0.10698900931449 + x[6]* 1.626316962915 + x[7]* -0.0072453457587813 + x[8]* -0.0092316317016417 + x[9]* -0.011162384381199
    y=y+ -0.29386565010948 * torch.cos(fterm) + 0.65135209764835 * torch.sin(fterm)
    fterm = x[0]* 0.0063606034904218 + x[1]* 0.071501324376628 + x[2]* -0.48863571434896 + x[3]* 0.14135732422574 + x[4]* -3.4840052050416 + x[5]* 0.24675318111587 + x[6]* -1.6449484288109 + x[7]* -0.56672822756487 + x[8]* -1.1501495313197 + x[9]* -0.86455150613677
    y=y+ 0.055390731056039 * torch.cos(fterm) + 0.04739767721103 * torch.sin(fterm)
    fterm = x[0]* 1.0269981893507 + x[1]* -0.082951886079255 + x[2]* 0.22708479699296 + x[3]* -0.42111959856584 + x[4]* 0.27806038339095 + x[5]* -0.11855285965653 + x[6]* -0.80495473348493 + x[7]* 1.1492815648505 + x[8]* 0.15774188423831 + x[9]* -0.28823932935373
    y=y+ -0.42557167422989 * torch.cos(fterm) + 0.13885157000596 * torch.sin(fterm)
    fterm = x[0]* 0.39965111097893 + x[1]* 0.13710749855091 + x[2]* 0.26849312382293 + x[3]* 0.81666717467468 + x[4]* 2.4997475602785 + x[5]* 0.25828347457931 + x[6]* 5.392425989108 + x[7]* 0.3779202358358 + x[8]* -1.3746223919269 + x[9]* 1.2911730603403
    y=y+ 0.025005799753262 * torch.cos(fterm) + 0.02534531537369 * torch.sin(fterm)
    fterm = x[0]* 0.75270340663039 + x[1]* 0.2712441950529 + x[2]* -0.051316020031209 + x[3]* 0.81218286537538 + x[4]* 0.33584399773702 + x[5]* -0.073024756911837 + x[6]* -0.078118634689732 + x[7]* -1.044296955018 + x[8]* -0.78289236871129 + x[9]* 1.2210752978127
    y=y+ 0.0032374278105146 * torch.cos(fterm) + 0.068392460514006 * torch.sin(fterm)
    # y = y * 5.1489801806121 + 14.886169231027
    return y
