# envs/disturbance_wrapper.py
import numpy as np

class DisturbanceWrapper(gym.Wrapper):
    """Adds random external disturbances to the environment to test policy robustness."""
    def __init__(self, env, force_mag=50.0, interval=100, noise_actions=False):
        """
        force_mag: magnitude of force (Newtons) to apply as disturbance
        interval: apply a disturbance every `interval` steps
        noise_actions: if True, add small noise to actions each step instead of force
        """
        super().__init__(env)
        self.force_mag = force_mag
        self.interval = interval
        self.noise_actions = noise_actions
        self.step_count = 0

    def reset(self, **kwargs):
        self.step_count = 0
        return self.env.reset(**kwargs)
    
    def step(self, action):
        # Optionally add noise to the action
        if self.noise_actions:
            noise = np.random.normal(loc=0.0, scale=0.1, size=action.shape)
            action = np.clip(action + noise, -1.0, 1.0)  # assuming action space in [-1,1]
        obs, reward, done, info = self.env.step(action)
        self.step_count += 1
        # Every `interval` steps, apply a force to the robot's base (e.g., push it)
        if self.step_count % self.interval == 0:
            self._apply_external_force()
        return obs, reward, done, info

    def _apply_external_force(self):
        sim = self.env.unwrapped.sim
        if sim is None: 
            return
        # Apply an impulse or a persistent force for one step on the base
        # For simplicity, apply instantaneous force on root (if identifiable)
        try:
            body_id = sim.model.body_name2id('torso')  # common naming
        except Exception:
            body_id = 0  # assume root body id 0 if name not found
        # Apply force in a random horizontal direction
        direction = np.random.choice([0, 1])  # 0 = push forward/back, 1 = push sideways
        force = np.zeros(3)
        if direction == 0:
            force[0] = np.random.choice([-1, 1]) * self.force_mag  # push forward or backward
        else:
            force[2] = np.random.choice([-1, 1]) * self.force_mag  # push left or right (if applicable)
        # Apply the force for this simulation step
        sim.data.xfrc_applied[body_id, :3] = force  # apply translational force
