# envs/incline_wrapper.py
import math

class InclineWrapper(gym.Wrapper):
    """Applies an incline to a MuJoCo environment by adjusting gravity or the ground slope."""
    def __init__(self, env, slope_degrees=0.0):
        super().__init__(env)
        self.slope_angle = slope_degrees  # store slope angle (degrees)
        # Modify the gravity vector in the MuJoCo model to simulate incline.
        # Assuming original gravity is [0, 0, -9.81] (for 3D) or [0, -9.81, 0] (for planar with y-up).
        sim = env.unwrapped.sim
        if sim is not None:
            # For planar 2D tasks: typically gravity is [0, -9.81, 0] (x forward, y vertical up)
            # We rotate gravity around the out-of-plane axis to create an incline along x.
            theta = math.radians(slope_degrees)
            g = sim.model.opt.gravity  # this is a numpy array of shape (3,)
            # If y is vertical axis and x is forward axis:
            g_base = 9.81
            g_x = -g_base * math.sin(theta)   # horizontal component (points backward along +x when uphill)
            g_y = -g_base * math.cos(theta)   # vertical downward component
            sim.model.opt.gravity[:] = [g_x, -g_y, 0]  # adjust gravity vector
            # Note: We set gravity such that when slope_angle > 0 (uphill), gravity has a component pulling opposite to +x direction.
