# envs/actuator_wrapper.py

class ActuatorModificationWrapper(gym.Wrapper):
    """Modifies actuator outputs to simulate reduced power or failures."""
    def __init__(self, env, failure_mode=None, scale=0.5, failure_step=None, failed_index=None):
        """
        failure_mode: None or 'scale' or 'drop'. 
            - 'scale': scale all actions by `scale` (simulate weaker actuators).
            - 'drop': set a specific actuator's output to 0 after `failure_step`.
        scale: scaling factor for 'scale' mode.
        failure_step: the timestep at which to drop an actuator (for 'drop' mode).
        failed_index: index of actuator to drop (if None, choose random).
        """
        super().__init__(env)
        self.failure_mode = failure_mode
        self.scale = scale
        self.failure_step = failure_step
        self.failed_index = failed_index
        self.current_step = 0

    def reset(self, **kwargs):
        self.current_step = 0
        # Optionally randomize which actuator will fail if not specified
        if self.failed_index is None and self.failure_mode == 'drop':
            action_dim = self.env.action_space.shape[0]
            self.failed_index = np.random.randint(0, action_dim)
        return self.env.reset(**kwargs)

    def step(self, action):
        self.current_step += 1
        if self.failure_mode == 'scale':
            # Scale down all action outputs (e.g., 50% power)
            action = action * self.scale
        elif self.failure_mode == 'drop':
            # Zero-out one actuator after a certain time step
            if self.current_step >= (self.failure_step or np.inf):
                idx = self.failed_index or 0
                action = action.copy()
                action[idx] = 0.0  # disable this actuator
        obs, reward, done, info = self.env.step(action)
        return obs, reward, done, info
