# from: https://huggingface.co/kalpeshk2011/dipper-paraphraser-xxl

from time import time

import fire
import pandas as pd
import torch

from nltk.tokenize import sent_tokenize
from termcolor import colored
from transformers import (
    T5ForConditionalGeneration,
    T5Tokenizer,
)
from tqdm import tqdm

class DipperParaphraser(object):
    def __init__(self, model="kalpeshk2011/dipper-paraphraser-xxl", verbose=True):
        time1 = time()
        self.tokenizer = T5Tokenizer.from_pretrained(
            'google/t5-v1_1-xxl',
            device_map="auto",
            torch_dtype=torch.bfloat16,
        )
        self.model = T5ForConditionalGeneration.from_pretrained(model)
        if verbose:
            print(f"{model} model loaded in {time() - time1}")
        self.model.cuda()
        self.model.eval()

    def paraphrase(self, input_text, lex_diversity, order_diversity, prefix="", sent_interval=3, **kwargs):
        """Paraphrase a text using the DIPPER model.

        Args:
            input_text (str): The text to paraphrase. Make sure to mark the sentence to be paraphrased between <sent> and </sent> blocks, keeping space on either side.
            lex_diversity (int): The lexical diversity of the output, choose multiples of 20 from 0 to 100. 0 means no diversity, 100 means maximum diversity.
            order_diversity (int): The order diversity of the output, choose multiples of 20 from 0 to 100. 0 means no diversity, 100 means maximum diversity.
            **kwargs: Additional keyword arguments like top_p, top_k, max_length.
        """
        assert lex_diversity in [0, 20, 40, 60, 80, 100], "Lexical diversity must be one of 0, 20, 40, 60, 80, 100."
        assert order_diversity in [0, 20, 40, 60, 80, 100], "Order diversity must be one of 0, 20, 40, 60, 80, 100."

        lex_code = int(100 - lex_diversity)
        order_code = int(100 - order_diversity)

        input_text = " ".join(input_text.split())
        sentences = sent_tokenize(input_text)
        prefix = " ".join(prefix.replace("\n", " ").split())
        output_text = ""

        for sent_idx in range(0, len(sentences), sent_interval):
            curr_sent_window = " ".join(sentences[sent_idx:sent_idx + sent_interval])
            final_input_text = f"lexical = {lex_code}, order = {order_code}"
            if prefix:
                final_input_text += f" {prefix}"
            final_input_text += f" <sent> {curr_sent_window} </sent>"

            final_input = self.tokenizer([final_input_text], return_tensors="pt")
            final_input = {k: v.cuda() for k, v in final_input.items()}

            with torch.inference_mode():
                outputs = self.model.generate(**final_input, **kwargs)
            outputs = self.tokenizer.batch_decode(outputs, skip_special_tokens=True)
            prefix += " " + outputs[0]
            output_text += " " + outputs[0]

        return output_text

def main(
    filename: str = "/usr/WS1/rrivera/data/style_transfer/neurips/data/shards/MTD_reddit_12000_correct_Mistral-7B-Instruct-v0.3_N=5.jsonlshard-1-4",
    debug: bool = False,
):
    print(colored("Loading:", "green"), filename)
    dp = DipperParaphraser()
    dipper_outputs = []
    df = pd.read_json(filename, lines=True)
    if debug:
        df = df.iloc[:10]
    pbar = tqdm(total=len(df))
    for _, row in df.iterrows():
        generation = row["respond_reddit"]
        # Just like OUTFOX:
        output_l60_o60_sample = dp.paraphrase(generation, lex_diversity=60, order_diversity=60, prefix='', do_sample=True, top_p=0.75, top_k=None, max_length=256)
        dipper_outputs.append(output_l60_o60_sample)
        pbar.update(1)
    df["paraphrase_dipper"] = dipper_outputs
    filename += ".dipper"
    print(colored("Saving to:", "cyan"), filename)
    if debug:
        breakpoint()
    if not debug:
        df.to_json(filename, lines=True, orient="records")

if __name__ == "__main__":
    fire.Fire(main)