import os
import json
import argparse

from utils import build_style_transfer_prompts

PROMPT_TYPES = {
    'paraphrase': lambda text: f"Paraphrase: {text}",
    'sparaphrase': build_style_transfer_prompts  # placeholder for user-provided prompt function
}

def main(input_path: str, out_dir: str, prompt_type: str, debug: bool):
    if prompt_type not in PROMPT_TYPES:
        raise ValueError(f"Unknown prompt_type '{prompt_type}'. Choose from {list(PROMPT_TYPES.keys())}.")

    if debug:
        out_dir += ".debug"
        N = 100
    else:
        N = None
    ii = 0
    
    os.makedirs(out_dir, exist_ok=True)
    base = os.path.basename(input_path).replace('.jsonl','')
    prompts_path = os.path.join(out_dir, f"{base}_{prompt_type}_prompts.jsonl")

    builder = PROMPT_TYPES[prompt_type]
    with open(input_path, 'r') as fin, open(prompts_path, 'w') as fout:
        for line in fin:
            record = json.loads(line)
            
            if prompt_type == "sparaphrase":
                source_paraphrases = [record["paraphrase_respond_reddit"]]
                target_texts = [record["transfer_reference_text"]]
                target_paraphrases = [record["paraphrase_transfer_reference_text"]]
                prompt = builder(source_paraphrases, target_texts, target_paraphrases, 
                                 for_prompting=True, model_name="other")[0]
                fout.write(json.dumps({'prompt': prompt}) + "\n")
            else:
                text = record.get('respond_reddit', '')
                prompt = builder(text)
                fout.write(json.dumps({'prompt': prompt}) + "\n")
            
            ii += 1
            if N is not None and ii >= N:
                break

    print(f"Wrote {prompt_type} prompts to {prompts_path}")


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--input', required=True, help='Path to original .jsonl')
    parser.add_argument('--out_dir', required=True, help='Directory for prompts files')
    parser.add_argument('--prompt_type', default='paraphrase', choices=list(PROMPT_TYPES.keys()), help='Type of prompt to build')
    parser.add_argument("--debug", default=False, action="store_true")
    args = parser.parse_args()
    main(args.input, args.out_dir, args.prompt_type, args.debug)
