import os
import json
import argparse

def main(prompts_dir: str, out_path: str, model: str, url: str):
    """
    Reads every *_prompts.jsonl in prompts_dir, and writes out_path
    where each line is:
      {
        "custom_id": "request-N",
        "method": "POST",
        "url": "/v1/chat/completions",
        "body": { … }
      }
    """
    i = 0
    with open(out_path, "w") as fout:
        for fname in sorted(os.listdir(prompts_dir)):
            if not fname.endswith("_prompts.jsonl"):
                continue
            path = os.path.join(prompts_dir, fname)
            with open(path) as fin:
                for line in fin:
                    i += 1
                    record = json.loads(line)
                    user_text = record["prompt"]
                    entry = {
                        "custom_id": f"request-{i}",
                        "method": "POST",
                        "url": url,
                        "body": {
                            "model": model,
                            "messages": [
                                {"role": "user",   "content": user_text}
                            ],
                            # you can tweak max_tokens or add temperature, top_p here
                            "max_tokens": 128 + 32,
                        }
                    }
                    fout.write(json.dumps(entry) + "\n")
    print(f"Wrote {i} requests to {out_path}")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Build batch‐API JSONL for GPT completions"
    )
    parser.add_argument(
        "--prompts_dir", required=True,
        help="Directory containing *_prompts.jsonl files"
    )
    parser.add_argument(
        "--out", required=True,
        help="Path to write the batch‐API JSONL"
    )
    parser.add_argument(
        "--model", default="gpt-4o-mini",
        help="Model name (e.g. gpt-4o-mini)"
    )
    parser.add_argument(
        "--url", default="/v1/chat/completions",
        help="API path to call"
    )
    args = parser.parse_args()
    main(args.prompts_dir, args.out, args.model, args.url)
