import yaml
import numpy as np
from pathlib import Path

class MetamatDsConfig:
    def __init__(self, file_path=Path(__file__).parent / 'metamaterials.yaml'):

        with open(file_path, 'r') as file:
            cfg_dict = yaml.safe_load(file)  # Load YAML into a dictionary

        self.num_lay = cfg_dict['num_lay']
        self.num_mat = cfg_dict['num_mat']
        self.num_ang = cfg_dict['num_ang']
        self.num_wave = cfg_dict['num_wave']

        self.wave = np.linspace(float(cfg_dict['wave']['start']),
                                float(cfg_dict['wave']['end']),
                                int(cfg_dict['wave']['points']))

        self.y_dim = cfg_dict['y_dim']
        self.sub_y_num = cfg_dict['sub_y_num']

        self.th_index = self.num_lay * self.num_mat
        self.x_dim = (self.num_lay * self.num_mat) + self.num_lay

        self.dataset_path = (Path(__file__).parent.parent /
                             'data' /
                             f"{self.num_lay}l_{cfg_dict['dataset_size']}s.h5")

    def print_config(self):
        """
        Print all attributes of the configuration in a well-formatted way.
        """
        print("=" * 40)
        print(f"{'MetamatDsConfig Configuration':^40}")
        print("=" * 40)
        for attr, value in self.__dict__.items():
            if isinstance(value, np.ndarray):
                print(f"{attr:<15}: NumPy Array of shape {value.shape}")
            else:
                print(f"{attr:<15}: {value}")
        print("=" * 40)
