import os
import sys
import pandas as pd
import torch 
import numpy as np 
import torch.nn.functional as F

from tqdm import tqdm
from src.utils.data_loader import DataLoader
from src.utils.scores_loader import ScoresLoader
from torch.utils.data import Dataset
from sklearn.preprocessing import PolynomialFeatures

from torch.nn.utils.rnn import pad_sequence

path = os.getcwd()

if path not in sys.path:
    sys.path.append(path)
    

class TimeseriesFeaturesDataset(Dataset):
    def __init__(self, path_features:str, path_data:str, path_scores:str, device:str, seed:int, testing=False):
        self.df_features = pd.read_csv(path_features) 
        if testing : 
            self.df_features = self.df_features.head(10)
        self.path_scores = path_scores
        self.path_data = path_data
        self.device = device
        self.df_shuffled = self.df_features.sample(frac=1, random_state=seed).reset_index(drop=True)
        
        # Load scores and labels with ts_names
        print("loading scores and labels")
        self.load_with_ts_names = []
        self.ts_names = []
        for idx in tqdm(range(len(self.df_features)), desc="Formatting names for loading scores and labels"):
            self.ts_names.append(self.df_features.iloc[idx, 0])# remove .csv extension exemple : "dataset/ts_name.csv" -> "dataset/ts_name"

        print("loading scores from files")
        scores_list = ScoresLoader(self.path_scores).load(self.ts_names)[0]

        print("loading labels from files")
        labels_list = DataLoader(self.path_data).load_timeseries(self.ts_names)[1]

        # Création directe des dicts :
        print("Creating dictionaries for scores and labels")       
        self.dic_ts_to_scores = dict(zip(self.ts_names, scores_list))
        self.dic_ts_to_labels = dict(zip(self.ts_names, labels_list))

    def __len__(self):
        return len(self.df_features)

    def __getitem__(self, idx):
        ts_name = self.df_features.iloc[idx, 0]

        features_array = np.array(self.df_features.iloc[idx, 1:].values, dtype=np.float32)  # Conversion explicite
        features = torch.tensor(features_array, dtype=torch.float32, requires_grad=False).to(self.device)
        
        scores = torch.tensor(self.dic_ts_to_scores[ts_name], dtype=torch.float32, requires_grad=False).to(self.device)
        label = torch.tensor(self.dic_ts_to_labels[ts_name], dtype=torch.float32, requires_grad=False).to(self.device)
        
        return features, scores, label
    
if __name__ == "__main__":
    pass