import random
import cv2
from PIL import Image
import os


def shuffle_except_first(lst):
    if len(lst) <= 1:
        return lst[:], []

    original = lst[1:]
    indices = list(range(1, len(lst)))

    shuffled_indices = indices[:]
    random.shuffle(shuffled_indices)
    shuffled_part = [lst[i] for i in shuffled_indices]
    shuffled_list = [lst[0]] + shuffled_part

    return shuffled_list, shuffled_indices


def get_video_frames(video_path: str) -> list:
    cap = cv2.VideoCapture(video_path)

    if not cap.isOpened():
        raise ValueError("Error: Cannot open video.")

    box = (0, 0, 640, 330) ## cut off time
    all_frames = []
    while True:
        ret, frame = cap.read()
        if not ret:
            break
        all_frames.append(Image.fromarray(frame).crop(box))

    cap.release()
    cv2.destroyAllWindows()
    return all_frames


def save_images_as_gif(images, output_path, duration=500, loop=0):
    """
    Save a list of PIL images as an animated GIF.

    Args:
        images (List[PIL.Image.Image]): List of PIL Image objects.
        output_path (str): File path to save the output GIF (e.g., 'output.gif').
        duration (int): Duration between frames in milliseconds.
        loop (int): Number of loops (0 means infinite).
    """
    if not images:
        raise ValueError("Image list is empty.")

    images[0].save(
        output_path,
        save_all=True,
        append_images=images[1:],
        duration=duration,
        loop=loop,
        format="GIF"
    )


def save_frames_to_path(frames, path):
    all_paths = []
    os.makedirs(path, exist_ok=True)
    for i, img in enumerate(frames):
        name = os.path.join(path, f'frame_{i}.jpg')
        img.save(name)
        all_paths.append(name)
    return all_paths