#!/usr/bin/env bash
# run_grid_experiments.sh
set -euo pipefail

# ----- CLI -----
RESULTS_DIR=""
PROGRAM_FOLDER=""
MAX_COMBOS=0     # 0 = unlimited combos in parallel
MAX_PROCS=0      # passed to inner script; 0 = unlimited program workers

declare -a CONFIGS=()
declare -a PROGRAMS=()
declare -a TOPOLOGIES=()

usage() {
  cat >&2 <<USAGE
Usage:
  $0 --results-dir DIR --program-folder DIR \\
     --config PATH [--config PATH ...] \\
     --program NAME [--program NAME ...] \\
     --topology NAME [--topology NAME ...] \\
     [--max-combos N] [--max-procs N]

Notes:
  * All runs share one common results directory (a timestamped subdir under --results-dir).
  * --max-combos limits how many config×topology runs are active at once.
  * --max-procs limits concurrent program workers inside each run (forwarded).
USAGE
}

while [[ $# -gt 0 ]]; do
  case "$1" in
    --results-dir)   RESULTS_DIR=$2; shift 2;;
    --program-folder)PROGRAM_FOLDER=$2; shift 2;;
    --config|-c)     CONFIGS+=("$2"); shift 2;;
    --program|-p)    PROGRAMS+=("$2"); shift 2;;
    --topology|-t)   TOPOLOGIES+=("$2"); shift 2;;
    --max-combos)    MAX_COMBOS=$2; shift 2;;
    --max-procs)     MAX_PROCS=$2; shift 2;;
    -h|--help)       usage; exit 0;;
    --) shift; break;;
    -*) echo "Unknown option: $1" >&2; usage; exit 1;;
    *)  echo "Unexpected argument: $1" >&2; usage; exit 1;;
  esac
done

# ----- validation -----
[[ -n "${RESULTS_DIR}"    ]] || { echo "--results-dir is required" >&2; exit 1; }
[[ -n "${PROGRAM_FOLDER}" ]] || { echo "--program-folder is required" >&2; exit 1; }
(( ${#CONFIGS[@]}    ))      || { echo "At least one --config is required" >&2; exit 1; }
(( ${#PROGRAMS[@]}   ))      || { echo "At least one --program is required" >&2; exit 1; }
(( ${#TOPOLOGIES[@]} ))      || { echo "At least one --topology is required" >&2; exit 1; }

[[ -d "$RESULTS_DIR"    ]] || { echo "Results dir '$RESULTS_DIR' not found"; exit 1; }
[[ -d "$PROGRAM_FOLDER" ]] || { echo "Program folder '$PROGRAM_FOLDER' not found"; exit 1; }

# ----- common parent directory -----
STAMP=$(date +'%Y%m%d_%H%M%S')
COMMON_DIR="${RESULTS_DIR}/grid_${STAMP}"
mkdir -p "$COMMON_DIR"
echo "➜ Common results directory: $COMMON_DIR"

# ----- small helper to throttle background jobs -----
throttle() {
  local limit=$1
  (( limit > 0 )) || return 0
  while (( $(jobs -rp | wc -l) >= limit )); do
    sleep 0.2
  done
}

# ----- submit all config × topology combos -----
for cfg in "${CONFIGS[@]}"; do
  [[ -f "$cfg" ]] || { echo "Config '$cfg' not found" >&2; exit 1; }
  for topo in "${TOPOLOGIES[@]}"; do
    echo "→ Queueing: config=$(basename "$cfg")  topology=$topo"
    ./run_experiments.sh \
      --topology-type "$topo" \
      --max-procs "$MAX_PROCS" \
      "$COMMON_DIR" "$cfg" "$PROGRAM_FOLDER" "${PROGRAMS[@]}" &
    throttle "$MAX_COMBOS"
  done
done

# ----- wait for all combos -----
wait
echo "✓ All grid runs completed. Results under: $COMMON_DIR"
