import os
import chess
import chess.svg
from IPython.display import SVG, display

"""
Standard Algebraic Notation (SAN)
1. d4 d5
Forsyth-Edwards Notation (FEN)
rnbqkbnr/ppp1pppp/8/3p4/3P4/8/PPP1PPPP/RNBQKBNR w KQkq - 0 2
"""

def find_checkmate_move(moves_san):
    """
    Find the move that results in checkmate if it exists.
    :param moves_san: A list of chess moves in Standard Algebraic Notation (SAN).
    :return: The move that results in checkmate, or "No checkmate move found".
    """
    # Initialize a new chess board
    board = chess.Board()
    
    # Apply the given moves to the board
    for move_san in moves_san:
        try:
            move = board.parse_san(move_san)
            board.push(move)
        except ValueError:
            return "Invalid move in input"
    
    # Generate all possible legal moves
    for move in board.legal_moves:
        board_copy = board.copy()
        board_copy.push(move)
        
        # Check if the move results in checkmate
        if board_copy.is_checkmate():
            return board.san(move)
    
    return "No checkmate move found"

# %%

def preprocess_input(input_str):
    """
    Preprocess the input string by removing move numbers and periods.
    :param input_str: The input string of chess moves.
    :return: A list of preprocessed chess moves.
    """
    moves_san = input_str.split()
    moves_san = [move for move in moves_san if not move.endswith('.')]
    return moves_san

# %%

import imageio
from IPython.display import Image
import cairosvg

# Function to visualize the board after each move
def visualize_final(san_moves):
    # Initialize a chess board
    board = chess.Board()

    # Loop through the moves and display the board step by step
    for i, move in enumerate(san_moves):
        # Push the move in Standard Algebraic Notation (SAN)
        board.push_san(move)
        # Display the current board as SVG
        if i == len(san_moves)-1:
            display(SVG(chess.svg.board(board=board)))
    return board

def visualize_one(board, san):
    board1 = board.copy()
    board.push_san(san)
    display(SVG(chess.svg.board(board=board)))
    return board1, board

# Function to visualize moves and generate an animated GIF
def animate_game(san_moves, output_filename='chess_game.gif', delay=1):
    board = chess.Board()
    images = []
    
    # Create folder to store temporary PNGs
    tmp_dir = "tmp_chess"
    if not os.path.exists(tmp_dir):
        os.makedirs(tmp_dir)

    file_names = []
    # Apply moves one by one and save board states as PNGs
    for i, move in enumerate(san_moves):
        board.push_san(move)
        svg_data = chess.svg.board(board=board)
        png_filename = f"{tmp_dir}/board_{i}.png"
        file_names.append(png_filename)
        cairosvg.svg2png(bytestring=svg_data, write_to=png_filename)
        images.append(imageio.imread(png_filename))
    
    # Generate GIF from images
    if os.path.exists(output_filename):
        os.remove(output_filename)
    imageio.mimsave(output_filename, images, duration=delay)
    
    # Clean up temporary PNGs
    for png_file in file_names:
        os.remove(png_file)

    # Display the resulting GIF
    display(Image(filename=output_filename))
    return Image(filename=output_filename)

# %%

# Example usage
input_str = '1. e4 e5 2. Nf3 d6 3. d4 exd4 4. Nxd4 Nf6 5. Nc3 Qe7 6. Bd3 d5 7. O-O dxe4 8. Re1 Be6 9. Nxe6 fxe6 10. Bxe4 Nxe4 11. Nxe4 Nd7 12. Bg5 Qb4 13. Qg4 Qd4 14. Qxe6+ Be7 15.	'
moves_san = preprocess_input(input_str)
print(moves_san)
#animate_game(moves_san)
target = find_checkmate_move(moves_san)
print(target)
predicted = "Nd6#"

board = visualize_final(moves_san)
old, new = visualize_one(board, target)
old, new = visualize_one(old, predicted)
print(new.is_checkmate())
# %%
