import re
import transformers
import torch
from openai import OpenAI


from utils.promptsg1 import distiller, \
    problem_solver, problem_solver_prompt, general_code_template, \
    evaluator, evaluator_prompt, \
    eval_distiller, eval_distiller_prompt, \
    final_evaluator, final_evaluator_prompt
from utils import execute_py_code, peel_py_mk_wrapper, extract_field, eval_for_exact_matching_with_no_punctuation
from utils.bot_templates import bot_template_game24, bot_template_checkmate, bot_template_word_sorting
from utils.chroma import ChromaVdb

import multiprocessing
import time


MAXTOK = 1024
TEMP = 0.4

# 46.4%, naive 42.6%, any 73.6%, tn 51.1%
class LLM:
    def __init__(self, model_id, api_key=None, api_base=None):
        self.model_id = model_id

        if api_key is None:
            self.local = True
            self.pipeline = transformers.pipeline(
                "text-generation",
                model=self.model_id,
                model_kwargs={"torch_dtype": torch.bfloat16},
                device_map = 'auto'
            )
        else:
            self.local = False
            self.client = OpenAI(
                # This is the default and can be omitted
                api_key=api_key,
                base_url=api_base
            )

    def __call__(self, system_prompt, user_prompt):
        messages = []

        if system_prompt:
            messages.append(
                {"role": "system", "content": system_prompt},
            )
        messages.append(
            {"role": "user", "content": user_prompt},
        )

        if not self.local:
            response = self.client.chat.completions.create(
                model=self.model_id,
                messages=messages,

                max_tokens=MAXTOK,
                temperature=TEMP,
                #top_p=0.9,
            )
            
            # Extract and return the assistant's response
            respond = response.choices[0].message.content
            return respond
        else:
            prompt = self.pipeline.tokenizer.apply_chat_template(
                messages, 
                tokenize=False, 
                add_generation_prompt=True
            )

            terminators = [
                self.pipeline.tokenizer.eos_token_id,
                self.pipeline.tokenizer.convert_tokens_to_ids("<|eot_id|>")
            ]

            outputs = self.pipeline(
                prompt,
                max_new_tokens=MAXTOK,
                eos_token_id=terminators,
                do_sample=True,
                temperature=TEMP,
                #top_p=0.9,
            )

            respond = outputs[0]["generated_text"][len(prompt):]
            return respond
        
class Workflow:
    def __init__(self, model_id=None, api_key=None, api_base=None):
        self.model_id = model_id
        self.api_key = api_key
        self.api_base = api_base

        self.llm = LLM(self.model_id, self.api_key, self.api_base)

    def run(self, question, query=None, target=None):
        l = []
        d = {
            'success': 0,
            'non-sat': 0,
            'error': 0,
 
            'num_attempts': 0,
            'corrected': 0,
            'wronged': 0,
            'others': 0,

            'tp': 0,
            'tn': 0,
            'unsure': 0,
            'selected': False,
            'first_correct': False,
        }

        wrong_selected = False
        any_selected = False

        for i in range(12):
            res = self.llm('You are a brilliant problem solver, try to solve the user questions correctly and cleverly. Give only one most possible trial.', question)
            field = extract_field(res, 'Answer')
            if not field:
                res2 = self.llm('You are a answer formatter. Your goal is to give user a clean formatted result from a unstructred plain text result. The desirable result should be in the format as the following example:\nAnswer:\nababdon bell critus\n\n Extract the result from the plain text answer exactly into this format, no more no less. Do not answer the question yourself.', f"Question:\n{question}\nPlain text answer:\n{res}. Give the formatted answer:")
                field = extract_field(res2, 'Answer')

            #How to analysis: compare the words in the solution two at a time, e.g. given B A D C, the provided solution is A B C D. Your analysis is to first compare A and B, then B and C, and at last C and D.
            final_evaluator = """
You are a evaluator who is good at reasoning and math.

Given a question and a solution to this problem, please verify whether this solution solves the problem correctly.
You analyze step-by-step and make sure that your final answer is consistent with your analysis!
Do not show your expected answer, just analyze the given answer and the question.

DO NOT try to solve or modify the answer.

At last give your evaluation result in two of the following options:

EVAL: CORRECT

or

EVAL: WRONG
"""

            final_evaluator_prompt = """
User's question:
{question}

Problem Solver's solution:
{distilled_answer}

Eval whether this solution answer the question correctly?
"""
            final_evaluator_text = final_evaluator_prompt.format(question=question, distilled_answer=field)
            correct = self.llm(final_evaluator, final_evaluator_text)

            cfield_cnt = 0
            NUM_EVAL = 1
            NUM_MAJOR = (NUM_EVAL - 1) / 2
            for j in range(NUM_EVAL):
                final_evaluator_text = final_evaluator_prompt.format(question=question, distilled_answer=field)
                correct = self.llm(final_evaluator, final_evaluator_text)

                cfield = extract_field(correct, 'EVAL')
                if not cfield:
                    correct2 = self.llm('You are a answer formatter. Your goal is to give user a clean formatted result from a unstructred plain text result. The desirable result should be in the format as the following example:\nEVAL: CORRECT\n\nThe answer should be only one line. Extract the result from the plain text answer exactly into this format, no more no less. Do not answer the question yourself.', f"Question:\n{final_evaluator_text}\nPlain text answer:\n{correct}. Give the formatted answer:")
                    cfield = extract_field(correct2, 'Answer')
                
                if "CORRECT" in cfield or "Correct" in cfield or "correct" in cfield:
                    cfield_cnt += 1
            cfield = cfield_cnt > NUM_MAJOR
            
            cgt = eval_for_exact_matching_with_no_punctuation(query, field, target)

            if i == 0:
                d['first_correct'] = cgt
            any_selected = cfield or any_selected

            if cgt:
                d['success'] += 1
                if cfield:
                    d['tp'] += 1
                    d['selected'] = True
            
            if not cfield and not cgt:
                d['tn'] += 1
            
            if cfield and not cgt:
                print("FP")
                if not d['selected']:
                    print("wrong selected")
                    wrong_selected = True
                print(correct)
                d['non-sat'] += 1
            elif not cfield and cgt:
                print("FN")
                print(correct)

            
            d['num_attempts'] += 1
            d['others'] += 1

        if wrong_selected:
            d['selected'] = False
        if not any_selected:
            d['selected'] = d['first_correct']
        return d