import numpy as np
import gym
import matplotlib.pyplot as plt
from gym.spaces import Box, Dict
from IPython import embed

import sys
np.set_printoptions(threshold=sys.maxsize)
np.set_printoptions(edgeitems=30, linewidth=100000, formatter=dict(float=lambda x: "%.3g" % x))

class GoalReachingWrapper(gym.Wrapper):
    def __init__(
            self,
            env,
            heatmap_shape,
            beta=1.0,
            add_true_rew=False,
            episodic=False,
            use_cnn=False,
            partial_obs=False,
            salesman=False,
            level_string=""
        ):
        
        super().__init__(env)

        self.partial_obs = partial_obs
        self.salesman = salesman
        self.use_cnn = use_cnn
        self.episodic = episodic
        self.level_string = level_string
        self.x, self.y = -1, -1

        self.augment = True
        self.proc_gen = False

        # initialize objects for measuring coverage
        self.covered_states = np.zeros(heatmap_shape)
        self.episodic_covered_states = np.zeros(heatmap_shape)
        self.env = env
        obs = self.env.reset(random_gen=False, level_string=level_string)
        
        # initialize heatmap for augmentation        
        self.heatmap_shape = heatmap_shape
        self.heatmap = np.ones(heatmap_shape) + 1000 # set all visited already

        if partial_obs == False and self.pixel_obs == False:
            obs_ = np.argmax(obs, axis=0)

            # set area around goal not visited
            goal_pos = np.where(obs_ == 1)
            goal_pos_x = goal_pos[0][0]
            goal_pos_y = goal_pos[1][0]
            self.heatmap[goal_pos_x - 3: goal_pos_x + 4, goal_pos_y - 3: goal_pos_y + 4] = 1

            self.covered_states[np.where(obs_ == 2)] = -1
            self.episodic_covered_states[np.where(obs_ == 2)] = -1
        else:
            state = self.env.get_state()
            for obj in state["Objects"]:
                if obj["Name"] == "wall":
                    x, y = obj["Location"]
                    self.episodic_covered_states[x,y] = -1
                    self.covered_states[x,y] = -1
                elif obj["Name"] == "avatar":
                    x, y = obj["Location"]
                    self.x, self.y = x, y
                    self.episodic_covered_states[x,y] = 1
                    self.covered_states[x,y] = 1

        # count-related params
        self.add_true_rew = add_true_rew
        self.beta = beta

        # build observation space
        self.old_obs_space_shape = env.observation_space.shape

        # will NOT use a CNN for partial observations of size 5x5, but can still use CNN for the heatmaps
        if use_cnn == True and partial_obs == False:
            img_obs = {"image" : Box(-np.inf, np.inf, shape = (1, self.old_obs_space_shape[1], self.old_obs_space_shape[2]))}
        else:
            img_obs = {"image" : Box(-np.inf, np.inf, shape = (np.array(self.old_obs_space_shape).prod(), ))}

        self.observation_space = Dict({
            "image" : img_obs["image"]
        })
                    
        if self.augment == True:
            if use_cnn == True:
                heatmap_obs = {"heatmap_and_pos" : Box(-np.inf, np.inf, shape = (2, heatmap_shape[0], heatmap_shape[1])),}
            else:
                heatmap_obs = {"heatmap_and_pos" : Box(-np.inf, np.inf, shape = (np.array(heatmap_shape).prod() * 2, ))}
                
            self.observation_space["heatmap_and_pos"] = heatmap_obs["heatmap_and_pos"]

        print("======= OBS SPACE =======")
        print(self.observation_space)
    
    def step(self, action):
        # step environment
        obs, r, d, info = self.env.step(action)

        # get new agent pos
        if self.partial_obs or self.pixel_obs:
            state = self.env.get_state()
            for obj in state["Objects"]:
                if obj["Name"] == "avatar":
                    x, y = obj["Location"]
                    break
        else:
            x, y = np.unravel_index(np.argmax(obs[0], axis=None), obs[0].shape)

        self.x, self.y = x, y

        try:
            goal_pos = np.where(np.argmax(obs, axis=0) == 1)
            goal_pos_x = goal_pos[0][0]
            goal_pos_y = goal_pos[1][0]

            if self.x == goal_pos_x and self.y == goal_pos_y:
                d = True

        except:
            d = True

        # compute intrinsic reward
        if self.salesman == True:
            count_rew = 1 if self.heatmap[x, y] == 1 else 0
        else:
            count_rew = self.beta / np.sqrt(self.heatmap[x, y])

        # comput coverage
        self.covered_states[x, y] = 1
        self.episodic_covered_states[x, y] = 1
        coverage = np.around(len(np.where(self.covered_states == 1)[0]) /  (len(np.where(self.covered_states == 0)[0]) + len(np.where(self.covered_states == 1)[0])) * 100, 2)
        episodic_coverage = np.around(len(np.where(self.episodic_covered_states == 1)[0]) /  (len(np.where(self.episodic_covered_states == 0)[0]) + len(np.where(self.episodic_covered_states == 1)[0])) * 100, 2)
        maze_coverage = {"coverage" : coverage, "episodic_coverage": episodic_coverage}
        maze_coverage["task_reward"] = r
        maze_coverage["heatmap"] = self.heatmap

        # increment count
        self.heatmap[x, y] += 1

        # add true reward
        if self.add_true_rew == True:
            r = r + count_rew
        else:
            r = count_rew
            
        # compute obs
        obs = self.get_obs(obs)
        return obs, r, d, {**info, **maze_coverage}

    def get_obs(self, obs):
        
        # only get player, floor, and walls
        obs = obs[[0,2]]

        # get agent position in the observation (also works with partial_obs)
        x, y = np.unravel_index(np.argmax(obs[0], axis=None), obs[0].shape)
        
        # tranform to one channel only with indexes
        obs = np.argmax(obs, axis=0)

        # manually set agent position
        max_num = np.max(obs)
        obs[x, y] = max_num + 1

        # build observation
        if self.use_cnn == True and self.partial_obs == False:
            img_obs = {"image" : np.array(obs) / 2}
        else:
            img_obs = {"image" : np.array(obs).flatten() / 2}
        
        obs = {"image" : img_obs["image"]}
                
        if self.augment == True:
            pos_map = np.zeros((self.heatmap_shape))[None, :]
            pos_map[0, self.x, self.y] = 1
            
            # if using salesman reward we only care about the binary map
            if self.salesman == True:
                heatmap_ = np.sign(self.heatmap - 1)
            else:
                heatmap_ = self.heatmap / np.max(self.heatmap) 

            if self.use_cnn == True:
                obs["heatmap_and_pos"] = np.stack([
                    heatmap_[None,:],
                    pos_map
                ]).squeeze()
            else:
                obs["heatmap_and_pos"] = np.concatenate([
                    heatmap_.flatten(),
                    pos_map.flatten()
                ])
            
        return obs

    def reset(self):
        # reset env
        if self.proc_gen:
            obs = self.env.reset(random_gen=True)
        else:
            obs = self.env.reset(random_gen=False, level_string=self.level_string)

        # always reset in this goal-reaching objective
        self.heatmap = np.ones(self.heatmap_shape) + 1000

        # reset for computing episode metrics
        self.episodic_covered_states = np.zeros((self.heatmap_shape))
        if self.partial_obs == True or self.pixel_obs == True:
            state = self.env.get_state()
            for obj in state["Objects"]:
                if obj["Name"] == "wall":
                    x, y = obj["Location"]
                    self.episodic_covered_states[x,y] = -1
                    self.covered_states[x,y] = -1
                elif obj["Name"] == "avatar":
                    x, y = obj["Location"]
                    self.episodic_covered_states[(x,y)] = 1
                    self.covered_states[x,y] = 1
        else:
            x, y = np.unravel_index(np.argmax(obs[0], axis=None), obs[0].shape)            
            obs_ = np.argmax(obs, axis=0)

            # set area around goal not visited
            goal_pos = np.where(obs_ == 1)
            goal_pos_x = goal_pos[0][0]
            goal_pos_y = goal_pos[1][0]
            self.heatmap[goal_pos_x - 3: goal_pos_x + 4, goal_pos_y - 3: goal_pos_y + 4] = 1

            self.episodic_covered_states[np.where(obs_ == 2)] = -1
            self.episodic_covered_states[x,y] = 1
            self.covered_states[x,y] = 1

        self.x, self.y = x, y
        obs = self.get_obs(obs)
        return obs