"""
Provides utility functions for logging and debugging with strict INFO-only output when verbose=False.
"""

import logging
import json
import yaml
from rich.console import Console
from rich.logging import RichHandler
from rich.theme import Theme
import os
import re
import csv
import time
from typing import Dict


class InfoOnlyFilter(logging.Filter):
    def filter(self, record: logging.LogRecord) -> bool:
        return record.levelno == logging.INFO

# --------------------------------------------------
# Rich Setting
# --------------------------------------------------
custom_theme = Theme(
    {
        "user": "green",
        "agent": "blue",
        "info": "cyan",
        "warning": "yellow",
        "error": "red",
        "success": "green",
    }
)
console = Console(theme=custom_theme)


def setup_logger(name: str, verbose: bool = False) -> logging.Logger:
    """
    Set up a Logger instance with RichHandler.
    - verbose=True  -> DEBUG level
    - verbose=False -> only INFO level
    """
    logger = logging.getLogger(name)

    if logger.hasHandlers():
        logger.handlers.clear()

    if verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO

    logger.setLevel(level)

    handler = RichHandler(
        console=console,
        show_time=True,
        show_level=True,
        show_path=False,
    )
    handler.setLevel(level)
    formatter = logging.Formatter(
        "%(name)s | %(filename)s:%(lineno)d | %(levelname)s | %(message)s"
    )
    handler.setFormatter(formatter)

    #if not verbose:
    handler.addFilter(InfoOnlyFilter())

    logger.addHandler(handler)

    return logger


# --------------------------------------------------
# YAML & JSON Util
# --------------------------------------------------
def load_config(config_path: str) -> dict:
    """
    Load and parse the YAML configuration file.
    """
    logger = setup_logger("config-loader", logging.INFO)
    try:
        with open(config_path, "r", encoding="utf-8") as file:
            config = yaml.safe_load(file)
        return config
    except FileNotFoundError:
        logger.error(f"The configuration file '{config_path}' was not found.")
        return None
    except yaml.YAMLError as e:
        logger.error(f"Failed to parse YAML file - {e}")
        return None


def load_jsonl(path: str):
    """
    Load records from a JSONL file.
    """
    logger = setup_logger("jsonl-loader", logging.INFO)
    records = []
    try:
        with open(path, "r", encoding="utf-8") as f:
            for line in f:
                records.append(json.loads(line.strip()))
        logger.debug(f"Loaded {len(records)} records from {path}")
    except FileNotFoundError:
        logger.error(f"JSONL file not found: {path}")
    except json.JSONDecodeError as e:
        logger.error(f"JSON decode error in {path} - {e}")
    return records

def dump_json_output(output, save_path=None):
    if save_path is None:
        raise ValueError("save_path must be provided.")
    with open(save_path, 'w', encoding='utf-8') as f:
        json.dump(output, f, ensure_ascii=False, indent='\t')

def load_json(path: str):
    """
    Load a JSON file.
    """
    logger = setup_logger("json-loader", logging.INFO)
    try:
        with open(path, "r", encoding="utf-8") as f:
            data = json.load(f)
        logger.debug(f"Loaded JSON from {path}")
        return data
    except FileNotFoundError:
        logger.error(f"JSON file not found: {path}")
    except json.JSONDecodeError as e:
        logger.error(f"JSON decode error in {path} - {e}")
    return None

def write_csv_row(values, filename):
    open_trial = 0

    while True:
        if open_trial > 10:
            raise Exception("something wrong")

        try:

            with open(filename, "a", encoding="utf-8") as f:

                writer = csv.writer(f)
                writer.writerow(values)
            break
        except:
            print("open failed")
            continue
        

# Q1: No  
# Q2: No  
# Q3: No  
# Q4: No  
# Q5: Yes  
# Q6: Partially  
# Q7: No  
# Q8: No  
# Q9: No  
# Q10: Partially
def parse_evaluation_results(output: str, checklist_num: int):
    pattern = r'(?:<)?Q(\d+)(?:>)?: (Yes|No|Partially)'
    matches = re.findall(pattern, output)
    if not matches:
        return None
    
    parsed_dict = {f"{num}": answer for num, answer in matches}
    if len(parsed_dict) != checklist_num:
        return None
    
    return parsed_dict

def parse_evaluation_results_individual(output: str, checklist_num: int):
    pattern = r'(?:<)?Q(\d+)(?:>)?: (A|B|Tie)'
    matches = re.findall(pattern, output)
    if not matches:
        return None
    
    parsed_dict = {f"{num}": answer for num, answer in matches}
    if len(parsed_dict) != checklist_num:
        return None
    
    return parsed_dict


def parse_llm_judge_results(output):
    # Updated pattern to match [RESULT] A/B, [Response A/B], and [Result] Response A/B formats
    pattern = r"""
        \[RESULT\]\s*(A|B)|                     # Matches [RESULT] A or B directly
        \[RESULT:\s*(A|B)\]|                    # Matches [RESULT: A] or [RESULT: B]
        \[Response\s+(A|B)\]|                   # Matches [Response A] or [Response B]
        \[Result\]\s+Response\s+(A|B)|          # Matches [Result] Response A or B
        \[Result:\s*(A|B)\]|                    # Matches [Result: A] or [Result: B]
        \[Result\]\s*(A|B)|                     # Matches [Result] A or B directly
    """

    matches = re.findall(pattern, output, re.IGNORECASE | re.VERBOSE)

    # Flatten the matches and filter out empty strings, then take the first valid result
    results = [item for sublist in matches for item in sublist if item]
    if not results:
        return None, None

    result = results[0]
    # Attempt to extract feedback based on the presence of "[RESULT]" or falling back to the entire output
    feedback = (
        output.split("[RESULT]")[0].strip()
        if "[RESULT]" in output
        else output.split("\n")[0].strip()
    )
    return feedback, result



def parse_llm_judge_results_absolute(output):
    # Extended pattern to match more variations of result presentation
    # pattern = r"""
    #     (?:\[RESULT\]|Score:|score:|Result:|\[Result\]|score of|\(|\[|\])\s*  # Match different prefixes including '[RESULT]', 'Score:', etc.
    #     (?:\[RESULT\]|Score:|score:|Result:|\[Result\]|score of|\(|\[|\])\s*
    #     |(\d+)\s*                               # Catch trailing numbers
    #     |\((\d+)\)                              # Catch numbers within parentheses
    #     |\[(\d+)\]                              # Catch numbers within brackets
    # """
    pattern = r"""(?:\[RESULT\]|Score|\[SCORE\]|\[RESULT\]:|Score:|score:|Result:|\[Result\]|score of)\s*(?:\(\s*|\[\s*|)\s*(\d+)"""
    matches = re.search(pattern, output, re.IGNORECASE | re.VERBOSE)

    if matches:
        # Extract the first group that matches (ignoring None)
        result = next((int(match) for match in matches.groups() if match), None)
        if result is not None:
            feedback = (
                output.split("[RESULT]")[0].strip() if "[RESULT]" in output else output
            )
            return feedback, result

    return None, None

def parse_llm_judge_results_wo_feedback(output):
    if output in ['A', 'B', 'Tie']:
        return output
    
    return None

def convert_question_to_instruction(sentence):
    sentence = sentence.strip()
    sentence = re.sub(r'(?i)Did the response', 'The response should', sentence)
    sentence = re.sub(r'(?i)Does the response', 'The response should', sentence)

    if sentence.endswith('?'):
        sentence = sentence[:-1] + '.'
    return sentence

def check_all_checklist_completion(evaluation_result):
    for _, decision in evaluation_result.items():
        if decision.lower() != 'yes':
            return False
    return True

if __name__ == '__main__':
    example_config_path = '../configs/example.yaml'
    config = load_config(example_config_path)
    print(config)
