"""SAMPLING ONLY."""

import torch
import numpy as np
from tqdm import tqdm
from functools import partial
import os

from ldm.modules.diffusionmodules.util import make_ddim_sampling_parameters, make_ddim_timesteps, noise_like, \
    extract_into_tensor


class DDIMSampler(object):
    def __init__(self, model, schedule="linear", **kwargs):
        super().__init__()
        self.model = model
        self.ddpm_num_timesteps = model.num_timesteps
        self.schedule = schedule

    def register_buffer(self, name, attr):
        if type(attr) == torch.Tensor:
            if attr.device != torch.device("cuda"):
                attr = attr.to(torch.device("cuda"))
        setattr(self, name, attr)

    def make_schedule(self, ddim_num_steps, ddim_discretize="uniform", ddim_eta=0., verbose=True):
        self.ddim_timesteps = make_ddim_timesteps(ddim_discr_method=ddim_discretize, num_ddim_timesteps=ddim_num_steps,
                                                  num_ddpm_timesteps=self.ddpm_num_timesteps,verbose=verbose)
        alphas_cumprod = self.model.alphas_cumprod
        assert alphas_cumprod.shape[0] == self.ddpm_num_timesteps, 'alphas have to be defined for each timestep'
        to_torch = lambda x: x.clone().detach().to(torch.float32).to(self.model.device)

        self.register_buffer('betas', to_torch(self.model.betas))
        self.register_buffer('alphas_cumprod', to_torch(alphas_cumprod))
        self.register_buffer('alphas_cumprod_prev', to_torch(self.model.alphas_cumprod_prev))

        # calculations for diffusion q(x_t | x_{t-1}) and others
        self.register_buffer('sqrt_alphas_cumprod', to_torch(np.sqrt(alphas_cumprod.cpu())))
        self.register_buffer('sqrt_one_minus_alphas_cumprod', to_torch(np.sqrt(1. - alphas_cumprod.cpu())))
        self.register_buffer('log_one_minus_alphas_cumprod', to_torch(np.log(1. - alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recip_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recipm1_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu() - 1)))

        # ddim sampling parameters
        ddim_sigmas, ddim_alphas, ddim_alphas_prev = make_ddim_sampling_parameters(alphacums=alphas_cumprod.cpu(),
                                                                                   ddim_timesteps=self.ddim_timesteps,
                                                                                   eta=ddim_eta,verbose=verbose)
        self.register_buffer('ddim_sigmas', ddim_sigmas)
        self.register_buffer('ddim_alphas', ddim_alphas)
        self.register_buffer('ddim_alphas_prev', ddim_alphas_prev)
        self.register_buffer('ddim_sqrt_one_minus_alphas', np.sqrt(1. - ddim_alphas))
        sigmas_for_original_sampling_steps = ddim_eta * torch.sqrt(
            (1 - self.alphas_cumprod_prev) / (1 - self.alphas_cumprod) * (
                        1 - self.alphas_cumprod / self.alphas_cumprod_prev))
        self.register_buffer('ddim_sigmas_for_original_num_steps', sigmas_for_original_sampling_steps)

    # @torch.no_grad()
    def sample(self,
               S,
               batch_size,
               shape,
               conditioning=None,
               callback=None,
               normals_sequence=None,
               img_callback=None,
               quantize_x0=False,
               eta=0.,
               mask=None,
               x0=None,
               temperature=1.,
               noise_dropout=0.,
               score_corrector=None,
               corrector_kwargs=None,
               verbose=True,
               x_T=None,
               log_every_t=100,
               unconditional_guidance_scale=1.,
               unconditional_conditioning=None,
               time_eps=None,
               time_eps1=None,
               att_mask=None,
               att_mask1=None,
               att_idx=None,
               g_weight=70.,
               g_steps=5,
               prev_latents=None,
               loss_thre=0.15,
               # this has to come in the same format as the conditioning, # e.g. as encoded tokens, ...
               **kwargs
               ):
        if conditioning is not None:
            if isinstance(conditioning, dict):
                cbs = conditioning[list(conditioning.keys())[0]].shape[0]
                if cbs != batch_size:
                    print(f"Warning: Got {cbs} conditionings but batch-size is {batch_size}")
            else:
                if conditioning.shape[0] != batch_size:
                    print(f"Warning: Got {conditioning.shape[0]} conditionings but batch-size is {batch_size}")

        self.make_schedule(ddim_num_steps=S, ddim_eta=eta, verbose=verbose)
        # sampling
        C, H, W = shape
        size = (batch_size, C, H, W)
        print(f'Data shape for DDIM sampling is {size}, eta {eta}')

        samples, intermediates, all_samples, all_x0, all_attns, curr_latents = self.ddim_sampling(conditioning, size,
                                                    callback=callback,
                                                    img_callback=img_callback,
                                                    quantize_denoised=quantize_x0,
                                                    mask=mask, x0=x0,
                                                    ddim_use_original_steps=False,
                                                    noise_dropout=noise_dropout,
                                                    temperature=temperature,
                                                    score_corrector=score_corrector,
                                                    corrector_kwargs=corrector_kwargs,
                                                    x_T=x_T,
                                                    log_every_t=log_every_t,
                                                    unconditional_guidance_scale=unconditional_guidance_scale,
                                                    unconditional_conditioning=unconditional_conditioning, time_eps=time_eps, time_eps1=time_eps1, att_mask=att_mask, att_mask1=att_mask1, att_idx=att_idx, g_weight=g_weight, g_steps=g_steps, prev_latents=prev_latents, loss_thre=loss_thre
                                                    )
        return samples, intermediates, all_samples, all_x0, all_attns, curr_latents

    # @torch.no_grad()
    def ddim_sampling(self, cond, shape,
                      x_T=None, ddim_use_original_steps=False,
                      callback=None, timesteps=None, quantize_denoised=False,
                      mask=None, x0=None, img_callback=None, log_every_t=100,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None, time_eps=None, time_eps1=None, att_mask=None, att_mask1=None, att_idx=None, g_weight=70., g_steps=5, prev_latents=None, loss_thre=0.15):
        device = self.model.betas.device
        b = shape[0]
        if x_T is None:
            img = torch.randn(shape, device=device)
        else:
            img = x_T

        if timesteps is None:
            timesteps = self.ddpm_num_timesteps if ddim_use_original_steps else self.ddim_timesteps
        elif timesteps is not None and not ddim_use_original_steps:
            subset_end = int(min(timesteps / self.ddim_timesteps.shape[0], 1) * self.ddim_timesteps.shape[0]) - 1
            timesteps = self.ddim_timesteps[:subset_end]

        intermediates = {'x_inter': [img], 'pred_x0': [img]}
        time_range = reversed(range(0,timesteps)) if ddim_use_original_steps else np.flip(timesteps)
        total_steps = timesteps if ddim_use_original_steps else timesteps.shape[0]
        print(f"Running DDIM Sampling with {total_steps} timesteps")

        iterator = tqdm(time_range, desc='DDIM Sampler', total=total_steps)
        count_iter = 0
        all_samples = []
        
        all_x0 = []
        all_attns = []
        ca_loss = torch.tensor(100000.)
        curr_latents = []

        for i, step in enumerate(iterator):
            save_attns = []
            index = total_steps - i - 1
            ts = torch.full((b,), step, device=device, dtype=torch.long)

            if mask is not None:
                assert x0 is not None
                img_orig = self.model.q_sample(x0, ts)  # TODO: deterministic forward pass?
                img = img_orig * mask + (1. - mask) * img


            outs, ca_loss, return_latent = self.p_sample_ddim(img, cond, ts, index=index, use_original_steps=ddim_use_original_steps,
                                      quantize_denoised=quantize_denoised, temperature=temperature,
                                      noise_dropout=noise_dropout, score_corrector=score_corrector,
                                      corrector_kwargs=corrector_kwargs,
                                      unconditional_guidance_scale=unconditional_guidance_scale,
                                      unconditional_conditioning=unconditional_conditioning, time_eps=time_eps, time_eps1=time_eps1, att_mask=att_mask, att_mask1=att_mask1, att_idx=att_idx, g_weight=g_weight, g_steps=g_steps, save_attns=save_attns, loss=ca_loss, prev_latents=prev_latents, loss_thre=loss_thre)
            
            curr_latents.append(return_latent)
            
            # Save cross attention
            # assert len(save_attns) > 0
            all_attns.append(save_attns)

            img, pred_x0 = outs
            all_samples.append(img.detach().clone())
            all_x0.append(pred_x0.detach().clone())
            if callback: callback(i)
            if img_callback: img_callback(pred_x0, i)

            if index % log_every_t == 0 or index == total_steps - 1:
                intermediates['x_inter'].append(img)
                intermediates['pred_x0'].append(pred_x0)

        return img, intermediates, all_samples, all_x0, all_attns, curr_latents

    # @torch.no_grad()
    def p_sample_ddim(self, x, c, t, index, repeat_noise=False, use_original_steps=False, quantize_denoised=False,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None, time_eps=None, time_eps1=None, att_mask=None, att_mask1=None, att_idx=None, g_weight=70., g_steps=5, save_attns=None, loss=None, prev_latents=None, loss_thre=0.15):
        b, *_, device = *x.shape, x.device
        
        if unconditional_conditioning is None or unconditional_guidance_scale == 1.:
            att_mask = None
            e_t = self.model.apply_model(x, t, att_mask, None, [], [], [], c)
        else:
            guidance = False
            with torch.enable_grad():
                att_sigmas = np.array(((1 - self.model.alphas_cumprod[self.ddim_timesteps].cpu()) / self.model.alphas_cumprod[self.ddim_timesteps].cpu()) ** 0.5)
                att_sigmas = np.concatenate([[0.0], att_sigmas]).astype(np.float32)
                att_sigmas = torch.from_numpy(att_sigmas).to(0)
                prev_t = t - 1000 // 50
                alpha_prod_t = self.model.alphas_cumprod[t]
                alpha_prod_t_prev = self.model.alphas_cumprod[prev_t] if prev_t >= 0 else torch.tensor(1.0)
                beta_prod_t = 1 - alpha_prod_t
                beta_prod_t_prev = 1 - alpha_prod_t_prev
                variance = (beta_prod_t_prev / beta_prod_t) * (1 - alpha_prod_t / alpha_prod_t_prev)
                iteration = 0
                if att_mask1 != None:
                    loss_thre = loss_thre + 0.03
                    lr = 0.1 # 0.3
                else:
                    loss_thre = loss_thre
                    lr = 0.7
                while loss.item() / g_weight > loss_thre and iteration < 5 and (50 - index - 1) < g_steps:
                    guidance = True
                    x = x.requires_grad_(True)
                    mid_attns = []
                    up_attns = []
                    down_attns = []
                    e_t_cond = self.model.apply_model(x, t, att_mask, None, mid_attns, up_attns, down_attns, c)
                    # print(len(mid_attns))
                    # print(mid_attns[0].shape)
                    # print(len(up_attns))
                    # print(up_attns[0].shape)
                    # print(len(down_attns))
                    # print(down_attns[0].shape)
                    # os._exit(0)

                    computed_loss = 0
                    for att_map in mid_attns:
                        att_b, i, j = att_map.shape
                        H = W = int(i ** 0.5)
                        resize_mask = torch.nn.functional.interpolate(att_mask, (H, W))
                        resize_mask = (resize_mask.squeeze(0) > 0.).int()
                        ca_map_obj = att_map[:, :, att_idx].reshape(att_b, H, W)
                        activation_value = (ca_map_obj * resize_mask).reshape(att_b, -1).sum(dim=-1) / ca_map_obj.reshape(att_b, -1).sum(dim=-1)
                        computed_loss += torch.mean((1 - activation_value) ** 2) # + torch.mean((1 - activation_value1) ** 2)) / 2.
                    
                    for att_map in up_attns[:3]: # [:3]
                        # att_map = up_attns[0]
                        att_b, i, j = att_map.shape
                        H = W = int(i ** 0.5)
                        resize_mask = torch.nn.functional.interpolate(att_mask, (H, W))
                        resize_mask = (resize_mask.squeeze(0) > 0.).int()
                        ca_map_obj = att_map[:, :, att_idx].reshape(att_b, H, W)
                        activation_value = (ca_map_obj * resize_mask).reshape(att_b, -1).sum(dim=-1) / ca_map_obj.reshape(att_b, -1).sum(dim=-1)
                        computed_loss += torch.mean((1 - activation_value) ** 2) # + torch.mean((1 - activation_value1) ** 2)) / 2.
                    
                    for att_map in down_attns[3:]: # [3:]
                        # att_map = up_attns[0]
                        att_b, i, j = att_map.shape
                        H = W = int(i ** 0.5)
                        resize_mask = torch.nn.functional.interpolate(att_mask, (H, W))
                        resize_mask = (resize_mask.squeeze(0) > 0.).int()
                        ca_map_obj = att_map[:, :, att_idx].reshape(att_b, H, W)
                        activation_value = (ca_map_obj * resize_mask).reshape(att_b, -1).sum(dim=-1) / ca_map_obj.reshape(att_b, -1).sum(dim=-1)
                        computed_loss += torch.mean((1 - activation_value) ** 2) # + torch.mean((1 - activation_value1) ** 2)) / 2.

                    computed_loss = computed_loss / 7. # 7.
                    print(f'loss: {computed_loss}')
                    computed_loss = computed_loss * g_weight

                    loss = computed_loss
                    
                    grad_cond = torch.autograd.grad(loss.requires_grad_(True), [x])[0]
                    print(f'grad: {grad_cond.mean()}')

                    x = x - lr * grad_cond
                    iteration += 1
                    torch.cuda.empty_cache()

            # print('update')
            # print(x.shape)
            # os._exit(0)
            with torch.no_grad():
                x_in = torch.cat([x] * 2)
                t_in = torch.cat([t] * 2)
                c_in = torch.cat([unconditional_conditioning, c])
                
                curr_t = t[0].item()
                
                # if not (curr_t > 850):
                #     att_mask =None
                
                mid_attns = []
                up_attns = []
                down_attns = []
                
                e_t_uncond, e_t = self.model.apply_model(x_in, t_in, att_mask, save_attns, mid_attns, up_attns, down_attns, c_in).chunk(2)
                e_t = e_t_uncond + unconditional_guidance_scale * (e_t - e_t_uncond)

        if score_corrector is not None:
            assert self.model.parameterization == "eps"
            e_t = score_corrector.modify_score(self.model, e_t, x, t, c, **corrector_kwargs)

        alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas
        alphas_prev = self.model.alphas_cumprod_prev if use_original_steps else self.ddim_alphas_prev
        sqrt_one_minus_alphas = self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas
        sigmas = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas
        # select parameters corresponding to the currently considered timestep
        a_t = torch.full((b, 1, 1, 1), alphas[index], device=device)
        a_prev = torch.full((b, 1, 1, 1), alphas_prev[index], device=device)
        sigma_t = torch.full((b, 1, 1, 1), sigmas[index], device=device)
        sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index],device=device)

        # current prediction for x_0
        pred_x0 = (x - sqrt_one_minus_at * e_t) / a_t.sqrt()
        if quantize_denoised:
            pred_x0, _, *_ = self.model.first_stage_model.quantize(pred_x0)
        # direction pointing to x_t
        dir_xt = (1. - a_prev - sigma_t**2).sqrt() * e_t
        noise = sigma_t * noise_like(x.shape, device, repeat_noise) * temperature
        if noise_dropout > 0.:
            noise = torch.nn.functional.dropout(noise, p=noise_dropout)
        x_prev = a_prev.sqrt() * pred_x0 + dir_xt + noise

        
        ### Replace prev
        if index > 10 and index != 0 and (prev_latents is not None) and att_mask1 != None:
            resize_mask = torch.nn.functional.interpolate(att_mask1, (x_prev.shape[-2], x_prev.shape[-1]))
            eps_mask = resize_mask > 0.
            x_prev = eps_mask.int() * x_prev + (1 - eps_mask.int()) * prev_latents[50 - index - 1]
            x_prev = x_prev.to(torch.float16)
        return_latent = x_prev.detach().clone()
        return (x_prev, pred_x0), loss, return_latent

    @torch.no_grad()
    def stochastic_encode(self, x0, t, use_original_steps=False, noise=None):
        # fast, but does not allow for exact reconstruction
        # t serves as an index to gather the correct alphas
        if use_original_steps:
            sqrt_alphas_cumprod = self.sqrt_alphas_cumprod
            sqrt_one_minus_alphas_cumprod = self.sqrt_one_minus_alphas_cumprod
        else:
            sqrt_alphas_cumprod = torch.sqrt(self.ddim_alphas)
            sqrt_one_minus_alphas_cumprod = self.ddim_sqrt_one_minus_alphas

        if noise is None:
            noise = torch.randn_like(x0)
        return (extract_into_tensor(sqrt_alphas_cumprod, t, x0.shape) * x0 +
                extract_into_tensor(sqrt_one_minus_alphas_cumprod, t, x0.shape) * noise)

    @torch.no_grad()
    def decode(self, x_latent, cond, t_start, unconditional_guidance_scale=1.0, unconditional_conditioning=None,
               use_original_steps=False):

        timesteps = np.arange(self.ddpm_num_timesteps) if use_original_steps else self.ddim_timesteps
        timesteps = timesteps[:t_start]

        time_range = np.flip(timesteps)
        total_steps = timesteps.shape[0]
        print(f"Running DDIM Sampling with {total_steps} timesteps")

        iterator = tqdm(time_range, desc='Decoding image', total=total_steps)
        x_dec = x_latent
        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((x_latent.shape[0],), step, device=x_latent.device, dtype=torch.long)
            x_dec, _ = self.p_sample_ddim(x_dec, cond, ts, index=index, use_original_steps=use_original_steps,
                                          unconditional_guidance_scale=unconditional_guidance_scale,
                                          unconditional_conditioning=unconditional_conditioning)
        return x_dec