import torch
import torch.nn as nn
import torch.nn.functional as F
from typing import Type, List


class PreActBlock(nn.Module):
    """Pre-activation version of the BasicBlock used in a ResNet.
    The flow goes x (input) -> BN -> ReLU -> Conv instead of x (input) -> Conv -> BN -> ReLU.

    Args:
        in_planes (int): Number of input channels.
        planes (int): Number of output channels.
        stride (int, optional): Stride size. Default is 1.
    """

    expansion: int = 1

    def __init__(self, in_planes: int, planes: int, stride: int = 1) -> None:
        super(PreActBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False
        )
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(
            planes, planes, kernel_size=3, stride=1, padding=1, bias=False
        )
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                )
            )

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        """Forward pass for the PreActBlock."""
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, "shortcut") else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out += shortcut
        return out


class PreActResNet(nn.Module):
    """Pre-activation ResNet model.

    Args:
        block (Type[PreActBlock]): Block type for the pre-activation ResNet.
        num_blocks (List[int]): List containing the number of blocks per layer.
        num_classes (int, optional): Number of output classes. Default is 10.
    """

    def __init__(
        self, block: Type[PreActBlock], num_blocks: List[int], num_classes: int = 10
    ) -> None:
        super(PreActResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.fc = nn.Linear(512, num_classes)

    def _make_layer(
        self, block: Type[PreActBlock], planes: int, num_blocks: int, stride: int
    ) -> nn.Sequential:
        """Create a layer for the PreActResNet.

        Args:
            block (Type[PreActBlock]): Block type for the layer.
            planes (int): Number of planes.
            num_blocks (int): Number of blocks.
            stride (int): Stride size.

        Returns:
            Residual block with the desired configuration.
        """
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        """Forward pass for the PreActResNet."""
        out = self.conv1(x)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.fc(out)
        return out
