#!/bin/bash

# Configuration
lr=$1
seed=$2
epoch=$3
optimiser=$4
weight_decay=$5
dataset=$6
NUM_EIGENTHINGS=50
MODE="power_iter" #  lanczos,  power_iter
RESULTS_DIR="./experiment_results/${dataset}"

# Step 3: Find the generated checkpoint
# Assuming the checkpoint is saved in the format: weight_gradient_hist/finetune_resnet_${}_*/lr_${lr}_seed_${seed}/*_epoch_*.pth
CHECKPOINT_PATTERN="./weight_gradient_hist/finetune_resnet_${dataset}_${optimiser}/lr_${lr}_wd_${weight_decay}/_seed_${seed}/*_epoch_${epoch}.pth"
CHECKPOINT_DIR=$(find . -path "$CHECKPOINT_PATTERN" -type f | head -1)


# Create results directory
mkdir -p "$RESULTS_DIR"

# Results summary file
SUMMARY_FILE="$RESULTS_DIR/eigenvalue_summary.txt"
echo "Experiment Results Summary" > "$SUMMARY_FILE"
echo "=========================" >> "$SUMMARY_FILE"
echo "Date: $(date)" >> "$SUMMARY_FILE"
echo "" >> "$SUMMARY_FILE"

# Main experiment

echo "==============================================="
echo "Running experiment: SEED=$seed, LR=$lr, OPT=$optimiser, epoch=$epoch, WD=$weight_decay"
echo "==============================================="

# Create experiment-specific directory
EXP_DIR="$RESULTS_DIR/${optimiser}_seed_${seed}_lr_${lr}_wd_${weight_decay}"
mkdir -p "$EXP_DIR"

# Log file for this experiment
LOG_FILE="$EXP_DIR/experiment.log"
echo "Starting experiment: SEED=$seed, OPT=$optimiser, LR=$lr, WD=$weight_decay" > "$LOG_FILE"
echo "Timestamp: $(date)" >> "$LOG_FILE"
echo "" >> "$LOG_FILE"


if [ -z "$CHECKPOINT_DIR" ]; then
    echo "ERROR: No checkpoint found for seed=$seed, lr=$lr" | tee -a "$LOG_FILE"
    echo "ERROR: No checkpoint found for seed=$seed, lr=$lr" >> "$SUMMARY_FILE"
    
fi

echo "Found checkpoint: $CHECKPOINT_DIR" | tee -a "$LOG_FILE"

# Step 4: Run main.py to calculate eigenvalues
echo "Step 2: Calculating eigenvalues..." | tee -a "$LOG_FILE"
EIGENVAL_OUTPUT="$EXP_DIR/eigenvalues.txt"

$ENV_ACTIVATION $TORCHRUN_CMD example/evalue_cal.py \
    --mode="$MODE" \
    --dataset="$dataset" \
    --num_eigenthings="$NUM_EIGENTHINGS" \
    --seed="$seed" \
    --cuda \
    --checkpoint_dir="$CHECKPOINT_DIR" \
    --output_excel="$EXP_DIR/eigenvalues_seed${seed}_lr${lr}_wd${weight_decay}_epoch${epoch}.xlsx" \
    2>&1 | tee "$EIGENVAL_OUTPUT"

if [ ${PIPESTATUS[0]} -ne 0 ]; then
    echo "ERROR: Eigenvalue calculation failed for seed=$seed, lr=$lr" | tee -a "$LOG_FILE"
    echo "ERROR: Eigenvalue calculation failed for seed=$seed, lr=$lr" >> "$SUMMARY_FILE"
fi

# Step 5: Extract and save results to summary
echo "Results for SEED=$seed, LR=$lr:" >> "$SUMMARY_FILE"
echo "Checkpoint: $CHECKPOINT_DIR" >> "$SUMMARY_FILE"
grep -A 10 "Eigenvals:" "$EIGENVAL_OUTPUT" >> "$SUMMARY_FILE"
echo "" >> "$SUMMARY_FILE"

echo "Completed experiment: SEED=$seed, LR=$lr" | tee -a "$LOG_FILE"
echo "Results saved to: $EXP_DIR" | tee -a "$LOG_FILE"
echo ""

echo "==============================================="
echo "All experiments completed!"
echo "Summary file: $SUMMARY_FILE"
echo "Individual results in: $RESULTS_DIR"
echo "==============================================="
