# -*- coding: utf-8 -*

import torch
import torch.nn as nn
import torch.nn.functional as F

"""
vgg: ['VGG', 'vgg11', 'vgg16']
"""

class VGG(nn.Module):

    def __init__(self, args, features, init_weights=True):
        super(VGG, self).__init__()
        self.args = args

        if args.data == 'MNIST':
            para_temp = 6
        elif args.data == 'cifar10' or args.data == 'cifar100':
            para_temp = 7

        self.features = features
        self.avgpool = nn.AdaptiveAvgPool2d((para_temp, para_temp))

        if args.dropout == 'no_drop':
            self.classifier = nn.Sequential(
                nn.Linear(512 * para_temp * para_temp, 4096),
                nn.ReLU(inplace=True),
                nn.Linear(4096, 4096),
                nn.ReLU(inplace=True),
                nn.Linear(4096, args.num_classes),
            )
        elif args.dropout == 'dropout_1':
            self.classifier = nn.Sequential(
                nn.Linear(512 * para_temp * para_temp, 4096),
                nn.ReLU(inplace=True),
                nn.Linear(4096, 4096),
                nn.ReLU(inplace=True),
                nn.Dropout(),
                nn.Linear(4096, args.num_classes),
            )
        else:
            self.classifier = nn.Sequential(
                nn.Linear(512 * para_temp * para_temp, 4096),
                nn.ReLU(inplace=True),
                nn.Dropout(),
                nn.Linear(4096, 4096),
                nn.ReLU(inplace=True),
                nn.Dropout(),
                nn.Linear(4096, args.num_classes),
            )

        if init_weights:
            self._initialize_weights()

    def forward(self, x):
        x = self.features(x)
        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.classifier(x)
        return x

    def _initialize_weights(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.constant_(m.bias, 0)


def make_layers(args, cfg, batch_norm=False):
    layers = []
    in_channels = args.input_dim
    bn_flag = 0
    for v in cfg:
        if v == 'M':
            bn_flag = 0
            # list A += B will append the B' elements into A.
            layers += [nn.MaxPool2d(kernel_size=2, stride=2)]
        else:
            bn_flag += 1
            conv2d = nn.Conv2d(in_channels, v, kernel_size=3, padding=2)
            if (batch_norm) and (bn_flag==1):
                layers += [conv2d, nn.BatchNorm2d(v), nn.ReLU(inplace=True)]
            else:
                layers += [conv2d, nn.ReLU(inplace=True)]
            in_channels = v
    return nn.Sequential(*layers)


cfgs = {
    'A': [64, 'M', 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'B': [64, 64, 'M', 128, 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'D': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 'M', 512, 512, 512, 'M', 512, 512, 512, 'M'],
    'E': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 256, 'M', 512, 512, 512, 512, 'M', 512, 512, 512, 512, 'M'],
}


def _vgg(args, cfg, batch_norm, **kwargs):
    model = VGG(args, make_layers(args, cfgs[cfg], batch_norm=batch_norm), **kwargs)
    return model


def vgg11(args, **kwargs):
    if args.bn == 'bn':
        """VGG 11-layer model (configuration "A") with batch normalization
        """
        return _vgg(args, 'A', True, **kwargs)
    else:
        """VGG 11-layer model (configuration "A")
        """
        return _vgg(args, 'A', False, **kwargs)


def vgg16(args, **kwargs):
    if args.bn == 'bn':
        """VGG 16-layer model (configuration "D") with batch normalization
        """
        return _vgg(args, 'D', True, **kwargs)
    else:
        """VGG 16-layer model (configuration "D")
        """
        return _vgg(args, 'D', False, **kwargs)
