"""Experiment entry for CRR baseline (BSuite)."""
import os
import pickle
import time
from typing import Dict, List, Optional, Tuple

import haiku as hk
import jax
import jax.numpy as jnp
import optax
import requests
import wandb
from absl import app, flags, logging
from acme.jax import networks as networks_lib
from acme.jax import utils
from acme.specs import make_environment_spec
from acme.utils.loggers import Logger
from dm_env import Environment

from internal.loggers import generate_experiment_name, logger_fn
from internal.notifier import DiscordNotif
from internal.tracer import PyTracer
from rosmo.agent.crr_discrete.actor import CRREvalActor
from rosmo.agent.crr_discrete.learning import CRRLearner
from rosmo.agent.crr_discrete.network import CRRNetworks
from rosmo.data.bsuite.dataset import env_loader
from rosmo.data.rl_unplugged import atari
from rosmo.types import Array
from rosmo.util.env_loop_observer import (
  EvaluationLoop,
  ExtendedEnvLoopObserver,
  LearningStepObserver,
)

platform = jax.lib.xla_bridge.get_backend().platform
num_devices = jax.lib.xla_bridge.device_count()
logging.warning(f"Compute platform: {platform} with {num_devices} devices.")

notif_url = os.environ.get("NOTIF_WEBHOOK_URL", "")
notif = DiscordNotif(url=notif_url)

# ===== Flags. ===== #
FLAGS = flags.FLAGS
flags.DEFINE_string("exp_id", None, "Experiment id.", required=True)
flags.DEFINE_boolean("debug", True, "Debug run.")
flags.DEFINE_boolean("profile", False, "Profiling run.")
flags.DEFINE_integer("run_number", 1, "Run number of RLU dataset.")
flags.DEFINE_integer(
  "data_percentage", 100, "Percentage of data used for training.", 0, 100
)
flags.DEFINE_integer("seed", int(time.time()), "Random seed.")
flags.DEFINE_float("beta", 0.5, "Beta.")
flags.DEFINE_integer(
  "num_action_samples", 4, "Number of sampled actions for value estimation."
)
flags.DEFINE_bool(
  "use_sarsa_target",
  True,
  "Compute on-policy target using iterator actions rather than sampled "
  "actions.",
)
flags.DEFINE_string("checkpoint", None, "Checkpoint to resume.")
flags.DEFINE_string("game", None, "Game name to run.", required=True)

flags.DEFINE_string("run_set", "", "Run set name for experiment grouping.")
flags.DEFINE_bool("local", False, "Local jobs.")

DATA_DIR = "/mnt_central/datasets/rl_unplugged/tensorflow_datasets"

CONFIG = {
  "num_bins": 20,
  "encoder_layers": [64, 64, 32],
  "prediction_layers": [32],
  "output_init_scale": 0.0,
  "discount": 0.99,
  "batch_size": 128 * 6,  # 6 to match unroll_length
  "learning_rate": 1e-4,
  "target_update_period": 100,
  "grad_updates_per_batch": 1,
  "log_interval": 400,
  "save_period": 10_000,
  "eval_period": 1_000,
  "evaluate_episodes": 2,
  "total_steps": 200_000,
}


# ===== Learner. ===== #
def get_learner(config, networks, data_iterator, logger) -> CRRLearner:
  """Get CRR learner."""
  coeff_fn = None
  learner = CRRLearner(
    networks,
    random_key=jax.random.PRNGKey(config["seed"]),
    discount=config["discount"],
    target_update_period=config["target_update_period"],
    num_bins=config["num_bins"],
    batch_size=config["batch_size"],
    iterator=data_iterator,
    optimizer=optax.adam(learning_rate=config["learning_rate"]),
    num_action_samples=config["num_action_samples"],
    beta=config["beta"],
    coeff_fn=coeff_fn,
    grad_updates_per_batch=config["grad_updates_per_batch"],
    use_sarsa_target=config["use_sarsa_target"],
    logger=logger,
    log_interval=config["log_interval"],
  )
  return learner


# ===== Eval Actor-Env Loop. ===== #
def get_actor_env_eval_loop(
  config, networks: CRRNetworks, environment, observers, logger
) -> Tuple[CRREvalActor, EvaluationLoop]:
  """Get actor, env and evaluation loop."""
  actor = CRREvalActor(
    networks,
    config,
  )
  eval_loop = EvaluationLoop(
    environment=environment,
    actor=actor,
    logger=logger,
    should_update=False,
    observers=observers,
  )
  return actor, eval_loop


def get_env_loop_observers() -> List[ExtendedEnvLoopObserver]:
  """Get environment loop observers."""
  observers = []
  learning_step_ob = LearningStepObserver()
  observers.append(learning_step_ob)
  return observers


# ===== Network ===== #
def get_networks(config, environment) -> CRRNetworks:
  """Get environment-specific networks."""
  env_spec = make_environment_spec(environment)
  logging.info(env_spec)
  num_actions = env_spec.actions.num_values
  num_atoms = config["num_bins"]

  dummy_action = jnp.array(env_spec.actions.generate_value())
  dummy_obs = utils.zeros_like(env_spec.observations)
  dummy_action = utils.add_batch_dim(dummy_action)
  dummy_obs = utils.add_batch_dim(dummy_obs)

  def _torso_fun(observations: Array) -> Array:
    network = hk.Sequential(
      [hk.Flatten(), hk.nets.MLP(config["encoder_layers"])]
    )
    state = network(observations)
    return state

  torso = hk.without_apply_rng(hk.transform(_torso_fun))
  torso_network = networks_lib.FeedForwardNetwork(
    lambda key: torso.init(key, dummy_obs), torso.apply
  )

  def _dummy_state(key):
    encoder_params = torso.init(key, dummy_obs)
    dummy_state = torso.apply(encoder_params, dummy_obs)
    return dummy_state

  def _policy_fun(state: Array) -> Array:
    network = hk.nets.MLP(config["prediction_layers"], activate_final=True)
    head_state = network(state)
    policy = networks_lib.CategoricalHead(num_actions)
    return policy(head_state)

  policy = hk.without_apply_rng(hk.transform(_policy_fun))
  policy_network = networks_lib.FeedForwardNetwork(
    lambda key: policy.init(key, _dummy_state(key)), policy.apply
  )

  def _critic_fun(torso_out: jnp.ndarray, action: jnp.ndarray):
    one_hot_action = hk.one_hot(action, env_spec.actions.num_values)
    network = hk.nets.MLP(
      config["prediction_layers"] + [num_atoms], activate_final=False
    )
    qvalue_logits = network(
      jnp.concatenate([torso_out, one_hot_action], axis=-1)
    )
    return qvalue_logits

  critic = hk.without_apply_rng(hk.transform(_critic_fun))
  critic_network = networks_lib.FeedForwardNetwork(
    lambda key: critic.init(key, _dummy_state(key), dummy_action), critic.apply
  )

  return CRRNetworks(
    torso_network=torso_network,
    policy_network=policy_network,
    critic_network=critic_network,
    log_prob=lambda params, actions: params.log_prob(actions),
    sample=lambda params, key: params.sample(seed=key),
    sample_eval=lambda params, key: params.mode(),
  )


# ===== Configurations ===== #
def get_config(game_name: str) -> Dict:
  """Get experiment configurations."""
  config = CONFIG.copy()
  config["benchmark"] = "bsuite"
  config["algo"] = "CRR"
  config["game_name"] = game_name
  config["seed"] = FLAGS.seed
  config["run_number"] = FLAGS.run_number
  config["ckpt_number"] = FLAGS.ckpt_number
  config["beta"] = FLAGS.beta
  config["num_action_samples"] = FLAGS.num_action_samples
  config["use_sarsa_target"] = FLAGS.use_sarsa_target
  config["data_percentage"] = FLAGS.data_percentage
  config["batch_size"] = 16 if FLAGS.debug else config["batch_size"]
  exp_full_name = generate_experiment_name(f"{FLAGS.exp_id}_{game_name}")
  config["exp_full_name"] = exp_full_name
  logging.info(f"Configs: {config}")
  return config


# ===== Misc. ===== #
def get_logger_fn(
  exp_full_name: str,
  job_name: str,
  is_eval: bool = False,
  config: Optional[Dict] = None,
) -> Logger:
  """Get logger function."""
  save_data = is_eval
  return logger_fn(
    exp_name=exp_full_name,
    label=job_name,
    save_data=save_data and not (FLAGS.debug or FLAGS.profile),
    use_tb=False,
    # use_sota=not (FLAGS.debug or FLAGS.profile),
    use_wb=not (FLAGS.debug or FLAGS.profile),
    use_sota=False,
    config=config,
  )


def main(_):
  """Main program."""
  logging.info(f"Debug mode: {FLAGS.debug}")
  # ===== Profiler. ===== #
  profile_dir = "./profile"
  os.makedirs(profile_dir, exist_ok=True)
  tracer = PyTracer("./", FLAGS.exp_id, with_jax=True)

  # ===== Setup. ===== #
  cfg = get_config(FLAGS.game)
  data_dir = DATA_DIR
  if FLAGS.local:
    data_dir += "/bsuite-v2"

  env, dataloader = env_loader(
    env_name=FLAGS.game,
    dataset_dir=data_dir,
    data_percentage=cfg["data_percentage"],
    batch_size=cfg["batch_size"],
    trajectory_length=1,
  )
  networks = get_networks(cfg, env)

  # ===== Resume. ===== #
  _RESUME = False
  _exp_full_name = None
  if FLAGS.checkpoint is not None:
    _RESUME = True
    ckpt_fn = FLAGS.checkpoint
    assert os.path.isfile(ckpt_fn)
    _exp_full_name = os.path.split(ckpt_fn)[-2].split("/")[-1]
    logging.warn(f"Resuming from {ckpt_fn}...")

  # ===== Essentials. ===== #
  learner = get_learner(
    cfg,
    networks,
    dataloader,
    get_logger_fn(
      cfg["exp_full_name"],
      "learner",
      config=cfg,
    ),
  )
  observers = get_env_loop_observers()
  actor, eval_loop = get_actor_env_eval_loop(
    cfg,
    networks,
    env,
    observers,
    get_logger_fn(cfg["exp_full_name"], "evaluator", is_eval=True, config=cfg),
  )
  evaluate_episodes = 2 if FLAGS.debug else cfg["evaluate_episodes"]

  # ===== Restore. ===== #
  init_step = 0
  if _RESUME:
    save_path = os.path.join("./checkpoint", _exp_full_name)
    assert os.path.isdir(save_path)
    with open(ckpt_fn, "rb") as f:
      train_state = pickle.load(f)
    fn = os.path.split(ckpt_fn)[-1].strip(".pkl")
    init_step = int(fn.split("_")[-1])
    learner.restore(train_state)
    for ob in observers:
      ob.restore(init_step + 1)
  else:
    save_path = os.path.join("./checkpoint", cfg["exp_full_name"])
    os.makedirs(save_path, exist_ok=True)

  if not (FLAGS.debug or FLAGS.profile):
    wb_name = cfg["exp_full_name"]
    wb_cfg = cfg
    if _RESUME:
      wb_name = f"{_exp_full_name}_resume"

    wandb.init(
      project="",
      entity="",
      name=wb_name,
      config=wb_cfg,
      sync_tensorboard=False,
    )
    notif.register(f"[Experiment started] {wb_name}")
    notif.execute()
    notif.register(f"[Experiment finished] {wb_name}")

  # ===== Training Loop. ===== #
  for i in range(init_step + 1, cfg["total_steps"]):
    learner.step(transform=True)
    for ob in observers:
      ob.step()

    if (i + 1) % cfg["save_period"] == 0:
      with open(os.path.join(save_path, f"ckpt_{i}.pkl"), "wb") as f:
        pickle.dump(learner.save(), f)

    if FLAGS.debug or (i + 1) % cfg["eval_period"] == 0:
      actor.update_params(learner.save().params)
      eval_loop.run(evaluate_episodes)

    if FLAGS.debug:
      break

    if i == 110:
      start = time.perf_counter()
      if FLAGS.profile:
        tracer.start()
        # tf.profiler.experimental.start(profile_dir)
        logging.info("Start tracing...")
    if i == 210:
      logging.info(f"100 steps took {time.perf_counter() - start} seconds")
      if FLAGS.profile:
        tracer.stop_and_save()
        # tf.profiler.experimental.stop()
        break

  # ===== Cleanup. ===== #
  learner._logger.close()
  eval_loop._logger.close()
  del env, networks, dataloader, learner, observers, actor, eval_loop

  # ===== Notif. ===== #
  notif.execute()


if __name__ == "__main__":
  app.run(main)
"""
import cv2
import numpy as np

from rosmo.data.rl_unplugged import atari

env = atari.environment(game="Breakout")
ts = env.reset()
cnt = 0
data = []
ts = env.step(np.array([1]))
while not ts.last() and cnt < 5000:
  ts = env.step(np.array([1]))
  data.append(env.render())
  cnt += 1

height, width, _ = data[0].shape
filename = "saved.mp4"
codec_id = "mp4v" # ID for a video codec.
fourcc = cv2.VideoWriter_fourcc(*codec_id)
writer = cv2.VideoWriter(filename, fourcc, 20, (width, height))
for frame in data:
  writer.write(frame)

writer.release()


env = atari.environment(game="Breakout")
ts = env.reset()
cnt = 0
while not ts.last() and cnt < 5000:
  ts = env.step(np.array([np.random.randint(0,3)]))
  cnt += 1

"""
