"""Experiment entry for CQL baseline."""
import os
import pickle
import time
from typing import Dict, Iterator, List, Optional, Tuple

import jax
import numpy as np
import optax
import requests
import tensorflow as tf
import wandb
from absl import app, flags, logging
from acme.specs import make_environment_spec
from acme.types import Transition
from acme.utils.loggers import Logger
from dm_env import Environment

from internal.loggers import generate_experiment_name, logger_fn
from internal.notifier import DiscordNotif
from internal.tracer import PyTracer
from rosmo.agent.cql_discrete.actor import CQLEvalActor
from rosmo.agent.cql_discrete.learning import CQLLearner
from rosmo.agent.cql_discrete.network import CQLNetworks, make_networks
from rosmo.data.rl_unplugged import atari
from rosmo.util.env_loop_observer import (
  EvaluationLoop,
  ExtendedEnvLoopObserver,
  LearningStepObserver,
)

platform = jax.lib.xla_bridge.get_backend().platform
num_devices = jax.lib.xla_bridge.device_count()
logging.warning(f"Compute platform: {platform} with {num_devices} devices.")

notif_url = os.environ.get("NOTIF_WEBHOOK_URL", "")
notif = DiscordNotif(url=notif_url)

# ===== Flags. ===== #
FLAGS = flags.FLAGS
flags.DEFINE_string("exp_id", None, "Experiment id.", required=True)
flags.DEFINE_boolean("debug", True, "Debug run.")
flags.DEFINE_boolean("profile", False, "Profiling run.")
flags.DEFINE_integer("run_number", 1, "Run number of RLU dataset.")
flags.DEFINE_integer(
  "data_percentage", 100, "Percentage of data used for training.", 0, 100
)
flags.DEFINE_integer("seed", int(time.time()), "Random seed.")
flags.DEFINE_float("minq_weight", 1.0, "Weight for Min-Q loss.")
flags.DEFINE_boolean("use_original", False, "Use original CQL networks.")
flags.DEFINE_string("checkpoint", None, "Checkpoint to resume.")
flags.DEFINE_string("game", None, "Game name to run.", required=True)

DATA_DIR = "/mnt_central/datasets/rl_unplugged/tensorflow_datasets"

CONFIG = {
  "num_atoms": 200,
  "channels": 64,
  "blocks_torso": 6,
  "blocks_qf": 2,
  "reduced_channels_head": 128,
  "fc_layers_qf": [128, 128],
  "output_init_scale": 0.0,
  "discount": 0.99,
  "batch_size": 512 * 6,  # 6 to match unroll_length
  "learning_rate": 1e-3,
  "adam_epsilon": 0.0003125,
  "huber_param": 1.0,
  "epsilon_eval": 0.001,
  "target_update_period": 250,
  "grad_updates_per_batch": 1,
  "log_interval": 400,
  "save_period": 10_000,
  "eval_period": 2_000,
  "evaluate_episodes": 2,
  "total_steps": 200_000,
}


# ===== Learner. ===== #
def get_learner(config, networks, data_iterator, logger) -> CQLLearner:
  """Get CQL learner."""

  learner = CQLLearner(
    networks,
    random_key=jax.random.PRNGKey(config["seed"]),
    target_update_period=config["target_update_period"],
    num_atoms=config["num_atoms"],
    minq_weight=config["minq_weight"],
    huber_param=config["huber_param"],
    batch_size=config["batch_size"],
    iterator=data_iterator,
    optimizer=optax.adam(
      learning_rate=config["learning_rate"],
      eps=config["adam_epsilon"],
    ),
    grad_updates_per_batch=config["grad_updates_per_batch"],
    logger=logger,
    log_interval=config["log_interval"],
  )
  return learner


# ===== Eval Actor-Env Loop. ===== #
def get_actor_env_eval_loop(
  config, networks: CQLLearner, environment, observers, logger
) -> Tuple[CQLEvalActor, EvaluationLoop]:
  """Get actor, env and evaluation loop."""
  environment_spec = make_environment_spec(environment)
  actor = CQLEvalActor(
    networks,
    environment_spec.actions.num_values,
    config,
  )
  eval_loop = EvaluationLoop(
    environment=environment,
    actor=actor,
    logger=logger,
    should_update=False,
    observers=observers,
  )
  return actor, eval_loop


def get_env_loop_observers() -> List[ExtendedEnvLoopObserver]:
  """Get environment loop observers."""
  observers = []
  learning_step_ob = LearningStepObserver()
  observers.append(learning_step_ob)
  return observers


def get_environment(config) -> Environment:
  environment = atari.environment(
    game=config["game_name"], stack_size=4
  )  # NOTE use sticky action
  return environment


# ===== Dataset & Buffer ===== #
def _add_next_action_extras(steps: Dict[str, np.ndarray]) -> Transition:
  return Transition(
    observation=steps["observation"][:, 0],
    action=steps["action"][:, 0],
    reward=steps["reward"][:, 0],
    discount=steps["discount"][:, 0],
    next_observation=steps["observation"][:, 1],
    extras={"next_action": steps["action"][:, 1]},
  )


def get_data_loader(config, environment) -> Iterator:
  """Get trajectory data loader."""
  # keep the same number of transitions for learning
  environment_spec = make_environment_spec(environment)
  trajectory_length = 2
  dataset = atari.create_atari_ds_loader(
    game=config["game_name"],
    run_number=config["run_number"],
    data_dir=DATA_DIR,
    num_actions=environment_spec.actions.num_values,
    stack_size=4,
    data_percent=config["data_percentage"],
    trajectory_length=trajectory_length,
    shuffle_num_steps=5000 if FLAGS.debug else 50000,
  )
  dataset = (
    dataset.repeat().batch(
      config["batch_size"]
    ).map(_add_next_action_extras).prefetch(tf.data.AUTOTUNE)
  )
  options = tf.data.Options()
  options.threading.max_intra_op_parallelism = 1
  dataset = dataset.with_options(options)
  iterator = dataset.as_numpy_iterator()
  return iterator


# ===== Network ===== #
def get_networks(config, environment) -> CQLNetworks:
  """Get environment-specific networks."""
  environment_spec = make_environment_spec(environment)
  logging.info(environment_spec)
  networks = make_networks(
    environment_spec,
    config["channels"],
    config["num_atoms"],
    config["output_init_scale"],
    config["blocks_torso"],
    config["blocks_qf"],
    config["reduced_channels_head"],
    config["fc_layers_qf"],
    original=FLAGS.use_original,
  )
  return networks


# ===== Configurations ===== #
def get_config(game_name: str) -> Dict:
  """Get experiment configurations."""
  config = CONFIG.copy()
  config["algo"] = "CQL"
  config["game_name"] = game_name
  config["seed"] = FLAGS.seed
  config["minq_weight"] = FLAGS.minq_weight
  config["use_original"] = FLAGS.use_original
  config["run_number"] = FLAGS.run_number
  config["ckpt_number"] = FLAGS.ckpt_number
  config["data_percentage"] = FLAGS.data_percentage
  config["batch_size"] = 16 if FLAGS.debug else config["batch_size"]
  exp_full_name = generate_experiment_name(f"{FLAGS.exp_id}_{game_name}")
  config["exp_full_name"] = exp_full_name
  config["tpu_vm"] = requests.get(
    "http://metadata/computeMetadata/v1/instance/description",
    headers={
      "Metadata-Flavor": "Google"
    },
  ).text
  logging.info(f"Configs: {config}")
  return config


# ===== Misc. ===== #
def get_logger_fn(
  exp_full_name: str,
  job_name: str,
  is_eval: bool = False,
  config: Optional[Dict] = None,
) -> Logger:
  """Get logger function."""
  save_data = is_eval
  return logger_fn(
    exp_name=exp_full_name,
    label=job_name,
    save_data=save_data and not (FLAGS.debug or FLAGS.profile),
    use_tb=False,
    # use_sota=not (FLAGS.debug or FLAGS.profile),
    use_wb=not (FLAGS.debug or FLAGS.profile),
    use_sota=False,
    config=config,
  )


def main(_):
  """Main program."""
  logging.info(f"Debug mode: {FLAGS.debug}")
  # ===== Profiler. ===== #
  profile_dir = "./profile"
  os.makedirs(profile_dir, exist_ok=True)
  tracer = PyTracer("./", FLAGS.exp_id, with_jax=True)

  # ===== Setup. ===== #
  cfg = get_config(FLAGS.game)

  env = get_environment(cfg)
  networks = get_networks(cfg, env)
  dataloader = get_data_loader(cfg, env)

  # ===== Resume. ===== #
  _RESUME = False
  _exp_full_name = None
  if FLAGS.checkpoint is not None:
    _RESUME = True
    ckpt_fn = FLAGS.checkpoint
    assert os.path.isfile(ckpt_fn)
    _exp_full_name = os.path.split(ckpt_fn)[-2].split("/")[-1]
    logging.warn(f"Resuming from {ckpt_fn}...")

  # ===== Essentials. ===== #
  learner = get_learner(
    cfg,
    networks,
    dataloader,
    get_logger_fn(
      _exp_full_name or cfg["exp_full_name"],
      "learner",
      config=cfg,
    ),
  )
  observers = get_env_loop_observers()
  actor, eval_loop = get_actor_env_eval_loop(
    cfg,
    networks,
    env,
    observers,
    get_logger_fn(
      _exp_full_name or cfg["exp_full_name"],
      "evaluator",
      is_eval=True,
      config=cfg,
    ),
  )
  evaluate_episodes = 2 if FLAGS.debug else cfg["evaluate_episodes"]

  # ===== Restore. ===== #
  init_step = 0
  if _RESUME:
    save_path = os.path.join("./checkpoint", _exp_full_name)
    assert os.path.isdir(save_path)
    with open(ckpt_fn, "rb") as f:
      train_state = pickle.load(f)
    fn = os.path.split(ckpt_fn)[-1].strip(".pkl")
    init_step = int(fn.split("_")[-1])
    learner.restore(train_state)
    for ob in observers:
      ob.restore(init_step + 1)
  else:
    save_path = os.path.join("./checkpoint", cfg["exp_full_name"])
    os.makedirs(save_path, exist_ok=True)

  if not (FLAGS.debug or FLAGS.profile):
    wb_name = cfg["exp_full_name"]
    wb_cfg = cfg
    if _RESUME:
      wb_name = f"{_exp_full_name}_resume"

    wandb.init(
      project="",
      entity="",
      name=wb_name,
      config=wb_cfg,
      sync_tensorboard=False,
    )

    notif.register(f"[Experiment started] {wb_name}")
    notif.execute()
    notif.register(f"[Experiment finished] {wb_name}")

  # ===== Training Loop. ===== #
  for i in range(init_step + 1, cfg["total_steps"]):
    learner.step()
    for ob in observers:
      ob.step()

    if (i + 1) % cfg["save_period"] == 0:
      with open(os.path.join(save_path, f"ckpt_{i}.pkl"), "wb") as f:
        pickle.dump(learner.save(), f)

    if FLAGS.debug or (i + 1) % cfg["eval_period"] == 0:
      actor.update_params(learner.save().params)
      eval_loop.run(evaluate_episodes)

    if FLAGS.debug:
      break

    if i == 110:
      start = time.perf_counter()
      if FLAGS.profile:
        tracer.start()
        # tf.profiler.experimental.start(profile_dir)
        logging.info("Start tracing...")
    if i == 210:
      logging.info(f"100 steps took {time.perf_counter() - start} seconds")
      if FLAGS.profile:
        tracer.stop_and_save()
        # tf.profiler.experimental.stop()
        break

  # ===== Cleanup. ===== #
  learner._logger.close()
  eval_loop._logger.close()
  del env, networks, dataloader, learner, observers, actor, eval_loop

  # ===== Notif. ===== #
  notif.execute()


if __name__ == "__main__":
  app.run(main)
