"""Experiment entry."""
import os
import pickle
import random
import time
from typing import Dict, Iterator, List, Optional, Tuple

import jax
import numpy as np
import requests
import tensorflow as tf
import wandb
from absl import app, flags, logging
from acme import EnvironmentLoop
from acme.specs import make_environment_spec
from acme.utils.loggers import Logger
from dm_env import Environment

from internal.loggers import generate_experiment_name, logger_fn
from internal.notifier import DiscordNotif
from internal.tracer import PyTracer
from rosmo.agent.muzero.actor import MuZeroEvalActor
from rosmo.agent.muzero.learning import SAMPLING_METHOD, MuZeroLearner
from rosmo.agent.muzero.network import Networks, make_networks
from rosmo.data.rl_unplugged import atari
from rosmo.data.rl_unplugged.utils import transform_timesteps
from rosmo.util.env_loop_observer import (
  EvaluationLoop,
  ExtendedEnvLoopObserver,
  LearningStepObserver,
)

platform = jax.lib.xla_bridge.get_backend().platform
num_devices = jax.lib.xla_bridge.device_count()
logging.warn(f"Compute platform: {platform} with {num_devices} devices.")

notif_url = os.environ.get("NOTIF_WEBHOOK_URL", "")
notif = DiscordNotif(url=notif_url)

# ===== Flags. ===== #
FLAGS = flags.FLAGS
flags.DEFINE_string("exp_id", None, "Experiment id.", required=True)
flags.DEFINE_boolean("debug", True, "Debug run.")
flags.DEFINE_boolean("profile", False, "Profiling run.")
flags.DEFINE_integer("run_number", 1, "Run number of RLU dataset.")
flags.DEFINE_integer(
  "data_percentage", 100, "Percentage of data used for training.", 0, 100
)
flags.DEFINE_integer("seed", int(time.time()), "Random seed.")
flags.DEFINE_boolean("use_bc", False, "Run behavior cloning baseline.")
flags.DEFINE_boolean(
  "use_qf", False, "Learn Q function for advantage estimation."
)
flags.DEFINE_enum(
  "improvement_op",
  None,
  ["mcts", "cmpo", "mcts_mpo"],
  "Policy improvement method.",
  required=True,
)
flags.DEFINE_enum(
  "dynamics_scaling",
  None,
  ["1", "0.5", "0.25", "0.125", "0.0625"],
  "Dynamics network channel scaling.",
)
flags.DEFINE_boolean("renormalize", False, "Renormalize state.")
flags.DEFINE_integer("unroll_steps", 5, "Unroll steps of the dynamics.")
flags.DEFINE_integer("stack_size", 4, "Number of frames stacked.")
flags.DEFINE_integer("batch_size", 512, "Batch size for training.")
flags.DEFINE_integer("num_simulations", 16, "Simulation budget.")
flags.DEFINE_enum("sampling", None, SAMPLING_METHOD, "How to sample actions.")
flags.DEFINE_boolean("pessimism", False, "Use pessimism for the model.")
flags.DEFINE_enum(
  "behavior", "", ["", "exp", "bin", "exp_only", "bin_only"],
  "Type of behavior loss."
)
flags.DEFINE_integer(
  "num_pessimism_samples", 1, "Contrastive sampling budget."
)
flags.DEFINE_float("pessimism_weight", 0, "Reward given for ood actions.")
flags.DEFINE_integer("search_depth", None, "MCTS search depth.")
flags.DEFINE_boolean(
  "safe_q_values", False, "Fill unvisited node's q_value as node value."
)
flags.DEFINE_boolean("value_no_search", False, "Value target not from search.")
flags.DEFINE_float(
  "dynamics_noise", 0.0, "Inject noise into dynamics if not zero."
)
flags.DEFINE_float("behavior_coef", 0.2, "Behavior loss coefficient.")
flags.DEFINE_string("checkpoint", None, "Checkpoint to resume.")
flags.DEFINE_string("game", None, "Game name to run.", required=True)
flags.DEFINE_integer("total_steps", 200_000, "Total learning steps.")
DATA_DIR = "/mnt_central/datasets/rl_unplugged/tensorflow_datasets"

CONFIG = {
  "td_steps": 5,
  "num_bins": 601,
  "channels": 64,
  "blocks_representation": 6,
  "blocks_prediction": 2,
  "blocks_transition": 2,
  "reduced_channels_head": 128,
  "fc_layers_reward": [128, 128],
  "fc_layers_value": [128, 128],
  "fc_layers_policy": [128, 128],
  "output_init_scale": 0.0,
  "discount_factor": 0.997**4,
  "clipping_threshold": 1.0,
  "evaluate_episodes": 2,
  "log_interval": 400,
  "learning_rate": 7e-4,
  "warmup_steps": 1_000,
  "learning_rate_decay": 0.1,
  "weight_decay": 1e-4,
  "max_grad_norm": 5.0,
  "target_update_interval": 200,
  "value_coef": 0.25,
  "policy_coef": 1.0,
  "save_period": 10_000,
  "eval_period": 1_000,
}


# ===== Learner. ===== #
def get_learner(config, networks, data_iterator, logger) -> MuZeroLearner:
  """Get MuZero learner."""
  learner = MuZeroLearner(
    networks,
    demonstrations=data_iterator,
    config=config,
    logger=logger,
  )
  return learner


# ===== Eval Actor-Env Loop. ===== #
def get_actor_env_eval_loop(config, networks, environment, observers,
                            logger) -> Tuple[MuZeroEvalActor, EnvironmentLoop]:
  """Get actor, env and evaluation loop."""
  actor = MuZeroEvalActor(
    networks,
    config,
  )
  eval_loop = EvaluationLoop(
    environment=environment,
    actor=actor,
    logger=logger,
    should_update=False,
    observers=observers,
  )
  return actor, eval_loop


def get_env_loop_observers() -> List[ExtendedEnvLoopObserver]:
  """Get environment loop observers."""
  observers = []
  learning_step_ob = LearningStepObserver()
  observers.append(learning_step_ob)
  return observers


def get_environment(config) -> Environment:
  environment = atari.environment(
    game=config["game_name"],
    stack_size=config["stack_size"],
  )  # NOTE use sticky action
  return environment


# ===== Dataset & Buffer ===== #
def get_data_loader(config, environment) -> Iterator:
  """Get trajectory data loader."""
  environment_spec = make_environment_spec(environment)
  trajectory_length = config["unroll_steps"] + config["td_steps"] + 1
  if FLAGS.use_bc or FLAGS.use_qf:
    # keep the same number of transitions for learning
    trajectory_length = config["unroll_steps"] + 1
  dataset = atari.create_atari_ds_loader(
    game=config["game_name"],
    run_number=config["run_number"],
    data_dir=DATA_DIR,
    num_actions=environment_spec.actions.num_values,
    stack_size=config["stack_size"],
    data_percent=config["data_percentage"],
    trajectory_length=trajectory_length,
    shuffle_num_steps=5000 if FLAGS.debug else 50000,
  )
  dataset = (
    dataset.repeat().batch(
      config["batch_size"]
    ).map(transform_timesteps).prefetch(tf.data.AUTOTUNE)
  )
  options = tf.data.Options()
  options.threading.max_intra_op_parallelism = 1
  dataset = dataset.with_options(options)
  iterator = dataset.as_numpy_iterator()
  return iterator


# ===== Network ===== #
def get_networks(config, environment) -> Networks:
  """Get environment-specific networks."""
  environment_spec = make_environment_spec(environment)
  logging.info(environment_spec)
  networks = make_networks(
    env_spec=environment_spec,
    channels=config["channels"],
    num_bins=config["num_bins"],
    output_init_scale=config["output_init_scale"],
    channel_scaling=config["dynamics_scaling"],
    blocks_representation=config["blocks_representation"],
    blocks_prediction=config["blocks_prediction"],
    blocks_transition=config["blocks_transition"],
    reduced_channels_head=config["reduced_channels_head"],
    fc_layers_reward=config["fc_layers_reward"],
    fc_layers_value=config["fc_layers_value"],
    fc_layers_policy=config["fc_layers_policy"],
    renormalize=config["renormalize"],
  )
  return networks


# ===== Configurations ===== #
def get_config(game_name: str) -> Dict:
  """Get experiment configurations."""
  config = CONFIG.copy()
  config["algo"] = "OS"
  if FLAGS.use_bc:
    config["algo"] = "BC"
  elif FLAGS.use_qf:
    config["algo"] = "Q-MPO"
  elif FLAGS.improvement_op == "mcts":
    config["algo"] = "MZU"
  elif FLAGS.improvement_op == "mcts_mpo":
    config["algo"] = "MCTS-Q"

  config["total_steps"] = FLAGS.total_steps
  config["seed"] = FLAGS.seed
  config["dynamics_scaling"] = (
    float(FLAGS.dynamics_scaling) if FLAGS.dynamics_scaling else -1
  )
  config["renormalize"] = FLAGS.renormalize
  config["behavior_coef"] = FLAGS.behavior_coef
  config["sampling"] = FLAGS.sampling or "exact"
  config["pessimism"] = FLAGS.pessimism
  config["behavior"] = FLAGS.behavior
  config["num_pessimism_samples"] = FLAGS.num_pessimism_samples
  config["pessimism_weight"] = FLAGS.pessimism_weight
  config["game_name"] = game_name
  config["run_number"] = FLAGS.run_number
  config["ckpt_number"] = FLAGS.ckpt_number
  config["unroll_steps"] = FLAGS.unroll_steps
  config["stack_size"] = FLAGS.stack_size
  config["num_simulations"] = FLAGS.num_simulations
  config["search_depth"] = FLAGS.search_depth or FLAGS.num_simulations
  config["safe_q_values"] = FLAGS.safe_q_values
  config["dynamics_noise"] = FLAGS.dynamics_noise
  config["value_no_search"] = FLAGS.value_no_search
  config["use_bc"] = FLAGS.use_bc
  config["use_qf"] = FLAGS.use_qf
  config["improvement_op"] = FLAGS.improvement_op
  config["data_percentage"] = FLAGS.data_percentage
  config["batch_size"] = 16 if FLAGS.debug else FLAGS.batch_size
  exp_full_name = generate_experiment_name(f"{FLAGS.exp_id}_{game_name}")
  config["exp_full_name"] = exp_full_name
  config["tpu_vm"] = requests.get(
    "http://metadata/computeMetadata/v1/instance/description",
    headers={
      "Metadata-Flavor": "Google"
    },
  ).text
  logging.info(f"Configs: {config}")
  return config


# ===== Misc. ===== #
def get_logger_fn(
  exp_full_name: str,
  job_name: str,
  is_eval: bool = False,
  config: Optional[Dict] = None,
) -> Logger:
  """Get logger function."""
  save_data = is_eval
  return logger_fn(
    exp_name=exp_full_name,
    label=job_name,
    save_data=save_data and not (FLAGS.debug or FLAGS.profile),
    use_tb=False,
    # use_sota=not (FLAGS.debug or FLAGS.profile),
    use_wb=not (FLAGS.debug or FLAGS.profile),
    use_sota=False,
    config=config,
  )


def main(_):
  """Main program."""
  logging.info(f"Debug mode: {FLAGS.debug}")
  random.seed(FLAGS.seed)
  np.random.seed(FLAGS.seed)

  # ===== Profiler. ===== #
  profile_dir = "./profile"
  os.makedirs(profile_dir, exist_ok=True)
  tracer = PyTracer("./", FLAGS.exp_id, with_jax=True)

  # ===== Setup. ===== #
  cfg = get_config(FLAGS.game)

  env = get_environment(cfg)
  networks = get_networks(cfg, env)
  dataloader = get_data_loader(cfg, env)

  # ===== Resume. ===== #
  _RESUME = False
  _exp_full_name = None
  if FLAGS.checkpoint is not None:
    _RESUME = True
    ckpt_fn = FLAGS.checkpoint
    assert os.path.isfile(ckpt_fn)
    _exp_full_name = os.path.split(ckpt_fn)[-2].split("/")[-1]
    logging.warn(f"Resuming from {ckpt_fn}...")

  # ===== Essentials. ===== #
  learner = get_learner(
    cfg,
    networks,
    dataloader,
    get_logger_fn(
      _exp_full_name or cfg["exp_full_name"],
      "learner",
      config=cfg,
    ),
  )
  observers = get_env_loop_observers()
  actor, eval_loop = get_actor_env_eval_loop(
    cfg,
    networks,
    env,
    observers,
    get_logger_fn(
      _exp_full_name or cfg["exp_full_name"],
      "evaluator",
      is_eval=True,
      config=cfg,
    ),
  )
  evaluate_episodes = 2 if FLAGS.debug else cfg["evaluate_episodes"]

  # ===== Restore. ===== #
  init_step = 0
  if _RESUME:
    save_path = os.path.join("./checkpoint", _exp_full_name)
    assert os.path.isdir(save_path)
    with open(ckpt_fn, "rb") as f:
      train_state = pickle.load(f)
    fn = os.path.split(ckpt_fn)[-1].strip(".pkl")
    init_step = int(fn.split("_")[-1])
    learner.restore(train_state, init_step)
    for ob in observers:
      ob.restore(init_step + 1)
  else:
    save_path = os.path.join("./checkpoint", cfg["exp_full_name"])
    os.makedirs(save_path, exist_ok=True)

  if not (FLAGS.debug or FLAGS.profile):
    wb_name = cfg["exp_full_name"]
    wb_cfg = cfg
    if _RESUME:
      wb_name = f"{_exp_full_name}_resume"

    wandb.init(
      project="", 
      entity="",
      name=wb_name,
      config=wb_cfg,
      sync_tensorboard=False,
    )
    vm_name = cfg.get("tpu_vm", "")
    notif.register(f"{vm_name} [Experiment started] {wb_name}")
    notif.execute()
    notif.register(f"{vm_name} [Experiment finished] {wb_name}")

  # ===== Training Loop. ===== #
  for i in range(init_step + 1, cfg["total_steps"]):
    learner.step()
    for ob in observers:
      ob.step()

    if (i + 1) % cfg["save_period"] == 0:
      with open(os.path.join(save_path, f"ckpt_{i}.pkl"), "wb") as f:
        pickle.dump(learner.save(), f)

    if FLAGS.debug or (i + 1) % cfg["eval_period"] == 0:
      actor.update_params(learner.save().params)
      eval_loop.run(evaluate_episodes)

    if FLAGS.debug:
      break

    if i == 110:
      start = time.perf_counter()
      if FLAGS.profile:
        tracer.start()
        # tf.profiler.experimental.start(profile_dir)
        logging.info("Start tracing...")
    if i == 210:
      logging.info(f"100 steps took {time.perf_counter() - start} seconds")
      if FLAGS.profile:
        tracer.stop_and_save()
        # tf.profiler.experimental.stop()
        break

  # ===== Cleanup. ===== #
  learner._logger.close()
  eval_loop._logger.close()
  del env, networks, dataloader, learner, observers, actor, eval_loop

  # ===== Notif. ===== #
  notif.execute()


if __name__ == "__main__":
  app.run(main)
