# coding=utf-8
# Copyright 2023-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import importlib
import math
import re
import warnings
from dataclasses import asdict, dataclass, field
from enum import Enum
from typing import List, Optional, Union

import torch
import torch.nn as nn
import torch.nn.functional as F
from transformers.pytorch_utils import Conv1D

from ..utils import PeftConfig, PeftType, transpose


@dataclass
class LoraConfig(PeftConfig):  # 继承子Peftconfig
    """
    This is the configuration class to store the configuration of a [`~peft.Lora`].

    Args:
        r (`int`): Lora attention dimension
        target_modules (`Union[List[str],str]`): The names of the modules to apply Lora to.
        lora_alpha (`float`): The alpha parameter for Lora scaling.
        lora_dropout (`float`): The dropout probability for Lora layers.
        merge_weights (`bool`):
            Whether to merge the weights of the Lora layers with the base transformer model in `eval` mode.
        fan_in_fan_out (`bool`): Set this to True if the layer to replace stores weight like (fan_in, fan_out)
        enable_lora ( `List[bool]`): Used with `lora.MergedLinear`.
        bias (`str`): Bias type for Lora. Can be 'none', 'all' or 'lora_only'
        modules_to_save (`List[str]`):List of modules apart from LoRA layers to be set as trainable
            and saved in the final checkpoint.
    """

    noisetrain: bool = field(default=False)  # 更改
    r: int = field(default=8, metadata={"help": "Lora attention dimension"})
    target_modules: Optional[Union[List[str], str]] = field(
        default=None,
        metadata={
            "help": "List of module names or regex expression of the module names to replace with Lora."
                    "For example, ['q', 'v'] or '.*decoder.*(SelfAttention|EncDecAttention).*(q|v)$' "
        },
    )
    lora_alpha: int = field(default=None, metadata={"help": "Lora alpha"})
    lora_nums: int = field(default=None, metadata={"help": "Numbers of Lora"})
    lora_dropout: float = field(default=None, metadata={"help": "Lora dropout"})
    merge_weights: bool = field(
        default=False, metadata={"help": "Merge weights of the original model and the Lora model"}
    )
    fan_in_fan_out: bool = field(
        default=False,
        metadata={"help": "Set this to True if the layer to replace stores weight like (fan_in, fan_out)"},
    )
    enable_lora: Optional[List[bool]] = field(default=None, metadata={"help": "Used with `lora.MergedLinear`."})
    bias: str = field(default="none", metadata={"help": "Bias type for Lora. Can be 'none', 'all' or 'lora_only'"})
    modules_to_save: Optional[List[str]] = field(
        default=None,
        metadata={
            "help": "List of modules apart from LoRA layers to be set as trainable and saved in the final checkpoint. "
                    "For example, in Sequence Classification or Token Classification tasks, "
                    "the final layer `classifier/score` are randomly initialized and as such need to be trainable and saved."
        },
    )

    def __post_init__(self):
        self.peft_type = PeftType.LORA


class LoraModel(torch.nn.Module):
    """
    Creates Low Rank Adapter (Lora) model from a pretrained transformers model.

    Args:
        model ([`transformers.PreTrainedModel`]): The model to be adapted.
        config ([`LoraConfig`]): The configuration of the Lora model.

    Returns:
        `torch.nn.Module`: The Lora model.

    Example::

        >>> from transformers import AutoModelForSeq2SeqLM, LoraConfig >>> from peft import LoraModel, LoraConfig >>>
        config = LoraConfig(
            peft_type="LORA", task_type="SEQ_2_SEQ_LM", r=8, lora_alpha=32, target_modules=["q", "v"],
            lora_dropout=0.01, )
        >>> model = AutoModelForSeq2SeqLM.from_pretrained("t5-base") >>> lora_model = LoraModel(config, model)

    **Attributes**:
        - **model** ([`transformers.PreTrainedModel`]) -- The model to be adapted.
        - **peft_config** ([`LoraConfig`]): The configuration of the Lora model.
    """

    def __init__(self, config, model):  # LoraConfig, CasualLM
        super().__init__()
        self.peft_config = config
        self.model = model
        self._find_and_replace()
        mark_only_lora_as_trainable(self.model, self.peft_config.bias) \
 \
                self.forward = self.model.forward

    def _find_and_replace(self):
        loaded_in_4bit = getattr(self.model, "is_loaded_in_4bit", False)
        loaded_in_8bit = getattr(self.model, "is_loaded_in_8bit", False)
        if (loaded_in_4bit or loaded_in_8bit):
            raise ImportError(
                "To use Lora with 8-bit or 4-bit quantization, please install the `bitsandbytes` package. "
                "You can install it with `pip install bitsandbytes`."
            )
        is_target_modules_in_base_model = False
        is_hf_device_map_available = hasattr(self.model, "hf_device_map")

        kwargs = {
            "r": self.peft_config.r,
            "lora_alpha": self.peft_config.lora_alpha,
            "lora_dropout": self.peft_config.lora_dropout,
            "lora_nums": self.peft_config.lora_nums,
            "fan_in_fan_out": self.peft_config.fan_in_fan_out,
            "merge_weights": (self.peft_config.merge_weights or self.peft_config.inference_mode)
                             and not is_hf_device_map_available,
        }

        key_list = [key for key, _ in self.model.named_modules()]
        for key in key_list:
            if isinstance(self.peft_config.target_modules, str):
                target_module_found = re.fullmatch(self.peft_config.target_modules, key)
            else:
                target_module_found = any(key.endswith(target_key) for target_key in self.peft_config.target_modules)
            if target_module_found:  # here
                if not is_target_modules_in_base_model:
                    is_target_modules_in_base_model = True
                parent, target, target_name = self._get_submodules(key)
                bias = target.bias is not None

                if isinstance(target, torch.nn.Linear) and self.peft_config.enable_lora is None:  # 替换线性层
                    new_module = Linear(target.in_features, target.out_features, bias=bias, **kwargs)
                    new_module.set_noisetrain(self.peft_config.noisetrain)  # 更改

                self._replace_module(parent, target_name, new_module, target)
        if not is_target_modules_in_base_model:
            raise ValueError(
                f"Target modules {self.peft_config.target_modules} not found in the base model. "
                f"Please check the target modules and try again."
            )

    def _get_submodules(self, key):
        parent = self.model.get_submodule(".".join(key.split(".")[:-1]))
        target_name = key.split(".")[-1]
        target = self.model.get_submodule(key)
        return parent, target, target_name

    def _replace_module(self, parent_module, child_name, new_module, old_module):
        setattr(parent_module, child_name, new_module)
        new_module.weight = old_module.weight
        if old_module.bias is not None:
            new_module.bias = old_module.bias
        if getattr(old_module, "state", None) is not None:
            new_module.state = old_module.state
            new_module.to(old_module.weight.device)

        # dispatch to correct device
        for name, module in new_module.named_modules():
            if "lora_" in name:
                module.to(old_module.weight.device)

    def __getattr__(self, name: str):
        """Forward missing attributes to the wrapped module."""
        try:
            return super().__getattr__(name)  # defer to nn.Module's logic
        except AttributeError:
            return getattr(self.model, name)

    @property
    def modules_to_save(self):
        return None

    def get_peft_config_as_dict(self, inference: bool = False):
        config = {k: v.value if isinstance(v, Enum) else v for k, v in asdict(self.peft_config).items()}
        if inference:
            config["inference_mode"] = True
        return config

    def _set_adapter_layers(self, enabled=True):
        for module in self.model.modules():
            if isinstance(module, LoraLayer):
                module.disable_adapters = False if enabled else True

    def enable_adapter_layers(self):
        self._set_adapter_layers(enabled=True)

    def disable_adapter_layers(self):
        self._set_adapter_layers(enabled=False)


# Below code is based on https://github.com/microsoft/LoRA/blob/main/loralib/layers.py
# and modified to work with PyTorch FSDP


#  ------------------------------------------------------------------------------------------
#  Copyright (c) Microsoft Corporation. All rights reserved.
#  Licensed under the MIT License (MIT). See LICENSE in the repo root for license information.
#  ------------------------------------------------------------------------------------------


# had to adapt it for `lora_only` to work
def mark_only_lora_as_trainable(model: nn.Module, bias: str = "none") -> None:
    for n, p in model.named_parameters():
        if "lora_" not in n:
            p.requires_grad = False
    if bias == "none":
        return
    elif bias == "all":
        for n, p in model.named_parameters():
            if "bias" in n:
                p.requires_grad = True
    elif bias == "lora_only":
        for m in model.modules():
            if isinstance(m, LoraLayer) and hasattr(m, "bias") and m.bias is not None:
                m.bias.requires_grad = True
    else:
        raise NotImplementedError


class LoraLayer:
    def __init__(
            self,
            r: int,
            lora_alpha: int,
            lora_dropout: float,
            merge_weights: bool,
    ):
        self.r = r
        self.lora_alpha = lora_alpha
        # Optional dropout
        if lora_dropout > 0.0:
            self.lora_dropout = nn.Dropout(p=lora_dropout)
        else:
            self.lora_dropout = lambda x: x
        # Mark the weight as unmerged
        self.merged = False
        self.merge_weights = merge_weights
        self.disable_adapters = False


class Linear(nn.Linear, LoraLayer):
    # Lora implemented in a dense layer
    def __init__(
            self,
            in_features: int,
            out_features: int,
            r: int = 0,
            lora_alpha: int = 1,
            lora_nums: int = 2,
            lora_dropout: float = 0.0,
            fan_in_fan_out: bool = False,
            # Set this to True if the layer to replace stores weight like (fan_in, fan_out)
            merge_weights: bool = True,

            **kwargs,
    ):
        nn.Linear.__init__(self, in_features, out_features, **kwargs)
        LoraLayer.__init__(self, r=r, lora_alpha=lora_alpha, lora_dropout=lora_dropout, merge_weights=merge_weights)

        self.lora_num = lora_nums

        self.fan_in_fan_out = fan_in_fan_out
        self.noisetrain = False  # 更改

        # Actual trainable parameters
        if r > 0:
            self.lora_route = nn.Linear(in_features, self.lora_num, bias=False)  # 路由网络
            setattr(self, f"lora_A", nn.Linear(in_features, r, bias=False))
            for i in range(self.lora_num):
                setattr(self, f"lora_B{i}", nn.Linear(r, out_features, bias=False))
                if i == 0:  # 更改
                    getattr(self, f"lora_B{i}").weight.requires_grad = False

            self.scaling = self.lora_alpha / self.r
            # Freezing the pre-trained weight matrix
            self.weight.requires_grad = False
        self.reset_parameters()
        if fan_in_fan_out:
            self.weight.data = self.weight.data.T

    def set_noisetrain(self, flag: bool):  # 更改
        self.noisetrain = flag
        if self.noisetrain:
            for i in range(self.lora_num):
                if i == 0:
                    getattr(self, f"lora_B{i}").weight.requires_grad = True
                else:
                    getattr(self, f"lora_B{i}").weight.requires_grad = False

    def reset_parameters(self):
        nn.Linear.reset_parameters(self)

        if hasattr(self, "lora_A"):
            nn.init.kaiming_uniform_(getattr(self, f"lora_A").weight, a=math.sqrt(5))
            for i in range(self.lora_num):
                nn.init.zeros_(getattr(self, f"lora_B{i}").weight)

            nn.init.kaiming_uniform_(self.lora_route.weight, a=math.sqrt(5))

    def train(self, mode: bool = True):
        nn.Linear.train(self, mode)
        self.lora_route.train(mode)
        getattr(self, f"lora_A").train(mode)
        for i in range(self.lora_num):
            getattr(self, f"lora_B{i}").train(mode)

    def eval(self):
        nn.Linear.eval(self)
        self.lora_route.eval()
        getattr(self, f"lora_A").eval()
        for i in range(self.lora_num):
            getattr(self, f"lora_B{i}").eval()

    def cv_squared(self, x):
        """The squared coefficient of variation of a sample.
        Useful as a loss to encourage a positive distribution to be more uniform.
        Epsilons added for numerical stability.
        Returns 0 for an empty Tensor.
        Args:
        x: a `Tensor`.
        Returns:
        a `Scalar`.
        """
        eps = 1e-10
        if x.shape[0] == 1:
            return torch.tensor([0], device=x.device, dtype=x.dtype)[0]
        return x.float().var() / (x.float().mean() ** 2 + eps)

    def cosine_similarity_joint(self, x, y, dim=0):  # 0 - 1
        # 将输入展平为向量
        x_flat = x.view(-1)
        y_flat = y.view(-1)

        # 计算余弦相似度
        cosine_sim = F.cosine_similarity(x_flat.unsqueeze(0), y_flat.unsqueeze(0), dim=dim)

        # 归一化到[0,1]范围
        normalized_cosine_sim = (cosine_sim + 1) / 2

        # 取平均值作为整体相似度
        return normalized_cosine_sim.mean().item()

    def compute_weight_similarity(self, x, y):  # -1 - 1
        # 展平成向量
        x_flat = x.view(-1)
        y_flat = y.view(-1)

        # 计算余弦相似度
        similarity = F.cosine_similarity(x_flat.unsqueeze(0), y_flat.unsqueeze(0))
        return similarity.item()

    def cosine_similarity_1(self, x, y):  # 0 - 1
        cosine_sim = F.cosine_similarity(x, y, dim=-1)
        normalized_cosine_sim = (cosine_sim + 1) / 2
        return normalized_cosine_sim

    # def forward(self, x: torch.Tensor, task_types=None):
    #     if self.disable_adapters:
    #         result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)
    #         raise ImportError(":(")
    #     elif self.r > 0 and not self.merged:
    #         result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)
    #
    #         if self.r > 0:
    #             # print("\nB matrices weights:")
    #             for i in range(self.lora_num):
    #                 lora_B = getattr(self, f"lora_B{i}")
    #                 # print(f"\nB{i} weight:")
    #                 # print(lora_B.weight)  # 打印完整权重
    #                 # print(f"B{i} weight shape:", lora_B.weight.shape)
    #                 # print(f"B{i} weight mean:", lora_B.weight.mean().item())
    #                 # print(f"B{i} weight std:", lora_B.weight.std().item())
    #             # 计算路由权重
    #             route_weight = nn.functional.softmax(self.lora_route(x), dim=-1, dtype=torch.float32).to(
    #                 result.dtype)
    #
    #             # print("\n原始路由权重 (first token):")
    #             # print(route_weight[0, 0])
    #             # 计算相似度矩阵
    #             lora_B1 = getattr(self, "lora_B0")
    #             similarities = torch.zeros(self.lora_num, device=x.device)
    #
    #             # print("\nSimilarities between B0 and other Bs:")
    #             # 计算相似度
    #             for i in range(1, self.lora_num):
    #                 lora_B_i = getattr(self, f"lora_B{i}")
    #                 similarity = self.cosine_similarity_joint(
    #                     lora_B1.weight,
    #                     lora_B_i.weight
    #                 )
    #                 similarities[i] = similarity
    #                 route_weight[:, :, i] = route_weight[:, :, i] + similarities[i]
    #                 # print(f"B0-B{i}: {similarity:.4f}")
    #
    #             # print("\n更新后的权重 (first token):")
    #             # print(route_weight[0, 0])
    #             # 应用相似度调整
    #
    #             route_weight = route_weight / route_weight.sum(dim=-1, keepdim=True)
    #             # print("\n最终归一化后的权重 (first token):")
    #             # print(route_weight[0, 0])
    #             lora_output = 0
    #             for i in range(self.lora_num):
    #                 current_output = getattr(self, f"lora_B{i}")(getattr(self, f"lora_A")(self.lora_dropout(x)))
    #                 lora_output = lora_output + route_weight[:, :, i:i + 1] * current_output
    #
    #             result = result + lora_output * self.scaling

    def forward(self, x: torch.Tensor, task_types=None):
        """
        实现动态 β 缩放：
          1) 计算原始路由权重 ω 和专家输出 y_i；
          2) 以 B0 作为 BD，计算 θ_iD = sim(y_i, y_D)；
          3) 对 i>=1 做补偿 ω_i' = (1 + θ_iD) * ω_i（不再做简单归一化）；
          4) β = sum_i ω_i / sum_i ω_i'（逐 token 动态计算，非可学习）；
          5) result = W0x + scaling * β * Σ_i ω_i' * y_i
        """
        eps = 1e-12

        if self.disable_adapters:
            return F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)

        if self.r <= 0 or self.merged:
            return F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)

        # 原始模型输出 W0 x
        result = F.linear(x, transpose(self.weight, self.fan_in_fan_out), bias=self.bias)

        # 计算各专家输出：y_i = B_i(A(x))
        a_out = getattr(self, f"lora_A")(self.lora_dropout(x))
        expert_outputs = []
        for i in range(self.lora_num):
            expert_outputs.append(getattr(self, f"lora_B{i}")(a_out))

        # 原始路由权重 ω = softmax(W_gate^T x)
        route_weight = F.softmax(self.lora_route(x), dim=-1, dtype=torch.float32).to(result.dtype)  # [B, T, N]

        # 以 B0 作为 BD，计算 θ_iD（按输出余弦相似度，先展平后在 token 维平均）
        base_expert_output = expert_outputs[0]  # y_D
        similarities = torch.zeros(self.lora_num, device=x.device, dtype=result.dtype)  # 存储 θ_iD（θ_0D 无用）
        flat_base = base_expert_output.view(-1, base_expert_output.size(-1))  # [B*T, H]
        base_norm = flat_base.norm(dim=-1, keepdim=True).clamp_min(eps)

        for i in range(1, self.lora_num):
            current_expert_output = expert_outputs[i]
            flat_current = current_expert_output.view(-1, current_expert_output.size(-1))  # [B*T, H]
            curr_norm = flat_current.norm(dim=-1, keepdim=True).clamp_min(eps)
            # 余弦相似度（-1~1），与论文一致，这里不强行映射到 [0,1]
            cos_sim = (flat_base / base_norm * flat_current / curr_norm).sum(dim=-1).mean()
            similarities[i] = cos_sim

        # 动态补偿权重：ω'（不对 BD 做补偿）
        compensated_weight = route_weight.clone()  # [B, T, N]
        for i in range(1, self.lora_num):
            compensated_weight[:, :, i] = route_weight[:, :, i] * (1.0 + similarities[i])

        # 动态 β：保持补偿前后的“权重总量”一致（逐 token 计算）
        s0 = route_weight.sum(dim=-1, keepdim=True)  # [B, T, 1], Σ ω_i
        s1 = compensated_weight.sum(dim=-1, keepdim=True).clamp_min(eps)  # [B, T, 1], Σ ω_i'
        beta = (s0 / s1).to(result.dtype)  # 非可学习，随前向动态

        # 按补偿后的权重聚合专家输出，并用 β 稳定缩放
        lora_output = 0.0
        for i in range(self.lora_num):
            lora_output = lora_output + compensated_weight[:, :, i:i + 1] * expert_outputs[i]

        result = result + (beta * lora_output) * self.scaling
        return result