using PyFormattedStrings
using MacroTools
using JLD2

"""
    @run_or_load(path, block, skip_load=false, params=(;), debug=false)

Load from path if exists, otherwise run block and save to path.

A macro that either runs a given block of code and saves the results to a specified file or loads the results from that file if it exists. The last expression in the block should return the values to be saved. The results are saved as a tuple, with the variable names inferred from the last expression in the block.

### Parameters:
- `path::String`: The file path where the results of the block will be saved or loaded from.
- `block::Expr`: A block of code that is executed if the file does not exist or if `skip_load` is `true`. The results of this block will be saved to the specified file.
- `skip_load::Bool`: A flag indicating whether to skip loading from the file. Defaults to `false`.
- `params::NamedTuple`: Optional named parameters that can be saved alongside the results. Defaults to an empty named tuple.
- `debug::Bool`: A flag to enable debug prints for detailed information during execution. Defaults to `false`.

### Behavior:
- If the file at `path` exists and `skip_load` is `false`, the macro will attempt to load the results from the file.
- It validates that the saved code block in the file matches the current block of code. If there is a discrepancy, it will print a warning and the saved code block.
- If the file does not exist or `skip_load` is `true`, it will execute the provided `block`, save the results, and store the code block into the file.
- The results are saved as a tuple, with the variable names inferred from the last expression in the block.

### Return Value:
Returns a `Tuple` containing the values of the variables resulting from the executed block.

### Example:
```julia
t = 0.5
m = 2
@run_or_load f"path_t{t:3.2f}_m{m}.jld2" begin
    x = 2 * t
    y = 3 * m
    z = 3
    x, y
end
```
Here, `m` and `t` are used from the outside scope. In this example, if the specified path does not exist, it will compute x and y, saving them to the file. If the file exists, it will load the saved values.

!!! warning 
    The variable `z` will not be saved or loaded or returned but is modified in the outside scope only if the result is not loaded!


Notes:
- Make sure the code block is consistent between runs to avoid warnings or errors related to variable names.
- The results are saved in a format compatible with the JLD2 library, allowing for easy storage and retrieval of Julia objects. 
"""
macro run_or_load(path, block, skip_load=false, params=(;), debug=false)
    debug && println("run_or_load with bebug prints")
    debug && println("params=",params)
    debug && println("path=",path)
    debug && println("skip_load=",skip_load)
    debug && println("block=",block)
    debug && println("typeof(block)=",typeof(block))

    block_str = remove_lines_numbers(string(block))
    debug && println("block_str=",block_str)

    last_expr = block.args[end]
    if isa(last_expr, Expr) && last_expr.head == :(=)
        var_names = last_expr.args[1]
    elseif isa(last_expr, Expr) && last_expr.head == :return
        var_names = last_expr.args[1]
    elseif isa(last_expr, Expr) && last_expr.head == :tuple
        var_names = expression_tuple_to_varnames(last_expr)
    elseif isa(last_expr, Symbol)
        var_names = last_expr
    else
        var_names = :unnamed_return
    end
    if isa(var_names, Symbol)
        var_names = [var_names] 
    elseif isa(var_names, Tuple)
        # do nothing
    else 
        var_names = var_names.args
    end
    debug && println("Last expression, i.e. return from block: ",last_expr, ", type: ",typeof(last_expr)) 
    debug && println("Return variable names: ", var_names) 

    return quote  
        local path = $(esc(path)) #(;$(esc(params))...)
        local skip_load=$(esc(skip_load)) 
        local var_names = $(esc(var_names))
        local debug = $(esc(debug)) 
        local block_str = $(esc(block_str))
	
		local ret
        if !skip_load && isfile(path)
            debug && println(f"Loading from '{path}'.")
            ret = []
            jldopen(path, "r") do file
                if !haskey(file, "_code_block")
                    error("The file does not contain a field _code_block. Make sure the file was saved with this macro before.")
                end
                # validate the saved result
                if file["_code_block"] != block_str
                    local cb = file["_code_block"]
                    println("\033[33mWarning: The code block has changed compared to what was saved. The loaded result may not be identical to the output of the code block!\033[0m") 
                    println(f"\033[33mSaved code block:\n{cb}\033[0m")
                    println("\033[33mIf the return names have changed, this will cause an error below.\033[0m")
                end

                for var_name in var_names  
                    push!(ret, file[string(var_name)])
                end 
            end  		
        else
            debug && println(f"Nothing to load, run and save to '{path}'.")
            ret = begin
                # execute the block 
                local ret2 = $(esc(block)) 

                jldopen(path, "w") do file
                    for (var_name, out) in zip(var_names,ret2)
                        debug && println(f"Saving {var_name}., value={out}")
                        file[string(var_name)] = out
                    end 
                    # write optional parameters to file 
                    for (key, value) in pairs($(esc(params)))
                        file[string(key)] = value
                    end
                    # write the code block to file
                    file["_code_block"] = remove_lines_numbers($(esc(block_str)))
                end
                
                ret2
            end 
        end 
        Tuple(ret)
    end
end

function remove_lines_numbers(code::String)
    # Regular expression to match lines that consist only of a block comment
    return replace(code, r"^\s*#=.*=#\s*$"m => "")
end

function expression_tuple_to_varnames(expr::Expr) 
    names = Symbol[]  # This will store the variable or placeholder names
    count = 1  # To track unnamed returns
    for arg in expr.args
        if arg isa Symbol
            push!(names, arg)  # Directly push the symbol name
        else
            # If it's not a simple variable, create a placeholder name
            push!(names, Symbol("unnamed_return$count"))
            count += 1
        end
    end
    return Tuple(names)  
end
 


