"""
    nice_points(color::String)

Create a dictionary with point styling parameters using the given color.
The stroke color is the input color, while the fill color is the same color with reduced opacity.

# Arguments
- `color::String`: Hex color string.

# Returns
A dictionary with keys `:strokecolor` and `:color` for plotting.
"""
function nice_points(color::String) 
    return Dict(
        :strokecolor => color, 
        :color => get_alpha_hex(color,0.65))
end

"""
    hex_to_rgb(value; full=false)

Convert a hex color string to an RGB tuple.

# Arguments
- `value::String`: Hex color string.
- `full::Bool=false`: If true, scales values to the [0, 255] range; otherwise, scales to [0, 1].

# Returns
An RGB tuple.
"""
function hex_to_rgb(value ;full=false) 
    """Convert a hex color to rgb tuple."""
    value = lstrip(value,'#')
    lv = length(value)
    step = lv ÷ 3 
    scale = 1.0 / 255.0
    if full 
        scale = 1.0
    end
    col = Tuple(scale*parse(Int,value[i+begin:i+step],base=16) for i = 0:step:lv-1)
 
    return col 
end

"""
    hex_to_rgb(value, transmit; full=false)

Convert a hex color string to an RGB tuple with an additional transparency value.

# Arguments
- `value::String`: Hex color string.
- `transmit::Float64`: Transparency value.
- `full::Bool=false`: If true, scales values to the [0, 255] range; otherwise, scales to [0, 1].

# Returns
An RGB tuple with an additional transparency component.
"""
function hex_to_rgb(value, transmit; full=false) 
    """Convert a hex color to rgb tuple."""
    value = lstrip(value,'#')
    lv = len(value)
    step = lv ÷ 3 
    scale = 1.0 / 255.0
    if full 
        scale = 1.0
    end
    col = Tuple(scale*parse(Int,value[i+begin:i+step],base=16) for i = 0:step:lv-1)
 
    return col + (transmit,) 
end

"""
    rgb_to_hex(value)

Convert an RGB tuple to a hex color string.

# Arguments
- `value::Tuple`: RGB tuple where each element is in the [0, 1] or [0, 255] range.

# Returns
A hex color string.
"""
function rgb_to_hex(value) 
    """Convert a rgb tuple to a hex color string."""
    if value[1] < 1 
        scale = 255.0
    else 
        scale = 1.0
    end
    rgb = [round(Int64,scale*k) for k in value]
    return @sprintf "#%02x%02x%02x" rgb[1] rgb[2] rgb[3]  
end

"""
    get_alpha_hex(value::String, alpha::Float64)

Apply alpha transparency to a hex color string.

# Arguments
- `value::String`: Hex color string.
- `alpha::Float64`: Alpha transparency value in the range [0, 1].

# Returns
A hex color string with transparency applied.
"""
function get_alpha_hex(value,alpha) 
    """Convert a hex color to an equivalent non-transparent version."""

    #first we get the rgb
    rgb = hex_to_rgb(value)

    # apply the transparency
    target = [alpha*k + (0.999-alpha) for k in rgb] 

    return rgb_to_hex(target)
end

"""
    get_colors()

Return a list of predefined hex color strings for consistent styling.

# Returns
A vector of hex color strings.
"""
function get_colors()
    colors = ["#779ECB", "#FF6961", "#FFB347", "#B8E2C8", "#CABEE8", "#D1A677"]
    return colors
end

"""
    get_many_colors()

Return a broader list of pastel hex color strings for various styles.

# Returns
A vector of pastel hex color strings.
"""
function get_many_colors() 

    cols = [
    "#B39EB5",  # Pastel Purple
    "#C4C3E9",  # Pastel Lavender
    "#AEC6CF",  # Pastel Teal 
    "#89CFF0",  # Pastel Cyan
    "#779ECB",  # Pastel Blue  
    "#FF6961",  # Pastel Red
    "#FF9AA2",  # Pastel Coral
    "#F4A460",  # Pastel Sandy Brown
    "#FFB347",  # Pastel Orange 
    "#D1A677",  # Pastel Brown
    "#D3B3A6",  # Pastel Taupe 
    "#CFCFC4",  # Pastel Magenta 
    "#B2B982",  # Pastel Olive
    "#77DD77",  # Pastel Green
    "#DA70D6",  # Pastel Orchid 
    "#FFB6C1",   # Pastel Light Pink 
]
  
    return cols 
end
