using CairoMakie
using MathTeXEngine


"""This file contains some modified functions from https://github.com/liuyxpp/MakiePublication.jl."""  
 
"""
    figsize(width_in_cm, height_width_ratio=0.68)

Convert figure size from centimeters to points and return width and height in points.

# Arguments
- `width_in_cm`: Figure width in centimeters.
- `height_width_ratio`: Ratio of height to width, default is 0.68.

# Returns
A tuple of `(width_in_point, height_in_point)` in points.
"""
function figsize(width_in_cm, height_width_ratio=0.68)
    width_in_inch = width_in_cm/2.54
    width_in_point = floor(Int, 72width_in_inch)
    height_in_point = floor(Int, width_in_point*height_width_ratio)
    return width_in_point, height_in_point
end
 
"""
    set_default_plot_params!()

Set default plot parameters for publication-quality figures using CairoMakie.
"""
function set_default_plot_params!()
    CairoMakie.activate!(type = "png", px_per_unit = 10.0 )
    set_theme!(get_theme_one_column_figure())
end
 
"""
    get_theme_one_column_figure()

Predefined theme for a one-column figure with a default width of 8.6 cm.
"""
get_theme_one_column_figure() = get_theme(;width=8.6)
"""
    get_theme_two_column_figure_tall()

Return a predefined theme for a tall two-column figure with double the width of a one-column figure (17.2 cm).
"""
get_theme_two_column_figure_tall() = get_theme(;width=2*8.6) 
"""
    get_theme_two_column_figure()

Return a predefined theme for a standard two-column figure with double the width of a one-column figure (17.2 cm) and a smaller height.
"""
get_theme_two_column_figure() = get_theme(;width=2*8.6, height_width_ratio=0.68/2) 

"""
    get_theme(;width=8.6,height_width_ratio=0.68)

Get theme based on customizable width and height-to-width ratio for publication-quality figures.

# Arguments
- `width`: Width of the figure in cm, default is 8.6.
- `height_width_ratio`: Ratio of height to width, default is 0.68.

# Returns
A `Theme` object with predefined settings for axes, colorbars, legends, lines, and markers.
"""
function get_theme(;width=8.6,height_width_ratio=0.68)
    markersize = 6

    axis_theme = (
        xlabelsize=11,
        ylabelsize=11,
        titlesize=11,
        spinewidth=1.1,
        xticklabelsize=11,
        yticklabelsize=11, 
        xgridvisible=false,
        ygridvisible=false,
        xtickalign=1,
        ytickalign=1,
        xticksize=5,
        yticksize=5,
        xtickwidth=0.8,
        ytickwidth=0.8,
        xticksmirrored = true,
        yticksmirrored = true,
        xminorticksvisible=true,
        yminorticksvisible=true,
        xminortickalign=1,
        yminortickalign=1,
        xminorticks=IntervalsBetween(5),
        yminorticks=IntervalsBetween(5),
        xminorticksize=3,
        yminorticksize=3,
        xminortickwidth=0.75,
        yminortickwidth=0.75,
        xlabelpadding=0,
        ylabelpadding=2,
    )

    polar_axis_theme = (
        spinewidth=1.1,
    )

    colorbar_theme = (
        spinewidth=1.1,
        labelsize=11,
        tickalign=1,
        ticksize=5,
        minorticksize=3,
        minortickalign=1,
        ticklabelpad=2,
        size=8,
        tickwidth=0.8,
        minortickwidth=0.75,
        minorticksvisible=true,
        ticklabelsize=11,
        labelpadding=2, 
    )

    line_theme = (
        cycle=Cycle([:color, :marker], covary=true),
        linewidth=1.5
    )

    scatter_theme = (
        cycle=Cycle([:color, :marker], covary=true),
        markersize=markersize,
        strokewidth=1.0,
    )

    legend_theme = (
        nbanks=1,
        framecolor=(:grey, 0.5),
        framevisible=false,
        labelsize=11,
        padding=(2, 2, 2, 2),
        margin=(0, 0, 0, 0),
        # position=:rt, # l=left, r=right, c=center; b=bottom, t=top, c=center
        rowgap=-10,
        colgap=4,
    )

    pal = (color=get_colors(),
        markercolor=get_colors(),
        linestyle= [nothing],
        marker=[:circle, :diamond, :square, :rect, :xcross, :hexagon],
        patchcolor=get_colors()
    ) 

    fonts = (; regular=texfont(:text),
        bold=texfont(:bold),
        italic=texfont(:italic),
        bold_italic=texfont(:bolditalic))

    theme_args = (figure_padding=2,
        size=figsize(width,height_width_ratio),
        colgap=8,
        rowgap=8, 
        palette=pal,
        Axis=axis_theme,
        PolarAxis=polar_axis_theme,
        Lines=line_theme,
        Scatter=scatter_theme,
        Legend=legend_theme,
        Colorbar=colorbar_theme,
        markersize=markersize) 

    t = Theme(; theme_args..., fonts=fonts) 
    return t
end