# This main script is adapted from https://github.com/hbzju/PiCO
import argparse
import builtins
import math
import os
import random
from select import select
import shutil
import time
import csv
import datetime
import warnings
import torch
import torch.nn 
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import tensorboard_logger as tb_logger
import numpy as np
from model import *
from resnet import *
from utils.utils_algo import *
from utils.utils_loss import *
from utils.cub200 import load_cub200
from utils.cifar10 import load_cifar10
from utils.cifar100 import load_cifar100
from utils.tiny_imagenet import load_tiny_imagenet
from utils.flower102 import load_flower102
from utils.options import parser
import copy

torch.autograd.set_detect_anomaly(True)

torch.set_printoptions(precision=2, sci_mode=False)

args = parser.parse_args()

def main():
    args = parser.parse_args()
    args.conf_ema_range = [float(item) for item in args.conf_ema_range.split(',')]
    iterations = args.lr_decay_epochs.split(',')
    args.lr_decay_epochs = list([])
    for it in iterations:
        args.lr_decay_epochs.append(int(it))
    print(args)

    if args.seed is not None:
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed
    
    model_path = 'ds{ds}_p{pr}_n{nr}_warmup{ps}_lr{lr}_sd{seed}_{chan}_{comm}'.format(
                                            ds=args.dataset,
                                            pr=args.partial_rate,
                                            ps=args.warmup_ep,
                                            lr=args.lr,
                                            seed=args.seed,
                                            nr=args.noisy_rate,
                                            chan=args.dist_url[-5:],
                                            comm=args.comment)
    args.exp_dir = os.path.join(args.exp_dir, model_path)
    if not os.path.exists(args.exp_dir):
        os.makedirs(args.exp_dir)
    
    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)

def main_worker(gpu, ngpus_per_node, args):
    cudnn.benchmark = True
    args.gpu = gpu
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        np.random.seed(args.seed)
        cudnn.deterministic = True
    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))
    # suppress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:
        def print_pass(*args):
            pass
        builtins.print = print_pass
    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
    
    # create model
    print("=> creating model '{}'".format(args.arch))
    model = Model(args, SupConResNet)

    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu], find_unused_parameters=True)
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model, find_unused_parameters=True)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
        # comment out the following line for debugging
        raise NotImplementedError("Only DistributedDataParallel is supported.")
    else:
        # AllGather implementation (batch shuffle, queue update, etc.) in
        # this code only supports DistributedDataParallel.
        raise NotImplementedError("Only DistributedDataParallel is supported.")
    
    # set optimizer
    optimizer = torch.optim.SGD(model.parameters(), args.lr,
                                momentum=args.momentum,
                                weight_decay=args.weight_decay)
    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))
    
    if args.dataset == 'cub200':
        input_size = 224  # fixed as 224
        train_loader, train_givenY, train_sampler, test_loader = load_cub200(input_size=input_size
            , partial_rate=args.partial_rate, noisy_rate=args.noisy_rate, batch_size=args.batch_size)
    elif args.dataset == 'flower102':
        input_size = 224  # fixed as 224
        train_loader, train_givenY, train_sampler, test_loader = load_flower102(args, input_size, partial_rate=args.partial_rate, noisy_rate=args.noisy_rate, batch_size=args.batch_size)
    elif args.dataset == 'cifar10':
        train_loader, train_givenY, train_sampler, test_loader = load_cifar10(args, partial_rate=args.partial_rate, batch_size=args.batch_size, noisy_rate=args.noisy_rate)
    elif args.dataset == 'cifar100':
        train_loader, train_givenY, train_sampler, test_loader = load_cifar100(args, partial_rate=args.partial_rate, batch_size=args.batch_size, hierarchical=args.hierarchical, noisy_rate=args.noisy_rate)
    elif args.dataset == 'tinyImagenet':
        train_loader, train_givenY, train_sampler, test_loader = load_tiny_imagenet(args)
    else:
        raise NotImplementedError("You have chosen an unsupported dataset. Please check and try again.")
    # this train loader is the partial label training loader

    print('Calculating uniform targets...')
    num_instance = train_givenY.shape[0]
    tempY = train_givenY.sum(dim=1).unsqueeze(1).repeat(1, train_givenY.shape[1])
    confidence = train_givenY.float()/tempY
    confidence = confidence.cuda()
    # calculate confidence

    loss_fn = partial_loss_SAT(confidence, args)
    loss_cont_fn = ESLConLoss()
    # set loss functions (with pseudo-targets maintained)

    if args.gpu==0:
        logger = tb_logger.Logger(logdir=os.path.join(args.exp_dir,'tensorboard'), flush_secs=2)
    else:
        logger = None

    print('\nStart Training\n')

    best_acc = 0

    with open(os.path.join(args.exp_dir, 'result.log'), 'a+') as f:
            f.write('dataset:{}\n partial rate:{}\n noisy rate:{}\n learning rate:{}\n COMMENT:{} \n'.format(args.dataset,\
                args.partial_rate, args.noisy_rate, args.lr, args.comment))
    
    contrast_dict = {
        'epoch':[],
        'num_true_pos':[],
        'num_false_pos':[],
        'num_true_neg':[],
        'num_false_neg':[]
    }
    
    for epoch in range(args.start_epoch, args.epochs):
        
        is_best = False
        if args.distributed:
            train_sampler.set_epoch(epoch)
        
        adjust_learning_rate(args, optimizer, epoch)

            
        acc_train = train(args, train_loader, model, loss_fn, loss_cont_fn, optimizer, epoch, logger)
        
        loss_fn.set_conf_ema_m(args=args, epoch=epoch)
        
        contrast_dict['epoch'].append(epoch)
        contrast_dict['num_false_neg'].append(loss_cont_fn.num_false_neg)
        contrast_dict['num_false_pos'].append(loss_cont_fn.num_false_pos)
        contrast_dict['num_true_neg'].append(loss_cont_fn.num_true_neg)
        contrast_dict['num_true_pos'].append(loss_cont_fn.num_true_pos)

        save_cont_dict = os.path.join(args.exp_dir, 'contrast_dict.csv')
        with open(save_cont_dict, 'w', newline='') as csvfile:
            writer = csv.DictWriter(csvfile, fieldnames=contrast_dict.keys())
            writer.writeheader()
            for row in zip(*contrast_dict.values()):
                writer.writerow(dict(zip(contrast_dict.keys(), row)))
        
        
        acc_test = test(args, model, test_loader, epoch, logger)
        
        time_now = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        with open(os.path.join(args.exp_dir, 'result.log'), 'a+') as f:
            f.write('[{}]Epoch {}: Train Acc {:.2f}, Test Acc {:.2f}, Best Acc {:.2f}. (lr {:.4f},  output_mmc {:.2f}, L_total:{:.2f}, loss_SAT:{:.2f}, loss_cont:{:.2f})\n'.format(time_now, epoch
                ,acc_train, acc_test, best_acc, optimizer.param_groups[0]['lr'], sum(loss_fn.realtime_MMC)/len(loss_fn.realtime_MMC)\
                    , sum(loss_fn.loss_total)/len(loss_fn.loss_total), sum(loss_fn.loss_SAT)/len(loss_fn.loss_SAT), sum(loss_fn.loss_cont)/len(loss_fn.loss_cont)))
        if acc_test > best_acc:
            best_acc = acc_test
            is_best = True

        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0):
            save_checkpoint({
                'epoch': epoch + 1,
                'arch': args.arch,
                'state_dict': model.state_dict(),
                'optimizer' : optimizer.state_dict(),
            }, is_best=is_best, filename='{}/checkpoint.pth.tar'.format(args.exp_dir),
            best_file_name='{}/checkpoint_best.pth.tar'.format(args.exp_dir))

def train(args, train_loader, model, loss_fn, loss_cont_fn, optimizer, epoch, tb_logger):
    batch_time = AverageMeter('Time', ':1.2f')
    data_time = AverageMeter('Data', ':1.2f')
    acc_cls = AverageMeter('Acc@Cls', ':2.2f')
    acc_proto = AverageMeter('Acc@Proto', ':2.2f')
    loss_cls_log = AverageMeter('Loss@Cls', ':2.2f')
    loss_cont_log = AverageMeter('Loss@Cont', ':2.2f')
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, acc_cls, acc_proto, loss_cls_log, loss_cont_log],
        prefix="Epoch: [{}]".format(epoch))
    
    stop_warmup = epoch >= args.warmup_ep
    
    # switch to train mode
    model.train()
    loss_fn.realtime_MMC = []
    loss_fn.loss_total = []
    loss_fn.loss_SAT = []
    loss_fn.loss_cont = []
    loss_cont_fn.counting_init()
    
    end = time.time()
    for i, (images_w, images_s, partial_labels, true_labels, index) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        X_w, X_s, Y, index = images_w.cuda(), images_s.cuda(), partial_labels.cuda(), index.cuda()
        Y_true = true_labels.long().detach().cuda()


        cls_out, predicted_scores, all_max_confi = model(X_w, X_s, Y, args=args, stop_warmup=stop_warmup)
        
        loss_fn.realtime_MMC.append(all_max_confi.mean().item())

        if stop_warmup:
            loss_fn.confidence_update(temp_un_conf=predicted_scores, batch_index=index, batchY=Y)
            # warm up ended

        # classification loss
        loss_cls = loss_fn(cls_out,Y, index, epoch, stop_warmup=stop_warmup)
        
        
        # mixup loss
        l = np.random.beta(4, 4)
        l = max(l, 1-l)
        pseudo_labels_w = loss_fn.confidence[index]
        idx = torch.randperm(X_w.size(0))
        X_w_rand = X_w[idx]
        pseudo_label_rand = pseudo_labels_w[idx]
        X_w_mix = l * X_w + (1 - l) * X_w_rand      
        pseudo_label_mix = l * pseudo_labels_w + (1 - l) * pseudo_label_rand
        logits_mix, _ = model.module.encoder_q(X_w_mix)
        # use encoder q to avoid DDP error
        # print(logits_mix.shape)
        # print(pseudo_label_mix.shape)
        _, loss_mix = ce_loss(logits_mix, targets=pseudo_label_mix)
        
        loss_cls = loss_mix + args.cls_weight * loss_cls
        loss = loss_cls
        
        loss_cont = torch.tensor(0.0)

        loss_cls_log.update(loss_cls.item())
        loss_cont_log.update(loss_cont.item())
        
        loss_fn.loss_total.append(loss.item())
        loss_fn.loss_SAT.append(loss_cls.item())
        loss_fn.loss_cont.append(loss_cont.item())

        # log accuracy
        acc = accuracy(cls_out, Y_true)[0]
        acc_cls.update(acc[0])

 
        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()
        # measure elapsed time

        batch_time.update(time.time() - end)
        end = time.time()
        if i % args.print_freq == 0:
            progress.display(i)

    if args.gpu == 0:
        tb_logger.log_value('Train Acc', acc_cls.avg, epoch)
        # tb_logger.log_value('Prototype Acc', acc_proto.avg, epoch)
        tb_logger.log_value('Classification Loss', loss_cls_log.avg, epoch)
        tb_logger.log_value('Contrastive Loss', loss_cont_log.avg, epoch)

    return acc_cls.avg
    # return np.mean(mean_candid), np.mean(mean_all)
    
    
def reliable_set_selection(args, epoch, sel_stats):
    dist = sel_stats['dist']
    n = dist.shape[0]
    is_rel = torch.zeros(n).bool().cuda()
    sorted_idx = torch.argsort(dist)
    chosen_num = int(n * args.pure_ratio)
    is_rel[sorted_idx[:chosen_num]] = True
    sel_stats['is_rel'] = is_rel
    # select near-prototype samples as reliable and clean

def test(args, model, test_loader, epoch, tb_logger):
    with torch.no_grad():
        print('==> Evaluation...')       
        model.eval()    
        top1_acc = AverageMeter("Top1")
        top5_acc = AverageMeter("Top5")
        for batch_idx, (images, labels) in enumerate(test_loader):
            images, labels = images.cuda(), labels.cuda()
            outputs = model(images, args, eval_only=True)    
            acc1, acc5 = accuracy(outputs, labels, topk=(1, 5))
            top1_acc.update(acc1[0])
            top5_acc.update(acc5[0])
        
        # average across all processes
        acc_tensors = torch.Tensor([top1_acc.avg,top5_acc.avg]).cuda(args.gpu)
        dist.all_reduce(acc_tensors)        
        acc_tensors /= args.world_size
        
        print('Accuracy is %.2f%% (%.2f%%)'%(acc_tensors[0],acc_tensors[1]))
        if args.gpu ==0:
            tb_logger.log_value('Top1 Acc', acc_tensors[0], epoch)
            tb_logger.log_value('Top5 Acc', acc_tensors[1], epoch)             
    return acc_tensors[0]
    
def save_checkpoint(state, is_best, filename='checkpoint.pth.tar', best_file_name='model_best.pth.tar'):
    torch.save(state, filename)
    if is_best:
        shutil.copyfile(filename, best_file_name)

if __name__ == '__main__':
    main()
