import json
import os
import glob
import sys

import cv2
import numpy as np


left_img_paths = glob.glob("_data/calib_dataset/ours_2/left/*.jpg")
chessboard_size = (13, 9)
square_side = 0.020  # 20mm side
criteria = (cv2.TERM_CRITERIA_EPS + cv2.TERM_CRITERIA_MAX_ITER, 30, 0.001)

objp = np.zeros((chessboard_size[0] * chessboard_size[1], 3), np.float32)
objp[:, :2] = np.mgrid[0:chessboard_size[0], 0:chessboard_size[1]].T.reshape(-1, 2) * square_side

objpoints = []  # 3D points in real world space
imgpointsL = []  # 2D points for left camera
imgpointsR = []  # 2D points for right camera

assert left_img_paths, "No images found at the provided path"
for left_path in left_img_paths:
    print(left_path)
    right_path = left_path.replace("/left/", "/right/")
    imgL = cv2.imread(left_path, cv2.IMREAD_GRAYSCALE)
    imgR = cv2.imread(right_path, cv2.IMREAD_GRAYSCALE)
    assert imgL.shape == imgR.shape

    retL, cornersL = cv2.findChessboardCorners(imgL, chessboard_size, None)
    retR, cornersR = cv2.findChessboardCorners(imgR, chessboard_size, None)

    # ------------------- Visualizing detected corners -----------------------
    # draw_L = cv2.drawChessboardCorners(np.stack([imgL]*3, axis=2), chessboard_size, cornersL, retL)
    # draw_R = cv2.drawChessboardCorners(np.stack([imgR]*3, axis=2), chessboard_size, cornersR, retR)
    # scale = 0.5
    # cv2.imshow("Calib", np.concat([cv2.resize(draw_L, None, fx=scale, fy=scale),
    #                                cv2.resize(draw_R, None, fx=scale, fy=scale)], axis=1))
    # if cv2.waitKey(0) == ord("q"): sys.exit()

    if retL and retR:
        objpoints.append(objp)
        cornersL = cv2.cornerSubPix(imgL, cornersL, (11, 11), (-1, -1), criteria)
        cornersR = cv2.cornerSubPix(imgR, cornersR, (11, 11), (-1, -1), criteria)
        imgpointsL.append(cornersL)
        imgpointsR.append(cornersR)
    else:
        raise Exception("Chessboard corners not detected!")

retL, intrL, distL, rvecsL, tvecsL = cv2.calibrateCamera(objpoints, imgpointsL, imgL.shape[::-1], None, None)
retR, intrR, distR, rvecsR, tvecsR = cv2.calibrateCamera(objpoints, imgpointsR, imgR.shape[::-1], None, None)

flags = cv2.CALIB_FIX_INTRINSIC  # Fix individual camera parameters
retval, intrL, distL, intrR, distR, R, T, E, F = cv2.stereoCalibrate(
    objpoints, imgpointsL, imgpointsR,
    intrL, distL, intrR, distR, imgL.shape[::-1],
    criteria=criteria, flags=flags
)

calib_dict = {
    "img_size": imgL.shape[::-1],
    "intrL": intrL.tolist(),
    "intrR": intrR.tolist(),
    "distL": distL.tolist(),
    "distR": distR.tolist(),
    "R": R.tolist(),
    "T": T.tolist(),
}

os.makedirs("_artifacts", exist_ok=True)
with open("calibration.json", "w") as f:
    json.dump(calib_dict, f, indent=4)
