import sys

import calib


def error_func(_):
    """Ignores bad calibration images"""
    pass


def main():
    left_img_dir = "./left"
    right_img_dir = left_img_dir.replace("left", "right")
    corners_size = (10, 7)
    board_type = "both"  # "chessb", "charuco" or "both"
    square_side = 95e-3  # in meters
    visualize = False
    save_file = "calibration.json"
    report_file = "calib_report.txt"

    # Iteration parameters
    mono_target_err = 0.5
    stereo_target_err = 1.0
    reject_per_iter = 1
    max_imgs_per_iter = 100
    max_iter = 100

    print("Left image dir: ", left_img_dir)
    print("Right image dir:", right_img_dir, "\n")

    if not str(save_file).endswith(".json"):
        raise ValueError("`save_file` should be a json")

    replace = calib.confirm_replace([save_file, report_file])
    if replace is not None and not replace:
        sys.exit()
    elif replace:
        print()

    board = calib.BoardConfig(
        corners_size,
        board_type,
        square_side,
        square_side * 0.75,
    )

    print("Calibrating left camera...")
    calib.log_fmt.prefix += "Left "
    ret, params_l, rerr_l = calib.iter_mb_calibrate(
        left_img_dir,
        board,
        mono_target_err,
        reject_per_iter,
        max_imgs_per_iter,
        max_iter,
        visualize,
        error_func,
    )
    if not ret:
        raise Exception("Left camera calibration failed!")

    calib.log_fmt.prefix = calib.log_fmt.prefix[:-5]
    rerr_l = sum(rerr_l.values()) / len(rerr_l)

    print("\nCalibrating right camera...")
    calib.log_fmt.prefix += "Right "
    ret, params_r, rerr_r = calib.iter_mb_calibrate(
        right_img_dir,
        board,
        mono_target_err,
        reject_per_iter,
        max_imgs_per_iter,
        max_iter,
        visualize,
        error_func,
    )
    if not ret:
        raise Exception("Right camera calibration failed!")

    calib.log_fmt.prefix = calib.log_fmt.prefix[:-6]
    rerr_r = sum(rerr_r.values()) / len(rerr_r)

    print("\nCalibrating stereo setup...")
    calib.log_fmt.prefix += "Stereo "
    ret, params_s, rerr_s = calib.iter_mb_stereo(
        left_img_dir,
        right_img_dir,
        params_l,
        params_r,
        board,
        stereo_target_err,
        reject_per_iter,
        max_imgs_per_iter,
        max_iter,
        visualize,
    )
    if not ret:
        raise Exception("Stereo camera calibration failed!")

    calib.log_fmt.prefix = calib.log_fmt.prefix[:-7]
    rerr_s = sum(rerr_s.values()) / len(rerr_s)

    print()
    print("Left Camera Reprojection Error: ", rerr_l)
    print("Right Camera Reprojection Error:", rerr_r)
    print("Stereo Reprojection Error:      ", rerr_s)

    calib.save_params(params_s, save_file)
    print("\nAll parameters saved to", save_file)

    report = calib.validation_report(
        params_s["R"], params_s["T"], rerr_l, rerr_r, rerr_s
    )
    with open(report_file, "w") as f:
        f.write(report)
    print("Validation report saved at", report_file)


if __name__ == "__main__":
    main()
