import os
import json

import cv2
import numpy as np


def process_calib_json(path: str, img_size: tuple[int, int]) -> tuple:
    """
    Process the calibration json file and generate the rectification transforms.

    Args:
        path: Path to the calibration file.
        img_size: Output image size after the subsequent `remap`.

    Returns:
        A tuple with the following items:
            1. Focal length of the stereo camera setup after rectification.
            2. Baseline of the stereo camera setup after rectification.
            3. A tuple of X and Y-axis maps to `remap` the left image.
            4. A tuple of X and Y-axis maps to `remap` the right image.
            5. A dictionary containing the following keys in addition to the
               contents of the json file:
               - "rectL": Rectification transform for the left camera images.
               - "rectR": Rectification transform for the right camera images.
               - "projL": Left camera projection matrix (rectified "intrL").
               - "projR": Right camera projection matrix (rectified "intrR").
               - "Q": Disparity to depth mapping matrix.
    """
    if not os.path.exists(path):
        raise FileNotFoundError(f"{path} doesn't exist!")
    with open(path, "r") as f:
        p = {k: np.array(v) for k, v in json.load(f).items()}

    rectify_output = cv2.stereoRectify(
        p["intrL"], p["distL"], p["intrR"], p["distR"],
        p["img_size"], p["R"], p["T"],
    )
    rectL, rectR, projL, projR, p["Q"], _, _ = rectify_output
    p["rectL"], p["rectR"] = rectL, rectR
    p["projL"], p["projR"] = projL, projR
    maps_L = cv2.initUndistortRectifyMap(p["intrL"], p["distL"], rectL,
                                         projL, img_size, cv2.CV_16SC2)
    maps_R = cv2.initUndistortRectifyMap(p["intrR"], p["distR"], rectR,
                                         projR, img_size, cv2.CV_16SC2)
    f = projR[0, 0].item()
    b = max(np.abs(projR[0, 3]), np.abs(projR[1, 3])).item() / f
    return f, b, maps_L, maps_R, p
