"""
This script sets up the Logger for this package.

Environment variable `CALIB_LOG_LEVEL` can be set to "DEBUG", "INFO", "WARNING",
"ERROR" or "CRITICAL" to set the initial logging level for the `logger`.

The following objects can be imported:
    1. logger: Can be used to change logging levels, etc.
    2. log_fmt: Can be used to change the prefix of the logging message.
        Use the `prefix` attribute for this.
"""

import os
import logging
from logging import LogRecord


class CalibFormatter(logging.Formatter):
    """
    Logging formatter with a modifiable prefix.
    """

    def __init__(
        self, fmt=None, datefmt=None, style="%", validate=True, *, defaults=None
    ):
        super().__init__(fmt, datefmt, style, validate, defaults=defaults)
        self.prefix = ""

    def format(self, record: LogRecord) -> str:
        return self.prefix + super().format(record)


def log_no_prefix(msg: str = "", level: int = logging.INFO):
    """
    Convenience function to log a message without any prefix.
    Uses the `logger` and `log_fmt` objects defined in `calib.logging`.

    Args:
        msg: Log message.
        level: Logging level.

    Returns:
        None
    """
    prefix = log_fmt.prefix
    log_fmt.prefix = ""
    logger.log(level, msg)
    log_fmt.prefix = prefix


# --------------------------------------------------------------------------------

logger = logging.getLogger("calib")
log_fmt = CalibFormatter("%(message)s")

handler = logging.StreamHandler()
handler.setFormatter(log_fmt)
logger.addHandler(handler)
logger.setLevel(os.getenv("CALIB_LOG_LEVEL", "INFO"))

logger.debug(f"{logger} setup complete")
