import os
from glob import glob
from pathlib import Path
import json

import cv2
import numpy as np

from .types import StrPath


def imgs_in_dir(dir_path: StrPath) -> list[StrPath]:
    """
    Returns a list of paths of all images immediately inside the given directory.
    The object type of paths returned will the same as `dir_path`.
    """
    img_exts = (".png", ".jpg", ".jpeg", ".webp", ".bmp")
    img_exts += tuple(x.upper() for x in img_exts)

    img_paths = glob(os.path.join(str(dir_path), "*"))
    img_paths = list(filter(lambda x: x.endswith(img_exts), img_paths))
    if not isinstance(dir_path, str):
        img_paths = [dir_path.__class__(p) for p in img_paths]
    return img_paths


def calc_win_size(img_shape: tuple) -> tuple[int, int]:
    """Calculate window size for a 1080p screen"""
    h, w = img_shape[:2]
    if w / h >= 16 / 9:
        scale = 1800 / w
    else:
        scale = 900 / h
    w, h = round(w * scale), round(h * scale)
    return w, h


def image_window(win_name: str, img: np.ndarray) -> int:
    """Image window resized to fit a 1080p screen"""
    win_size = calc_win_size(img.shape)
    cv2.namedWindow(win_name, cv2.WINDOW_NORMAL)
    cv2.resizeWindow(win_name, *win_size)
    cv2.imshow(win_name, img)
    keycode = ord("z") + 1
    while keycode > ord("z"):
        keycode = cv2.waitKey(0) & 0xFF
    cv2.destroyWindow(win_name)
    return keycode


def save_params(p: dict, path: StrPath) -> None:
    "Save a parameter dict at the specified path as a json file."
    path = Path(path)
    if not path.name.lower().endswith(".json"):
        raise ValueError("`p` can only be saved in a .json file")

    final_params = {}
    for k, v in p.items():
        if isinstance(v, tuple):
            v = list(v)
        elif isinstance(v, np.ndarray):
            v = v.tolist()
        elif isinstance(v, dict):
            continue  # Requires recursive logic
        else:
            raise TypeError(f"Invalid type in parameter dict: {type(v)}")
        final_params[k] = v

    path.parent.mkdir(parents=True, exist_ok=True)
    with open(path, "w") as f:
        json.dump(final_params, f, indent=4)


def confirm_replace(paths: list[StrPath]) -> bool | None:
    """
    Given a list of paths, asks the user if they would like to replace the existing
    files/directories. Returns `True` if the user agrees, `False` if they don't and
    `None` if none of the paths exist.
    """
    existing_paths = []
    for p in paths:
        if Path(p).exists():
            existing_paths.append(p)

    if existing_paths:
        print("Following files/directories will be overwritten:")
        for p in existing_paths:
            print(f" - {p}")
        ans = input("Continue? (y/N) ").strip().lower()
        if ans != "y":
            return False
    else:
        return None

    return True
