# code in this file is adpated from rpmcruz/autoaugment
# https://github.com/rpmcruz/autoaugment/blob/master/transformations.py
import random

import PIL, PIL.ImageOps, PIL.ImageEnhance, PIL.ImageDraw
import numpy as np
import torch
from torchvision.transforms.transforms import Compose

random_mirror = True


def ShearX(img, v):  # [-0.3, 0.3]
    assert -0.3 <= v <= 0.3
    if random_mirror and random.random() > 0.5:
        v = -v
    return img.transform(img.size, PIL.Image.AFFINE, (1, v, 0, 0, 1, 0))


def ShearY(img, v):  # [-0.3, 0.3]
    assert -0.3 <= v <= 0.3
    if random_mirror and random.random() > 0.5:
        v = -v
    return img.transform(img.size, PIL.Image.AFFINE, (1, 0, 0, v, 1, 0))


def TranslateX(img, v):  # [-150, 150] => percentage: [-0.45, 0.45]
    assert -0.45 <= v <= 0.45
    if random_mirror and random.random() > 0.5:
        v = -v
    v = v * img.size[0]
    return img.transform(img.size, PIL.Image.AFFINE, (1, 0, v, 0, 1, 0))


def TranslateY(img, v):  # [-150, 150] => percentage: [-0.45, 0.45]
    assert -0.45 <= v <= 0.45
    if random_mirror and random.random() > 0.5:
        v = -v
    v = v * img.size[1]
    return img.transform(img.size, PIL.Image.AFFINE, (1, 0, 0, 0, 1, v))


def TranslateXAbs(img, v):  # [-150, 150] => percentage: [-0.45, 0.45]
    assert 0 <= v <= 10
    if random.random() > 0.5:
        v = -v
    return img.transform(img.size, PIL.Image.AFFINE, (1, 0, v, 0, 1, 0))


def TranslateYAbs(img, v):  # [-150, 150] => percentage: [-0.45, 0.45]
    assert 0 <= v <= 10
    if random.random() > 0.5:
        v = -v
    return img.transform(img.size, PIL.Image.AFFINE, (1, 0, 0, 0, 1, v))


def Rotate(img, v):  # [-30, 30]
    assert -30 <= v <= 30
    if random_mirror and random.random() > 0.5:
        v = -v
    return img.rotate(v)


def AutoContrast(img, _):
    return PIL.ImageOps.autocontrast(img)


def Invert(img, _):
    return PIL.ImageOps.invert(img)


def Equalize(img, _):
    return PIL.ImageOps.equalize(img)


def Flip(img, _):  # not from the paper
    return PIL.ImageOps.mirror(img)


def Solarize(img, v):  # [0, 256]
    assert 0 <= v <= 256
    return PIL.ImageOps.solarize(img, v)


def Posterize(img, v):  # [4, 8]
    assert 4 <= v <= 8
    v = int(v)
    return PIL.ImageOps.posterize(img, v)


def Posterize2(img, v):  # [0, 4]
    assert 0 <= v <= 4
    v = int(v)
    return PIL.ImageOps.posterize(img, v)


def Contrast(img, v):  # [0.1,1.9]
    assert 0.1 <= v <= 1.9
    return PIL.ImageEnhance.Contrast(img).enhance(v)


def Color(img, v):  # [0.1,1.9]
    assert 0.1 <= v <= 1.9
    return PIL.ImageEnhance.Color(img).enhance(v)


def Brightness(img, v):  # [0.1,1.9]
    assert 0.1 <= v <= 1.9
    return PIL.ImageEnhance.Brightness(img).enhance(v)


def Sharpness(img, v):  # [0.1,1.9]
    assert 0.1 <= v <= 1.9
    return PIL.ImageEnhance.Sharpness(img).enhance(v)


def Cutout(img, v):  # [0, 60] => percentage: [0, 0.2]
    assert 0.0 <= v <= 0.2
    if v <= 0.:
        return img

    v = v * img.size[0]
    return CutoutAbs(img, v)


def CutoutAbs(img, v):  # [0, 60] => percentage: [0, 0.2]
    # assert 0 <= v <= 20
    if v < 0:
        return img
    w, h = img.size
    x0 = np.random.uniform(w)
    y0 = np.random.uniform(h)

    x0 = int(max(0, x0 - v / 2.))
    y0 = int(max(0, y0 - v / 2.))
    x1 = min(w, x0 + v)
    y1 = min(h, y0 + v)

    xy = (x0, y0, x1, y1)
    color = (125, 123, 114)
    # color = (0, 0, 0)
    img = img.copy()
    PIL.ImageDraw.Draw(img).rectangle(xy, color)
    return img


def SamplePairing(imgs):  # [0, 0.4]
    def f(img1, v):
        i = np.random.choice(len(imgs))
        img2 = PIL.Image.fromarray(imgs[i])
        return PIL.Image.blend(img1, img2, v)

    return f


def augment_list(for_autoaug=True):  # 16 oeprations and their ranges
    l = [
        (ShearX, -0.3, 0.3),  # 0
        (ShearY, -0.3, 0.3),  # 1
        (TranslateX, -0.45, 0.45),  # 2
        (TranslateY, -0.45, 0.45),  # 3
        (Rotate, -30, 30),  # 4
        (AutoContrast, 0, 1),  # 5
        (Invert, 0, 1),  # 6
        (Equalize, 0, 1),  # 7
        (Solarize, 0, 256),  # 8
        (Posterize, 4, 8),  # 9
        (Contrast, 0.1, 1.9),  # 10
        (Color, 0.1, 1.9),  # 11
        (Brightness, 0.1, 1.9),  # 12
        (Sharpness, 0.1, 1.9),  # 13
        (Cutout, 0, 0.2),  # 14
        # (SamplePairing(imgs), 0, 0.4),  # 15
    ]
    if for_autoaug:
        l += [
            (CutoutAbs, 0, 20),  # compatible with auto-augment
            (Posterize2, 0, 4),  # 9
            (TranslateXAbs, 0, 10),  # 9
            (TranslateYAbs, 0, 10),  # 9
        ]
    return l


augment_dict = {fn.__name__: (fn, v1, v2) for fn, v1, v2 in augment_list()}


def get_augment(name):
    return augment_dict[name]


def apply_augment(img, name, level):
    augment_fn, low, high = get_augment(name)
    return augment_fn(img.copy(), level * (high - low) + low)


class Lighting(object):
    """Lighting noise(AlexNet - style PCA - based noise)"""

    def __init__(self, alphastd, eigval, eigvec):
        self.alphastd = alphastd
        self.eigval = torch.Tensor(eigval)
        self.eigvec = torch.Tensor(eigvec)

    def __call__(self, img):
        if self.alphastd == 0:
            return img

        alpha = img.new().resize_(3).normal_(0, self.alphastd)
        rgb = self.eigvec.type_as(img).clone() \
            .mul(alpha.view(1, 3).expand(3, 3)) \
            .mul(self.eigval.view(1, 3).expand(3, 3)) \
            .sum(1).squeeze()

        return img.add(rgb.view(3, 1, 1).expand_as(img))

def fa_reduced_cifar10():
    p = [[["Contrast", 0.8320659688593578, 0.49884310562180767], ["TranslateX", 0.41849883971249136, 0.394023086494538]], [["Color", 0.3500483749890918, 0.43355143929883955], ["Color", 0.5120716140300229, 0.7508299643325016]], [["Rotate", 0.9447932604389472, 0.29723465088990375], ["Sharpness", 0.1564936149799504, 0.47169309978091745]], [["Rotate", 0.5430015349185097, 0.6518626678905443], ["Color", 0.5694844928020679, 0.3494533005430269]], [["AutoContrast", 0.5558922032451064, 0.783136004977799], ["TranslateY", 0.683914191471972, 0.7597025305860181]], [["TranslateX", 0.03489224481658926, 0.021025488042663354], ["Equalize", 0.4788637403857401, 0.3535481281496117]], [["Sharpness", 0.6428916269794158, 0.22791511918580576], ["Contrast", 0.016014045073950323, 0.26811312269487575]], [["Rotate", 0.2972727228410451, 0.7654251516829896], ["AutoContrast", 0.16005809254943348, 0.5380523650108116]], [["Contrast", 0.5823671057717301, 0.7521166301398389], ["TranslateY", 0.9949449214751978, 0.9612671341689751]], [["Equalize", 0.8372126687702321, 0.6944127225621206], ["Rotate", 0.25393282929784755, 0.3261658365286546]], [["Invert", 0.8222011603194572, 0.6597915864008403], ["Posterize", 0.31858707654447327, 0.9541013715579584]], [["Sharpness", 0.41314621282107045, 0.9437344470879956], ["Cutout", 0.6610495837889337, 0.674411664255093]], [["Contrast", 0.780121736705407, 0.40826152397463156], ["Color", 0.344019192125256, 0.1942922781355767]], [["Rotate", 0.17153139555621344, 0.798745732456474], ["Invert", 0.6010555860501262, 0.320742172554767]], [["Invert", 0.26816063450777416, 0.27152062163148327], ["Equalize", 0.6786829200236982, 0.7469412443514213]], [["Contrast", 0.3920564414367518, 0.7493644582838497], ["TranslateY", 0.8941657805606704, 0.6580846856375955]], [["Equalize", 0.875509207399372, 0.9061130537645283], ["Cutout", 0.4940280679087308, 0.7896229623628276]], [["Contrast", 0.3331423298065147, 0.7170041362529597], ["ShearX", 0.7425484291842793, 0.5285117152426109]], [["Equalize", 0.97344237365026, 0.4745759720473106], ["TranslateY", 0.055863458430295276, 0.9625142022954672]], [["TranslateX", 0.6810614083109192, 0.7509937355495521], ["TranslateY", 0.3866463019475701, 0.5185481505576112]], [["Sharpness", 0.4751529944753671, 0.550464012488733], ["Cutout", 0.9472914750534814, 0.5584925992985023]], [["Contrast", 0.054606784909375095, 0.17257080196712182], ["Cutout", 0.6077026782754803, 0.7996504165944938]], [["ShearX", 0.328798428243695, 0.2769563264079157], ["Cutout", 0.9037632437023772, 0.4915809476763595]], [["Cutout", 0.6891202672363478, 0.9951490996172914], ["Posterize", 0.06532762462628705, 0.4005246609075227]], [["TranslateY", 0.6908583592523334, 0.725612120376128], ["Rotate", 0.39907735501746666, 0.36505798032223147]], [["TranslateX", 0.10398364107399072, 0.5913918470536627], ["Rotate", 0.7169811539340365, 0.8283850670648724]], [["ShearY", 0.9526373530768361, 0.4482347365639251], ["Contrast", 0.4203947336351471, 0.41526799558953864]], [["Contrast", 0.24894431199700073, 0.09578870500994707], ["Solarize", 0.2273713345927395, 0.6214942914963707]], [["TranslateX", 0.06331228870032912, 0.8961907489444944], ["Cutout", 0.5110007859958743, 0.23704875994050723]], [["Cutout", 0.3769183548846172, 0.6560944580253987], ["TranslateY", 0.7201924599434143, 0.4132476526938319]], [["Invert", 0.6707431156338866, 0.11622795952464149], ["Posterize", 0.12075972752370845, 0.18024933294172307]], [["Color", 0.5010057264087142, 0.5277767327434318], ["Rotate", 0.9486115946366559, 0.31485546630220784]], [["ShearX", 0.31741302466630406, 0.1991215806270692], ["Invert", 0.3744727015523084, 0.6914113986757578]], [["Brightness", 0.40348479064392617, 0.8924182735724888], ["Brightness", 0.1973098763857779, 0.3939288933689655]], [["Color", 0.01208688664030888, 0.6055693000885217], ["Equalize", 0.433259451147881, 0.420711137966155]], [["Cutout", 0.2620018360076487, 0.11594468278143644], ["Rotate", 0.1310401567856766, 0.7244318146544101]], [["ShearX", 0.15249651845933576, 0.35277277071866986], ["Contrast", 0.28221794032094016, 0.42036586509397444]], [["Brightness", 0.8492912150468908, 0.26386920887886056], ["Solarize", 0.8764208056263386, 0.1258195122766067]], [["ShearX", 0.8537058239675831, 0.8415101816171269], ["AutoContrast", 0.23958568830416294, 0.9889049529564014]], [["Rotate", 0.6463207930684552, 0.8750192129056532], ["Contrast", 0.6865032211768652, 0.8564981333033417]], [["Equalize", 0.8877190311811044, 0.7370995897848609], ["TranslateX", 0.9979660314391368, 0.005683998913244781]], [["Color", 0.6420017551677819, 0.6225337265571229], ["Solarize", 0.8344504978566362, 0.8332856969941151]], [["ShearX", 0.7439332981992567, 0.9747608698582039], ["Equalize", 0.6259189804002959, 0.028017478098245174]], [["TranslateY", 0.39794770293366843, 0.8482966537902709], ["Rotate", 0.9312935630405351, 0.5300586925826072]], [["Cutout", 0.8904075572021911, 0.3522934742068766], ["Equalize", 0.6431186289473937, 0.9930577962126151]], [["Contrast", 0.9183553386089476, 0.44974266209396685], ["TranslateY", 0.8193684583123862, 0.9633741156526566]], [["ShearY", 0.616078299924283, 0.19219314358924766], ["Solarize", 0.1480945914138868, 0.05922109541654652]], [["Solarize", 0.25332455064128157, 0.18853037431947994], ["ShearY", 0.9518390093954243, 0.14603930044061142]], [["Color", 0.8094378664335412, 0.37029830225408433], ["Contrast", 0.29504113617467465, 0.065096365468442]], [["AutoContrast", 0.7075167558685455, 0.7084621693458267], ["Sharpness", 0.03555539453323875, 0.5651948313888351]], [["TranslateY", 0.5969982600930229, 0.9857264201029572], ["Rotate", 0.9898628564873607, 0.1985685534926911]], [["Invert", 0.14915939942810352, 0.6595839632446547], ["Posterize", 0.768535289994361, 0.5997358684618563]], [["Equalize", 0.9162691815967111, 0.3331035307653627], ["Color", 0.8169118187605557, 0.7653910258006366]], [["Rotate", 0.43489185299530897, 0.752215269135173], ["Brightness", 0.1569828560334806, 0.8002808712857853]], [["Invert", 0.931876215328345, 0.029428644395760872], ["Equalize", 0.6330036052674145, 0.7235531014288485]], [["ShearX", 0.5216138393704968, 0.849272958911589], ["AutoContrast", 0.19572688655120263, 0.9786551568639575]], [["ShearX", 0.9899586208275011, 0.22580547500610293], ["Brightness", 0.9831311903178727, 0.5055159610855606]], [["Brightness", 0.29179117009211486, 0.48003584672937294], ["Solarize", 0.7544252317330058, 0.05806581735063043]], [["AutoContrast", 0.8919800329537786, 0.8511261613698553], ["Contrast", 0.49199446084551035, 0.7302297140181429]], [["Cutout", 0.7079723710644835, 0.032565015538375874], ["AutoContrast", 0.8259782090388609, 0.7860708789468442]], [["Posterize", 0.9980262659801914, 0.6725084224935673], ["ShearY", 0.6195568269664682, 0.5444170291816751]], [["Posterize", 0.8687351834713217, 0.9978004914422602], ["Equalize", 0.4532646848325955, 0.6486748015710573]], [["Contrast", 0.2713928776950594, 0.15255249557027806], ["ShearY", 0.9276834387970199, 0.5266542862333478]], [["AutoContrast", 0.5240786618055582, 0.9325642258930253], ["Cutout", 0.38448627892037357, 0.21219415055662394]], [["TranslateX", 0.4299517937295352, 0.20133751201386152], ["TranslateX", 0.6753468310276597, 0.6985621035400441]], [["Rotate", 0.4006472499103597, 0.6704748473357586], ["Equalize", 0.674161668148079, 0.6528530101705237]], [["Equalize", 0.9139902833674455, 0.9015103149680278], ["Sharpness", 0.7289667720691948, 0.7623606352376232]], [["Cutout", 0.5911267429414259, 0.5953141187177585], ["Rotate", 0.5219064817468504, 0.11085141355857986]], [["TranslateX", 0.3620095133946267, 0.26194039409492476], ["Rotate", 0.3929841359545597, 0.4913406720338047]], [["Invert", 0.5175298901458896, 0.001661410821811482], ["Invert", 0.004656581318332242, 0.8157622192213624]], [["AutoContrast", 0.013609693335051465, 0.9318651749409604], ["Invert", 0.8980844358979592, 0.2268511862780368]], [["ShearY", 0.7717126261142194, 0.09975547983707711], ["Equalize", 0.7808494401429572, 0.4141412091009955]], [["TranslateX", 0.5878675721341552, 0.29813268038163376], ["Posterize", 0.21257276051591356, 0.2837285296666412]], [["Brightness", 0.4268335108566488, 0.4723784991635417], ["Cutout", 0.9386262901570471, 0.6597686851494288]], [["ShearX", 0.8259423807590159, 0.6215304795389204], ["Invert", 0.6663365779667443, 0.7729669184580387]], [["ShearY", 0.4801338723951297, 0.5220145420100984], ["Solarize", 0.9165803796596582, 0.04299335502862134]], [["Color", 0.17621114853558817, 0.7092601754635434], ["ShearX", 0.9014406936728542, 0.6028711944367818]], [["Rotate", 0.13073284972300658, 0.9088831512880851], ["ShearX", 0.4228105332316806, 0.7985249783662675]], [["Brightness", 0.9182753692730031, 0.0063635477774044436], ["Color", 0.4279825602663798, 0.28727149118585327]], [["Equalize", 0.578218285372267, 0.9611758542158054], ["Contrast", 0.5471552264150691, 0.8819635504027596]], [["Brightness", 0.3208589067274543, 0.45324733565167497], ["Solarize", 0.5218455808633233, 0.5946097503647126]], [["Equalize", 0.3790381278653, 0.8796082535775276], ["Solarize", 0.4875526773149246, 0.5186585878052613]], [["ShearY", 0.12026461479557571, 0.1336953429068397], ["Posterize", 0.34373988646025766, 0.8557727670803785]], [["Cutout", 0.2396745247507467, 0.8123036135209865], ["Equalize", 0.05022807681008945, 0.6648492261984383]], [["Brightness", 0.35226676470748264, 0.5950011514888855], ["Rotate", 0.27555076067000894, 0.9170063321486026]], [["ShearX", 0.320224630647278, 0.9683584649071976], ["Invert", 0.6905585196648905, 0.5929115667894518]], [["Color", 0.9941395717559652, 0.7474441679798101], ["Sharpness", 0.7559998478658021, 0.6656052889626682]], [["ShearY", 0.4004220568345669, 0.5737646992826074], ["Equalize", 0.9983495213746147, 0.8307907033362303]], [["Color", 0.13726809242038207, 0.9378850119950549], ["Equalize", 0.9853362454752445, 0.42670264496554156]], [["Invert", 0.13514636153298576, 0.13516363849081958], ["Sharpness", 0.2031189356693901, 0.6110226359872745]], [["TranslateX", 0.7360305209630797, 0.41849698571655614], ["Contrast", 0.8972161549144564, 0.7820296625565641]], [["Color", 0.02713118828682548, 0.717110684828096], ["TranslateY", 0.8118759006836348, 0.9120098002024992]], [["Sharpness", 0.2915428949403711, 0.7630303724396518], ["Solarize", 0.22030536162851078, 0.38654526772661757]], [["Equalize", 0.9949114839538582, 0.7193630656062793], ["AutoContrast", 0.00889496657931299, 0.2291400476524672]], [["Rotate", 0.7120948976490488, 0.7804359309791055], ["Cutout", 0.10445418104923654, 0.8022999156052766]], [["Equalize", 0.7941710117902707, 0.8648170634288153], ["Invert", 0.9235642581144047, 0.23810725859722381]], [["Cutout", 0.3669397998623156, 0.42612815083245004], ["Solarize", 0.5896322046441561, 0.40525016166956795]], [["Color", 0.8389858785714184, 0.4805764176488667], ["Rotate", 0.7483931487048825, 0.4731174601400677]], [["Sharpness", 0.19006538611394763, 0.9480745790240234], ["TranslateY", 0.13904429049439282, 0.04117685330615939]], [["TranslateY", 0.9958097661701637, 0.34853788612580905], ["Cutout", 0.2235829624082113, 0.3737887095480745]], [["ShearX", 0.635453761342424, 0.6063917273421382], ["Posterize", 0.8738297843709666, 0.4893042590265556]], [["Brightness", 0.7907245198402727, 0.7082189713070691], ["Color", 0.030313003541849737, 0.6927897798493439]], [["Cutout", 0.6965622481073525, 0.8103522907758203], ["ShearY", 0.6186794303078708, 0.28640671575703547]], [["ShearY", 0.43734910588450226, 0.32549342535621517], ["ShearX", 0.08154980987651872, 0.3286764923112455]], [["AutoContrast", 0.5262462005050853, 0.8175584582465848], ["Contrast", 0.8683217097363655, 0.548776281479276]], [["ShearY", 0.03957878500311707, 0.5102350637943197], ["Rotate", 0.13794708520303778, 0.38035687712954236]], [["Sharpness", 0.634288567312677, 0.6387948309075822], ["AutoContrast", 0.13437288694693272, 0.7150448869023095]], [["Contrast", 0.5198339640088544, 0.9409429390321714], ["Cutout", 0.09489154903321972, 0.6228488803821982]], [["Equalize", 0.8955909061806043, 0.7727336527163008], ["AutoContrast", 0.6459479564441762, 0.7065467781139214]], [["Invert", 0.07214420843537739, 0.15334721382249505], ["ShearX", 0.9242027778363903, 0.5809187849982554]], [["Equalize", 0.9144084379856188, 0.9457539278608998], ["Sharpness", 0.14337499858300173, 0.5978054365425495]], [["Posterize", 0.18894269796951202, 0.14676331276539045], ["Equalize", 0.846204299950047, 0.0720601838168885]], [["Contrast", 0.47354445405741163, 0.1793650330107468], ["Solarize", 0.9086106327264657, 0.7578807802091502]], [["AutoContrast", 0.11805466892967886, 0.6773620948318575], ["TranslateX", 0.584222568299264, 0.9475693349391936]], [["Brightness", 0.5833017701352768, 0.6892593824176294], ["AutoContrast", 0.9073141314561828, 0.5823085733964589]], [["TranslateY", 0.5711231614144834, 0.6436240447620021], ["Contrast", 0.21466964050052473, 0.8042843954486391]], [["Contrast", 0.22967904487976765, 0.2343103109298762], ["Invert", 0.5502897289159286, 0.386181060792375]], [["Invert", 0.7008423439928628, 0.4234003051405053], ["Rotate", 0.77270460187611, 0.6650852696828039]], [["Invert", 0.050618322309703534, 0.24277027926683614], ["TranslateX", 0.789703489736613, 0.5116446685339312]], [["Color", 0.363898083076868, 0.7870323584210503], ["ShearY", 0.009608425513626617, 0.6188625018465327]], [["TranslateY", 0.9447601615216088, 0.8605867115798349], ["Equalize", 0.24139180127003634, 0.9587337957930782]], [["Equalize", 0.3968589440144503, 0.626206375426996], ["Solarize", 0.3215967960673186, 0.826785464835443]], [["TranslateX", 0.06947339047121326, 0.016705969558222122], ["Contrast", 0.6203392406528407, 0.6433525559906872]], [["Solarize", 0.2479835265518212, 0.6335009955617831], ["Sharpness", 0.6260191862978083, 0.18998095149428562]], [["Invert", 0.9818841924943431, 0.03252098144087934], ["TranslateY", 0.9740718042586802, 0.32038951753031475]], [["Solarize", 0.8795784664090814, 0.7014953994354041], ["AutoContrast", 0.8508018319577783, 0.09321935255338443]], [["Color", 0.8067046326105318, 0.13732893832354054], ["Contrast", 0.7358549680271418, 0.7880588355974301]], [["Posterize", 0.5005885536838065, 0.7152229305267599], ["ShearX", 0.6714249591308944, 0.7732232697859908]], [["TranslateY", 0.5657943483353953, 0.04622399873706862], ["AutoContrast", 0.2787442688649845, 0.567024378767143]], [["ShearY", 0.7589839214283295, 0.041071003934029404], ["Equalize", 0.3719852873722692, 0.43285778682687326]], [["Posterize", 0.8841266183653291, 0.42441306955476366], ["Cutout", 0.06578801759412933, 0.5961125797961526]], [["Rotate", 0.4057875004314082, 0.20241115848366442], ["AutoContrast", 0.19331542807918067, 0.7175484678480565]], [["Contrast", 0.20331327116693088, 0.17135387852218742], ["Cutout", 0.6282459410351067, 0.6690015305529187]], [["ShearX", 0.4309850328306535, 0.99321178125828], ["AutoContrast", 0.01809604030453338, 0.693838277506365]], [["Rotate", 0.24343531125298268, 0.5326412444169899], ["Sharpness", 0.8663989992597494, 0.7643990609130789]], [["Rotate", 0.9785019204622459, 0.8941922576710696], ["ShearY", 0.3823185048761075, 0.9258854046017292]], [["ShearY", 0.5502613342963388, 0.6193478797355644], ["Sharpness", 0.2212116534610532, 0.6648232390110979]], [["TranslateY", 0.43222920981513757, 0.5657636397633089], ["ShearY", 0.9153733286073634, 0.4868521171273169]], [["Posterize", 0.12246560519738336, 0.9132288825898972], ["Cutout", 0.6058471327881816, 0.6426901876150983]], [["Color", 0.3693970222695844, 0.038929141432555436], ["Equalize", 0.6228052875653781, 0.05064436511347281]], [["Color", 0.7172600331356893, 0.2824542634766688], ["Color", 0.425293116261649, 0.1796441283313972]], [["Cutout", 0.7539608428122959, 0.9896141728228921], ["Solarize", 0.17811081117364758, 0.9064195503634402]], [["AutoContrast", 0.6761242607012717, 0.6484842446399923], ["AutoContrast", 0.1978135076901828, 0.42166879492601317]], [["ShearY", 0.25901666379802524, 0.4770778270322449], ["Solarize", 0.7640963173407052, 0.7548463227094349]], [["TranslateY", 0.9222487731783499, 0.33658389819616463], ["Equalize", 0.9159112511468139, 0.8877136302394797]], [["TranslateX", 0.8994836977137054, 0.11036053676846591], ["Sharpness", 0.9040333410652747, 0.007266095214664592]], [["Invert", 0.627758632524958, 0.8075245097227242], ["Color", 0.7525387912148516, 0.05950239294733184]], [["TranslateX", 0.43505193292761857, 0.38108822876120796], ["TranslateY", 0.7432578052364004, 0.685678116134759]], [["Contrast", 0.9293507582470425, 0.052266842951356196], ["Posterize", 0.45187123977747456, 0.8228290399726782]], [["ShearX", 0.07240786542746291, 0.8945667925365756], ["Brightness", 0.5305443506561034, 0.12025274552427578]], [["Invert", 0.40157564448143335, 0.5364745514006678], ["Posterize", 0.3316124671813876, 0.43002413237035997]], [["ShearY", 0.7152314630009072, 0.1938339083417453], ["Invert", 0.14102478508140615, 0.41047623580174253]], [["Equalize", 0.19862832613849246, 0.5058521685279254], ["Sharpness", 0.16481208629549782, 0.29126323102770557]], [["Equalize", 0.6951591703541872, 0.7294822018800076], ["ShearX", 0.8726656726111219, 0.3151484225786487]], [["Rotate", 0.17234370554263745, 0.9356543193000078], ["TranslateX", 0.4954374070084091, 0.05496727345849217]], [["Contrast", 0.347405480122842, 0.831553005022885], ["ShearX", 0.28946367213071134, 0.11905898704394013]], [["Rotate", 0.28096672507990683, 0.16181284050307398], ["Color", 0.6554918515385365, 0.8739728050797386]], [["Solarize", 0.05408073374114053, 0.5357087283758337], ["Posterize", 0.42457175211495335, 0.051807130609045515]], [["TranslateY", 0.6216669362331361, 0.9691341207381867], ["Rotate", 0.9833579358130944, 0.12227426932415297]], [["AutoContrast", 0.7572619475282892, 0.8062834082727393], ["Contrast", 0.1447865402875591, 0.40242646573228436]], [["Rotate", 0.7035658783466086, 0.9840285268256428], ["Contrast", 0.04613961510519471, 0.7666683217450163]], [["TranslateX", 0.4580462177951252, 0.6448678609474686], ["AutoContrast", 0.14845695613708987, 0.1581134188537895]], [["Color", 0.06795037145259564, 0.9115552821158709], ["TranslateY", 0.9972953449677655, 0.6791016521791214]], [["Cutout", 0.3586908443690823, 0.11578558293480945], ["Color", 0.49083981719164294, 0.6924851425917189]], [["Brightness", 0.7994717831637873, 0.7887316255321768], ["Posterize", 0.01280463502435425, 0.2799086732858721]], [["ShearY", 0.6733451536131859, 0.8122332639516706], ["AutoContrast", 0.20433889615637357, 0.29023346867819966]], [["TranslateY", 0.709913512385177, 0.6538196931503809], ["Invert", 0.06629795606579203, 0.40913219547548296]], [["Sharpness", 0.4704559834362948, 0.4235993305308414], ["Equalize", 0.7578132044306966, 0.9388824249397175]], [["AutoContrast", 0.5281702802395268, 0.8077253610116979], ["Equalize", 0.856446858814119, 0.0479755681647559]], [["Color", 0.8244145826797791, 0.038409264586238945], ["Equalize", 0.4933123249234237, 0.8251940933672189]], [["TranslateX", 0.23949314158035084, 0.13576027004706692], ["ShearX", 0.8547563771688399, 0.8309262160483606]], [["Cutout", 0.4655680937486001, 0.2819807000622825], ["Contrast", 0.8439552665937905, 0.4843617871587037]], [["TranslateX", 0.19142454476784831, 0.7516148119169537], ["AutoContrast", 0.8677128351329768, 0.34967990912346336]], [["Contrast", 0.2997868299880966, 0.919508054854469], ["AutoContrast", 0.3003418493384957, 0.812314984368542]], [["Invert", 0.1070424236198183, 0.614674386498809], ["TranslateX", 0.5010973510899923, 0.20828478805259465]], [["Contrast", 0.6775882415611454, 0.6938564815591685], ["Cutout", 0.4814634264207498, 0.3086844939744179]], [["TranslateY", 0.939427105020265, 0.02531043619423201], ["Contrast", 0.793754257944812, 0.6676072472565451]], [["Sharpness", 0.09833672397575444, 0.5937214638292085], ["Rotate", 0.32530675291753763, 0.08302275740932441]], [["Sharpness", 0.3096455511562728, 0.6726732004553959], ["TranslateY", 0.43268997648796537, 0.8755012330217743]], [["ShearY", 0.9290771880324833, 0.22114736271319912], ["Equalize", 0.5520199288501478, 0.34269650332060553]], [["AutoContrast", 0.39763980746649374, 0.4597414582725454], ["Contrast", 0.941507852412761, 0.24991270562477041]], [["Contrast", 0.19419400547588095, 0.9127524785329233], ["Invert", 0.40544905179551727, 0.770081532844878]], [["Invert", 0.30473757368608334, 0.23534811781828846], ["Cutout", 0.26090722356706686, 0.5478390909877727]], [["Posterize", 0.49434361308057373, 0.05018423270527428], ["Color", 0.3041910676883317, 0.2603810415446437]], [["Invert", 0.5149061746764011, 0.9507449210221298], ["TranslateY", 0.4458076521892904, 0.8235358255774426]], [["Cutout", 0.7900006753351625, 0.905578861382507], ["Cutout", 0.6707153655762056, 0.8236715672258502]], [["Solarize", 0.8750534386579575, 0.10337670467100568], ["Posterize", 0.6102379615481381, 0.9264503915416868]], [["ShearY", 0.08448689377082852, 0.13981233725811626], ["TranslateX", 0.13979689669329498, 0.768774869872818]], [["TranslateY", 0.35752572266759985, 0.22827299847812488], ["Solarize", 0.3906957174236011, 0.5663314388307709]], [["ShearY", 0.29155240367061563, 0.8427516352971683], ["ShearX", 0.988825367441916, 0.9371258864857649]], [["Posterize", 0.3470780859769458, 0.5467686612321239], ["Rotate", 0.5758606274160093, 0.8843838082656007]], [["Cutout", 0.07825368363221841, 0.3230799425855425], ["Equalize", 0.2319163865298529, 0.42133965674727325]], [["Invert", 0.41972172597448654, 0.34618622513582953], ["ShearX", 0.33638469398198834, 0.9098575535928108]], [["Invert", 0.7322652233340448, 0.7747502957687412], ["Cutout", 0.9643121397298106, 0.7983335094634907]], [["TranslateY", 0.30039942808098496, 0.229018798182827], ["TranslateY", 0.27009499739380194, 0.6435577237846236]], [["Color", 0.38245274994070644, 0.7030758568461645], ["ShearX", 0.4429321461666281, 0.6963787864044149]], [["AutoContrast", 0.8432798685515605, 0.5775214369578088], ["Brightness", 0.7140899735355927, 0.8545854720117658]], [["Rotate", 0.14418935535613786, 0.5637968282213426], ["Color", 0.7115231912479835, 0.32584796564566776]], [["Sharpness", 0.4023501062807533, 0.4162097130412771], ["Brightness", 0.5536372686153666, 0.03004023273348777]], [["TranslateX", 0.7526053265574295, 0.5365938133399961], ["Cutout", 0.07914142706557492, 0.7544953091603148]], [["TranslateY", 0.6932934644882822, 0.5302211727137424], ["Invert", 0.5040606028391255, 0.6074863635108957]], [["Sharpness", 0.5013938602431629, 0.9572417724333157], ["TranslateY", 0.9160516359783026, 0.41798927975391675]], [["ShearY", 0.5130018836722556, 0.30209438428424185], ["Color", 0.15017170588500262, 0.20653495360587826]], [["TranslateX", 0.5293300090022314, 0.6407011888285266], ["Rotate", 0.4809817860439001, 0.3537850070371702]], [["Equalize", 0.42243081336551014, 0.13472721311046565], ["Posterize", 0.4700309639484068, 0.5197704360874883]], [["AutoContrast", 0.40674959899687235, 0.7312824868168921], ["TranslateX", 0.7397527975920833, 0.7068339877944815]], [["TranslateY", 0.5880995184787206, 0.41294111378078946], ["ShearX", 0.3181387627799316, 0.4810010147143413]], [["Color", 0.9898680233928507, 0.13241525577655167], ["Contrast", 0.9824932511238534, 0.5081145010853807]], [["Invert", 0.1591854062582687, 0.9760371953250404], ["Color", 0.9913399302056851, 0.8388709501056177]], [["Rotate", 0.6427451962231163, 0.9486793975292853], ["AutoContrast", 0.8501937877930463, 0.021326757974406196]], [["Contrast", 0.13611684531087598, 0.3050858709483848], ["Posterize", 0.06618644756084646, 0.8776928511951034]], [["TranslateX", 0.41021065663839407, 0.4965319749091702], ["Rotate", 0.07088831484595115, 0.4435516708223345]], [["Sharpness", 0.3151707977154323, 0.28275482520179296], ["Invert", 0.36980384682133804, 0.20813616084536624]], [["Cutout", 0.9979060206661017, 0.39712948644725854], ["Brightness", 0.42451052896163466, 0.942623075649937]], [["Equalize", 0.5300853308425644, 0.010183500830128867], ["AutoContrast", 0.06930788523716991, 0.5403125318991522]], [["Contrast", 0.010385458959237814, 0.2588311035539086], ["ShearY", 0.9347048553928764, 0.10439028366854963]], [["ShearY", 0.9867649486508592, 0.8409258132716434], ["ShearX", 0.48031199530836444, 0.7703375364614137]], [["ShearY", 0.04835889473136512, 0.2671081675890492], ["Brightness", 0.7856432618509617, 0.8032169570159564]], [["Posterize", 0.11112884927351185, 0.7116956530752987], ["TranslateY", 0.7339151092128607, 0.3331241226029017]], [["Invert", 0.13527036207875454, 0.8425980515358883], ["Color", 0.7836395778298139, 0.5517059252678862]], [["Sharpness", 0.012541163521491816, 0.013197550692292892], ["Invert", 0.6295957932861318, 0.43276521236056054]], [["AutoContrast", 0.7681480991225756, 0.3634284648496289], ["Brightness", 0.09708289828517969, 0.45016725043529726]], [["Brightness", 0.5839450499487329, 0.47525965678316795], ["Posterize", 0.43096581990183735, 0.9332382960125196]], [["Contrast", 0.9725334964552795, 0.9142902966863341], ["Contrast", 0.12376116410622995, 0.4355916974126801]], [["TranslateX", 0.8572708473690132, 0.02544522678265526], ["Sharpness", 0.37902120723460364, 0.9606092969833118]], [["TranslateY", 0.8907359001296927, 0.8011363927236099], ["Color", 0.7693777154407178, 0.0936768686746503]], [["Equalize", 0.0002657688243309364, 0.08190798535970034], ["Rotate", 0.5215478065240905, 0.5773519995038368]], [["TranslateY", 0.3383007813932477, 0.5733428274739165], ["Sharpness", 0.2436110797174722, 0.4757790814590501]], [["Cutout", 0.0957402176213592, 0.8914395928996034], ["Cutout", 0.4959915628586883, 0.25890349461645246]], [["AutoContrast", 0.594787300189186, 0.9627455357634459], ["ShearY", 0.5136027621132064, 0.10419602450259002]], [["Solarize", 0.4684077211553732, 0.6592850629431414], ["Sharpness", 0.2382385935956325, 0.6589291408243176]], [["Cutout", 0.4478786947325877, 0.6893616643143388], ["TranslateX", 0.2761781720270474, 0.21750622627277727]], [["Sharpness", 0.39476077929016484, 0.930902796668923], ["Cutout", 0.9073012208742808, 0.9881122386614257]], [["TranslateY", 0.0933719180021565, 0.7206252503441172], ["ShearX", 0.5151400441789256, 0.6307540083648309]], [["AutoContrast", 0.7772689258806401, 0.8159317013156503], ["AutoContrast", 0.5932793713915097, 0.05262217353927168]], [["Equalize", 0.38017352056118914, 0.8084724050448412], ["ShearY", 0.7239725628380852, 0.4246314890359326]], [["Cutout", 0.741157483503503, 0.13244380646497977], ["Invert", 0.03395378056675935, 0.7140036618098844]], [["Rotate", 0.0662727247460636, 0.7099861732415447], ["Rotate", 0.3168532707508249, 0.3553167425022127]], [["AutoContrast", 0.7429303516734129, 0.07117444599776435], ["Posterize", 0.5379537435918104, 0.807221330263993]], [["TranslateY", 0.9788586874795164, 0.7967243851346594], ["Invert", 0.4479103376922362, 0.04260360776727545]], [["Cutout", 0.28318121763188997, 0.7748680701406292], ["AutoContrast", 0.9109258369403016, 0.17126397858002085]], [["Color", 0.30183727885272027, 0.46718354750112456], ["TranslateX", 0.9628952256033627, 0.10269543754135535]], [["AutoContrast", 0.6316709389784041, 0.84287698792044], ["Brightness", 0.5544761629904337, 0.025264772745200004]], [["Rotate", 0.08803313299532567, 0.306059720523696], ["Invert", 0.5222165872425064, 0.045935208620454304]], [["TranslateY", 0.21912346831923835, 0.48529224559004436], ["TranslateY", 0.15466734731903942, 0.8929485418495068]], [["ShearX", 0.17141022847016563, 0.8607600402165531], ["ShearX", 0.6890511341106859, 0.7540899265679949]], [["Invert", 0.9417455522972059, 0.9021733684991224], ["Solarize", 0.7693107057723746, 0.7268007946568782]], [["Posterize", 0.02376991543373752, 0.6768442864453844], ["Rotate", 0.7736875065112697, 0.6706331753139825]], [["Contrast", 0.3623841610390669, 0.15023657344457686], ["Equalize", 0.32975472189318666, 0.05629246869510651]], [["Sharpness", 0.7874882420165824, 0.49535778020457066], ["Posterize", 0.09485578893387558, 0.6170768580482466]], [["Brightness", 0.7099280202949585, 0.021523012961427335], ["Posterize", 0.2076371467666719, 0.17168118578815206]], [["Color", 0.8546367645761538, 0.832011891505731], ["Equalize", 0.6429734783051777, 0.2618995960561532]], [["Rotate", 0.8780793721476224, 0.5920897827664297], ["ShearX", 0.5338303685064825, 0.8605424531336439]], [["Sharpness", 0.7504493806631884, 0.9723552387375258], ["Sharpness", 0.3206385634203266, 0.45127845905824693]], [["ShearX", 0.23794709526711355, 0.06257530645720066], ["Solarize", 0.9132374030587093, 0.6240819934824045]], [["Sharpness", 0.790583587969259, 0.28551171786655405], ["Contrast", 0.39872982844590554, 0.09644706751019538]], [["Equalize", 0.30681999237432944, 0.5645045018157916], ["Posterize", 0.525966242669736, 0.7360106111256014]], [["TranslateX", 0.4881014179825114, 0.6317220208872226], ["ShearX", 0.2935158995550958, 0.23104608987381758]], [["Rotate", 0.49977116738568395, 0.6610761068306319], ["TranslateY", 0.7396566602715687, 0.09386747830045217]], [["ShearY", 0.5909773790018789, 0.16229529902832718], ["Equalize", 0.06461394468918358, 0.6661349001143908]], [["TranslateX", 0.7218443721851834, 0.04435720302810153], ["Cutout", 0.986686540951642, 0.734771197038724]], [["ShearX", 0.5353800096911666, 0.8120139502148365], ["Equalize", 0.4613239578449774, 0.5159528929124512]], [["Color", 0.0871713897628631, 0.7708895183198486], ["Solarize", 0.5811386808912219, 0.35260648120785887]], [["Posterize", 0.3910857927477053, 0.4329219555775561], ["Color", 0.9115983668789468, 0.6043069944145293]], [["Posterize", 0.07493067637060635, 0.4258000066006725], ["AutoContrast", 0.4740957581389772, 0.49069587151651295]], [["Rotate", 0.34086200894268937, 0.9812149332288828], ["Solarize", 0.6801012471371733, 0.17271491146753837]], [["Color", 0.20542270872895207, 0.5532087457727624], ["Contrast", 0.2718692536563381, 0.20313287569510108]], [["Equalize", 0.05199827210980934, 0.0832859890912212], ["AutoContrast", 0.8092395764794107, 0.7778945136511004]], [["Sharpness", 0.1907689513066838, 0.7705754572256907], ["Color", 0.3911178658498049, 0.41791326933095485]], [["Solarize", 0.19611855804748257, 0.2407807485604081], ["AutoContrast", 0.5343964972940493, 0.9034209455548394]], [["Color", 0.43586520148538865, 0.4711164626521439], ["ShearY", 0.28635408186820555, 0.8417816793020271]], [["Cutout", 0.09818482420382535, 0.1649767430954796], ["Cutout", 0.34582392911178494, 0.3927982995799828]], [["ShearX", 0.001253882705272269, 0.48661629027584596], ["Solarize", 0.9229221435457137, 0.44374894836659073]], [["Contrast", 0.6829734655718668, 0.8201750485099037], ["Cutout", 0.7886756837648936, 0.8423285219631946]], [["TranslateY", 0.857017093561528, 0.3038537151773969], ["Invert", 0.12809228606383538, 0.23637166191748027]], [["Solarize", 0.9829027723424164, 0.9723093910674763], ["Color", 0.6346495302126811, 0.5405494753107188]], [["AutoContrast", 0.06868643520377715, 0.23758659417688077], ["AutoContrast", 0.6648225411500879, 0.5618315648260103]], [["Invert", 0.44202305603311676, 0.9945938909685547], ["Equalize", 0.7991650497684454, 0.16014142656347097]], [["AutoContrast", 0.8778631604769588, 0.03951977631894088], ["ShearY", 0.8495160088963707, 0.35771447321250416]], [["Color", 0.5365078341001592, 0.21102444169782308], ["ShearX", 0.7168869678248874, 0.3904298719872734]], [["TranslateX", 0.6517203786101899, 0.6467598990650437], ["Invert", 0.26552491504364517, 0.1210812827294625]], [["Posterize", 0.35196021684368994, 0.8420648319941891], ["Invert", 0.7796829363930631, 0.9520895999240896]], [["Sharpness", 0.7391572148971984, 0.4853940393452846], ["TranslateX", 0.7641915295592839, 0.6351349057666782]], [["Posterize", 0.18485880221115913, 0.6117603277356728], ["Rotate", 0.6541660490605724, 0.5704041108375348]], [["TranslateY", 0.27517423188070533, 0.6610080904072458], ["Contrast", 0.6091250547289317, 0.7702443247557892]], [["Equalize", 0.3611798581067118, 0.6623615672642768], ["TranslateX", 0.9537265090885917, 0.06352772509358584]], [["ShearX", 0.09720029389103535, 0.7800423126320308], ["Invert", 0.30314352455858884, 0.8519925470889914]], [["Brightness", 0.06931529763458055, 0.57760829499712], ["Cutout", 0.637251974467394, 0.7184346129191052]], [["AutoContrast", 0.5026722100286064, 0.32025257156541886], ["Contrast", 0.9667478703047919, 0.14178519432669368]], [["Equalize", 0.5924463845816984, 0.7187610262181517], ["TranslateY", 0.7059479079159405, 0.06551471830655187]], [["Sharpness", 0.18161164512332928, 0.7576138481173385], ["Brightness", 0.19191138767695282, 0.7865880269424701]], [["Brightness", 0.36780861866078696, 0.0677855546737901], ["AutoContrast", 0.8491446654142264, 0.09217782099938121]], [["TranslateY", 0.06011399855120858, 0.8374487034710264], ["TranslateY", 0.8373922962070498, 0.1991295720254297]], [["Posterize", 0.702559916122481, 0.30257509683007755], ["Rotate", 0.249899495398891, 0.9370437251176267]], [["ShearX", 0.9237874098232075, 0.26241907483351146], ["Brightness", 0.7221766836146657, 0.6880749752986671]], [["Cutout", 0.37994098189193104, 0.7836874473657957], ["ShearX", 0.9212861960976824, 0.8140948561570449]], [["Posterize", 0.2584098274786417, 0.7990847652004848], ["Invert", 0.6357731737590063, 0.1066304859116326]], [["Sharpness", 0.4412790857539922, 0.9692465283229825], ["Color", 0.9857401617339051, 0.26755393929808713]], [["Equalize", 0.22348671644912665, 0.7370019910830038], ["Posterize", 0.5396106339575417, 0.5559536849843303]], [["Equalize", 0.8742967663495852, 0.2797122599926307], ["Rotate", 0.4697322053105951, 0.8769872942579476]], [["Sharpness", 0.44279911640509206, 0.07729581896071613], ["Cutout", 0.3589177366154631, 0.2704031551235969]], [["TranslateX", 0.614216412574085, 0.47929659784170453], ["Brightness", 0.6686234118438007, 0.05700784068205689]], [["ShearY", 0.17920614630857634, 0.4699685075827862], ["Color", 0.38251870810870003, 0.7262706923005887]], [["Solarize", 0.4951799001144561, 0.212775278026479], ["TranslateX", 0.8666105646463097, 0.6750496637519537]], [["Color", 0.8110864170849051, 0.5154263861958484], ["Sharpness", 0.2489044083898776, 0.3763372541462343]], [["Cutout", 0.04888193613483871, 0.06041664638981603], ["Color", 0.06438587718683708, 0.5797881428892969]], [["Rotate", 0.032427448352152166, 0.4445797818376559], ["Posterize", 0.4459357828482998, 0.5879865187630777]], [["ShearX", 0.1617179557693058, 0.050796802246318884], ["Cutout", 0.8142465452060423, 0.3836391305618707]], [["TranslateY", 0.1806857249209416, 0.36697730355422675], ["Rotate", 0.9897576550818276, 0.7483432452225264]], [["Brightness", 0.18278016458098223, 0.952352527690299], ["Cutout", 0.3269735224453044, 0.3924869905012752]], [["ShearX", 0.870832707718742, 0.3214743207190739], ["Cutout", 0.6805560681792573, 0.6984188155282459]], [["TranslateX", 0.4157118388833776, 0.3964216288135384], ["TranslateX", 0.3253012682285006, 0.624835513104391]], [["Contrast", 0.7678168037628158, 0.31033802162621793], ["ShearX", 0.27022424855977134, 0.3773245605126201]], [["TranslateX", 0.37812621869017593, 0.7657993810740699], ["Rotate", 0.18081890120092914, 0.8893511219618171]], [["Posterize", 0.8735859716088367, 0.18243793043074286], ["TranslateX", 0.90435994250313, 0.24116383818819453]], [["Invert", 0.06666709253664793, 0.3881076083593933], ["TranslateX", 0.3783333964963522, 0.14411014979589543]], [["Equalize", 0.8741147867162096, 0.14203839235846816], ["TranslateX", 0.7801536758037405, 0.6952401607812743]], [["Cutout", 0.6095335117944475, 0.5679026063718094], ["Posterize", 0.06433868172233115, 0.07139559616012303]], [["TranslateY", 0.3020364047315408, 0.21459810361176246], ["Cutout", 0.7097677414888889, 0.2942144632587549]], [["Brightness", 0.8223662419048653, 0.195700694016108], ["Invert", 0.09345407040803999, 0.779843655582099]], [["TranslateY", 0.7353462929356228, 0.0468520680237382], ["Cutout", 0.36530918247940425, 0.3897292909049672]], [["Invert", 0.9676896451721213, 0.24473302189463453], ["Invert", 0.7369271521408992, 0.8193267003356975]], [["Sharpness", 0.8691871972054326, 0.4441713912682772], ["ShearY", 0.47385584832119887, 0.23521684584675429]], [["ShearY", 0.9266946026184021, 0.7611986713358834], ["TranslateX", 0.6195820760253926, 0.14661428669483678]], [["Sharpness", 0.08470870576026868, 0.3380219099907229], ["TranslateX", 0.3062343307496658, 0.7135777338095889]], [["Sharpness", 0.5246448204194909, 0.3193061215236702], ["ShearX", 0.8160637208508432, 0.9720697396582731]], [["Posterize", 0.5249259956549405, 0.3492042382504774], ["Invert", 0.8183138799547441, 0.11107271762524618]], [["TranslateY", 0.210869733350744, 0.7138905840721885], ["Sharpness", 0.7773226404450125, 0.8005353621959782]], [["Posterize", 0.33067522385556025, 0.32046239220630124], ["AutoContrast", 0.18918147708798405, 0.4646281070474484]], [["TranslateX", 0.929502026131094, 0.8029128121556285], ["Invert", 0.7319794306118105, 0.5421878712623392]], [["ShearX", 0.25645940834182723, 0.42754710760160963], ["ShearX", 0.44640695310173306, 0.8132185532296811]], [["Color", 0.018436846416536312, 0.8439313862001113], ["Sharpness", 0.3722867661453415, 0.5103570873163251]], [["TranslateX", 0.7285989086776543, 0.4809027697099264], ["TranslateY", 0.9740807004893643, 0.8241085438636939]], [["Posterize", 0.8721868989693397, 0.5700907310383815], ["Posterize", 0.4219074410577852, 0.8032643572845402]], [["Contrast", 0.9811380092558266, 0.8498397471632105], ["Sharpness", 0.8380884329421594, 0.18351306571903125]], [["TranslateY", 0.3878939366762001, 0.4699103438753077], ["Invert", 0.6055556353233807, 0.8774727658400134]], [["TranslateY", 0.052317005261018346, 0.39471450378745787], ["ShearX", 0.8612486845942395, 0.28834103278807466]], [["Color", 0.511993351208063, 0.07251427040525904], ["Solarize", 0.9898097047354855, 0.299761565689576]], [["Equalize", 0.2721248231619904, 0.6870975927455507], ["Cutout", 0.8787327242363994, 0.06228061428917098]], [["Invert", 0.8931880335225408, 0.49720931867378193], ["Posterize", 0.9619698792159256, 0.17859639696940088]], [["Posterize", 0.0061688075074411985, 0.08082938731035938], ["Brightness", 0.27745128028826993, 0.8638528796903816]], [["ShearY", 0.9140200609222026, 0.8240421430867707], ["Invert", 0.651734417415332, 0.08871906369930926]], [["Color", 0.45585010413511196, 0.44705070078574316], ["Color", 0.26394624901633146, 0.11242877788650807]], [["ShearY", 0.9200278466372522, 0.2995901331149652], ["Cutout", 0.8445407215116278, 0.7410524214287446]], [["ShearY", 0.9950483746990132, 0.112964468262847], ["ShearY", 0.4118332303218585, 0.44839613407553636]], [["Contrast", 0.7905821952255192, 0.23360046159385106], ["Posterize", 0.8611787233956044, 0.8984260048943528]], [["TranslateY", 0.21448061359312853, 0.8228112806838331], ["Contrast", 0.8992297266152983, 0.9179231590570998]], [["Invert", 0.3924194798946006, 0.31830516468371495], ["Rotate", 0.8399556845248508, 0.3764892022932781]], [["Cutout", 0.7037916990046816, 0.9214620769502728], ["AutoContrast", 0.02913794613018239, 0.07808607528954048]], [["ShearY", 0.6041490474263381, 0.6094184590800105], ["Equalize", 0.2932954517354919, 0.5840888946081727]], [["ShearX", 0.6056801676269449, 0.6948580442549543], ["Cutout", 0.3028001021044615, 0.15117101733894078]], [["Brightness", 0.8011486803860253, 0.18864079729374195], ["Solarize", 0.014965327213230961, 0.8842620292527029]], [["Invert", 0.902244007904273, 0.5634673798052033], ["Equalize", 0.13422913507398349, 0.4110956745883727]], [["TranslateY", 0.9981773319103838, 0.09568550987216096], ["Color", 0.7627662124105109, 0.8494409737419493]], [["Cutout", 0.3013527640416782, 0.03377226729898486], ["ShearX", 0.5727964831614619, 0.8784196638222834]], [["TranslateX", 0.6050722426803684, 0.3650103962378708], ["TranslateX", 0.8392084589130886, 0.6479816470292911]], [["Rotate", 0.5032806606500023, 0.09276980118866307], ["TranslateY", 0.7800234515261191, 0.18896454379343308]], [["Invert", 0.9266027256244017, 0.8246111062199752], ["Contrast", 0.12112023357797697, 0.33870762271759436]], [["Brightness", 0.8688784756993134, 0.17263759696106606], ["ShearX", 0.5133700431071326, 0.6686811994542494]], [["Invert", 0.8347840440941976, 0.03774897445901726], ["Brightness", 0.24925057499276548, 0.04293631677355758]], [["Color", 0.5998145279485104, 0.4820093200092529], ["TranslateY", 0.6709586184077769, 0.07377334081382858]], [["AutoContrast", 0.7898846202957984, 0.325293526672498], ["Contrast", 0.5156435596826767, 0.2889223168660645]], [["ShearX", 0.08147389674998307, 0.7978924681113669], ["Contrast", 0.7270003309106291, 0.009571215234092656]], [["Sharpness", 0.417607614440786, 0.9532566433338661], ["Posterize", 0.7186586546796782, 0.6936509907073302]], [["ShearX", 0.9555300215926675, 0.1399385550263872], ["Color", 0.9981041061848231, 0.5037462398323248]], [["Equalize", 0.8003487831375474, 0.5413759363796945], ["ShearY", 0.0026607045117773565, 0.019262273030984933]], [["TranslateY", 0.04845391502469176, 0.10063445212118283], ["Cutout", 0.8273170186786745, 0.5045257728554577]], [["TranslateX", 0.9690985344978033, 0.505202991815533], ["TranslateY", 0.7255326592928096, 0.02103609500701631]], [["Solarize", 0.4030771176836736, 0.8424237871457034], ["Cutout", 0.28705805963928965, 0.9601617893682582]], [["Sharpness", 0.16865290353070606, 0.6899673563468826], ["Posterize", 0.3985430034869616, 0.6540651997730774]], [["ShearY", 0.21395578485362032, 0.09519358818949009], ["Solarize", 0.6692821708524135, 0.6462523623552485]], [["AutoContrast", 0.912360598054091, 0.029800239085051583], ["Invert", 0.04319256403746308, 0.7712501517098587]], [["ShearY", 0.9081969961839055, 0.4581560239984739], ["AutoContrast", 0.5313894814729159, 0.5508393335751848]], [["ShearY", 0.860528568424097, 0.8196987216301588], ["Posterize", 0.41134650331494205, 0.3686632018978778]], [["AutoContrast", 0.8753670810078598, 0.3679438326304749], ["Invert", 0.010444228965415858, 0.9581244779208277]], [["Equalize", 0.07071836206680682, 0.7173594756186462], ["Brightness", 0.06111434312497388, 0.16175064669049277]], [["AutoContrast", 0.10522219073562122, 0.9768776621069855], ["TranslateY", 0.2744795945215529, 0.8577967957127298]], [["AutoContrast", 0.7628146493166175, 0.996157376418147], ["Contrast", 0.9255565598518469, 0.6826126662976868]], [["TranslateX", 0.017225816199011312, 0.2470332491402908], ["Solarize", 0.44048494909493807, 0.4492422515972162]], [["ShearY", 0.38885252627795064, 0.10272256704901939], ["Equalize", 0.686154959829183, 0.8973517148655337]], [["Rotate", 0.29628991573592967, 0.16639926575004715], ["ShearX", 0.9013782324726413, 0.0838318162771563]], [["Color", 0.04968391374688563, 0.6138600739645352], ["Invert", 0.11177127838716283, 0.10650198522261578]], [["Invert", 0.49655016367624016, 0.8603374164829688], ["ShearY", 0.40625439617553727, 0.4516437918820778]], [["TranslateX", 0.15015718916062992, 0.13867777502116208], ["Brightness", 0.3374464418810188, 0.7613355669536931]], [["Invert", 0.644644393321966, 0.19005804481199562], ["AutoContrast", 0.2293259789431853, 0.30335723256340186]], [["Solarize", 0.004968793254801596, 0.5370892072646645], ["Contrast", 0.9136902637865596, 0.9510587477779084]], [["Rotate", 0.38991518440867123, 0.24796987467455756], ["Sharpness", 0.9911180315669776, 0.5265657122981591]], [["Solarize", 0.3919646484436238, 0.6814994037194909], ["Sharpness", 0.4920838987787103, 0.023425724294012018]], [["TranslateX", 0.25107587874378867, 0.5414936560189212], ["Cutout", 0.7932919623814599, 0.9891303444820169]], [["Brightness", 0.07863012174272999, 0.045175652208389594], ["Solarize", 0.889609658064552, 0.8228793315963948]], [["Cutout", 0.20477096178169596, 0.6535063675027364], ["ShearX", 0.9216318577173639, 0.2908690977359947]], [["Contrast", 0.7035118947423187, 0.45982709058312454], ["Contrast", 0.7130268070749464, 0.8635123354235471]], [["Sharpness", 0.26319477541228997, 0.7451278726847078], ["Rotate", 0.8170499362173754, 0.13998593411788207]], [["Rotate", 0.8699365715164192, 0.8878057721750832], ["Equalize", 0.06682350555715044, 0.7164702080630689]], [["ShearY", 0.3137466057521987, 0.6747433496011368], ["Rotate", 0.42118828936218133, 0.980121180104441]], [["Solarize", 0.8470375049950615, 0.15287589264139223], ["Cutout", 0.14438435054693055, 0.24296463267973512]], [["TranslateY", 0.08822241792224905, 0.36163911974799356], ["TranslateY", 0.11729726813270003, 0.6230889726445291]], [["ShearX", 0.7720112337718541, 0.2773292905760122], ["Sharpness", 0.756290929398613, 0.27830353710507705]], [["Color", 0.33825031007968287, 0.4657590047522816], ["ShearY", 0.3566628994713067, 0.859750504071925]], [["TranslateY", 0.06830147433378053, 0.9348778582086664], ["TranslateX", 0.15509346516378553, 0.26320778885339435]], [["Posterize", 0.20266751150740858, 0.008351463842578233], ["Sharpness", 0.06506971109417259, 0.7294471760284555]], [["TranslateY", 0.6278911394418829, 0.8702181892620695], ["Invert", 0.9367073860264247, 0.9219230428944211]], [["Sharpness", 0.1553425337673321, 0.17601557714491345], ["Solarize", 0.7040449681338888, 0.08764313147327729]], [["Equalize", 0.6082233904624664, 0.4177428549911376], ["AutoContrast", 0.04987405274618151, 0.34516208204700916]], [["Brightness", 0.9616085936167699, 0.14561237331885468], ["Solarize", 0.8927707736296572, 0.31176907850205704]], [["Brightness", 0.6707778304730988, 0.9046457117525516], ["Brightness", 0.6801448953060988, 0.20015313057149042]], [["Color", 0.8292680845499386, 0.5181603879593888], ["Brightness", 0.08549161770369762, 0.6567870536463203]], [["ShearY", 0.267802208078051, 0.8388133819588173], ["Sharpness", 0.13453409120796123, 0.10028351311149486]], [["Posterize", 0.775796593610272, 0.05359034561289766], ["Cutout", 0.5067360625733027, 0.054451986840317934]], [["TranslateX", 0.5845238647690084, 0.7507147553486293], ["Brightness", 0.2642051786121197, 0.2578358927056452]], [["Cutout", 0.10787517610922692, 0.8147986902794228], ["Contrast", 0.2190149206329539, 0.902210615462459]], [["TranslateX", 0.5663614214181296, 0.05309965916414028], ["ShearX", 0.9682797885154938, 0.41791929533938466]], [["ShearX", 0.2345325577621098, 0.383780128037189], ["TranslateX", 0.7298083748149163, 0.644325797667087]], [["Posterize", 0.5138725709682734, 0.7901809917259563], ["AutoContrast", 0.7966018627776853, 0.14529337543427345]], [["Invert", 0.5973031989249785, 0.417399314592829], ["Solarize", 0.9147539948653116, 0.8221272315548086]], [["Posterize", 0.601596043336383, 0.18969646160963938], ["Color", 0.7527275484079655, 0.431793831326888]], [["Equalize", 0.6731483454430538, 0.7866786558207602], ["TranslateX", 0.97574396899191, 0.5970255778044692]], [["Cutout", 0.15919495850169718, 0.8916094305850562], ["Invert", 0.8351348834751027, 0.4029937360314928]], [["Invert", 0.5894085405226027, 0.7283806854157764], ["Brightness", 0.3973976860470554, 0.949681121498567]], [["AutoContrast", 0.3707914135327408, 0.21192068592079616], ["ShearX", 0.28040127351140676, 0.6754553511344856]], [["Solarize", 0.07955132378694896, 0.15073572961927306], ["ShearY", 0.5735850168851625, 0.27147326850217746]], [["Equalize", 0.678653949549764, 0.8097796067861455], ["Contrast", 0.2283048527510083, 0.15507804874474185]], [["Equalize", 0.286013868374536, 0.186785848694501], ["Posterize", 0.16319021740810458, 0.1201304443285659]], [["Sharpness", 0.9601590830563757, 0.06267915026513238], ["AutoContrast", 0.3813920685124327, 0.294224403296912]], [["Brightness", 0.2703246632402241, 0.9168405377492277], ["ShearX", 0.6156009855831097, 0.4955986055846403]], [["Color", 0.9065504424987322, 0.03393612216080133], ["ShearY", 0.6768595880405884, 0.9981068127818191]], [["Equalize", 0.28812842368483904, 0.300387487349145], ["ShearY", 0.28812248704858345, 0.27105076231533964]], [["Brightness", 0.6864882730513477, 0.8205553299102412], ["Cutout", 0.45995236371265424, 0.5422030370297759]], [["Color", 0.34941404877084326, 0.25857961830158516], ["AutoContrast", 0.3451390878441899, 0.5000938249040454]], [["Invert", 0.8268247541815854, 0.6691380821226468], ["Cutout", 0.46489193601530476, 0.22620873109485895]], [["Rotate", 0.17879730528062376, 0.22670425330593935], ["Sharpness", 0.8692795688221834, 0.36586055020855723]], [["Brightness", 0.31203975139659634, 0.6934046293010939], ["Cutout", 0.31649437872271236, 0.08078625004157935]], [["Cutout", 0.3119482836150119, 0.6397160035509996], ["Contrast", 0.8311248624784223, 0.22897510169718616]], [["TranslateX", 0.7631157841429582, 0.6482890521284557], ["Brightness", 0.12681196272427664, 0.3669813784257344]], [["TranslateX", 0.06027722649179801, 0.3101104512201861], ["Sharpness", 0.5652076706249394, 0.05210008400968136]], [["AutoContrast", 0.39213552101583127, 0.5047021194355596], ["ShearY", 0.7164003055682187, 0.8063370761002899]], [["Solarize", 0.9574307011238342, 0.21472064809226854], ["AutoContrast", 0.8102612285047174, 0.716870148067014]], [["Rotate", 0.3592634277567387, 0.6452602893051465], ["AutoContrast", 0.27188430331411506, 0.06003099168464854]], [["Cutout", 0.9529536554825503, 0.5285505311027461], ["Solarize", 0.08478231903311029, 0.15986449762728216]], [["TranslateY", 0.31176130458018936, 0.5642853506158253], ["Equalize", 0.008890883901317648, 0.5146121040955942]], [["Color", 0.40773645085566157, 0.7110398926612682], ["Color", 0.18233100156439364, 0.7830036002758337]], [["Posterize", 0.5793809197821732, 0.043748553135581236], ["Invert", 0.4479962016131668, 0.7349663010359488]], [["TranslateX", 0.1994882312299382, 0.05216859488899439], ["Rotate", 0.48288726352035416, 0.44713829026777585]], [["Posterize", 0.22122838185154603, 0.5034546841241283], ["TranslateX", 0.2538745835410222, 0.6129055170893385]], [["Color", 0.6786559960640814, 0.4529749369803212], ["Equalize", 0.30215879674415336, 0.8733394611096772]], [["Contrast", 0.47316062430673456, 0.46669538897311447], ["Invert", 0.6514906551984854, 0.3053339444067804]], [["Equalize", 0.6443202625334524, 0.8689731394616441], ["Color", 0.7549183794057628, 0.8889001426329578]], [["Solarize", 0.616709740662654, 0.7792180816399313], ["ShearX", 0.9659155537406062, 0.39436937531179495]], [["Equalize", 0.23694011299406226, 0.027711152164392128], ["TranslateY", 0.1677339686527083, 0.3482126536808231]], [["Solarize", 0.15234175951790285, 0.7893840414281341], ["TranslateX", 0.2396395768284183, 0.27727219214979715]], [["Contrast", 0.3792017455380605, 0.32323660409845334], ["Contrast", 0.1356037413846466, 0.9127772969992305]], [["ShearX", 0.02642732222284716, 0.9184662576502115], ["Equalize", 0.11504884472142995, 0.8957638893097964]], [["TranslateY", 0.3193812913345325, 0.8828100030493128], ["ShearY", 0.9374975727563528, 0.09909415611083694]], [["AutoContrast", 0.025840721736048122, 0.7941037581373024], ["TranslateY", 0.498518003323313, 0.5777122846572548]], [["ShearY", 0.6042199307830248, 0.44809668754508836], ["Cutout", 0.3243978207701482, 0.9379740926294765]], [["ShearY", 0.6858549297583574, 0.9993252035788924], ["Sharpness", 0.04682428732773203, 0.21698099707915652]], [["ShearY", 0.7737469436637263, 0.8810127181224531], ["ShearY", 0.8995655445246451, 0.4312416220354539]], [["TranslateY", 0.4953094136709374, 0.8144161580138571], ["Solarize", 0.26301211718928097, 0.518345311180405]], [["Brightness", 0.8820246486031275, 0.571075863786249], ["ShearX", 0.8586669146703955, 0.0060476383595142735]], [["Sharpness", 0.20519233710982254, 0.6144574759149729], ["Posterize", 0.07976625267460813, 0.7480145046726968]], [["ShearY", 0.374075419680195, 0.3386105402023202], ["ShearX", 0.8228083637082115, 0.5885174783155361]], [["Brightness", 0.3528780713814561, 0.6999884884306623], ["Sharpness", 0.3680348120526238, 0.16953358258959617]], [["Brightness", 0.24891223104442084, 0.7973853494920095], ["TranslateX", 0.004256803835524736, 0.0470216343108546]], [["Posterize", 0.1947344282646012, 0.7694802711054367], ["Cutout", 0.9594385534844785, 0.5469744140592429]], [["Invert", 0.19012504762806026, 0.7816140211434693], ["TranslateY", 0.17479746932338402, 0.024249345245078602]], [["Rotate", 0.9669262055946796, 0.510166180775991], ["TranslateX", 0.8990602034610352, 0.6657802719304693]], [["ShearY", 0.5453049050407278, 0.8476872739603525], ["Cutout", 0.14226529093962592, 0.15756960661106634]], [["Equalize", 0.5895291156113004, 0.6797218994447763], ["TranslateY", 0.3541442192192753, 0.05166001155849864]], [["Equalize", 0.39530681662726097, 0.8448335365081087], ["Brightness", 0.6785483272734143, 0.8805568647038574]], [["Cutout", 0.28633258271917905, 0.7750870268336066], ["Equalize", 0.7221097824537182, 0.5865506280531162]], [["Posterize", 0.9044429629421187, 0.4620266401793388], ["Invert", 0.1803008045494473, 0.8073190766288534]], [["Sharpness", 0.7054649148075851, 0.3877207948962055], ["TranslateX", 0.49260224225927285, 0.8987462620731029]], [["Sharpness", 0.11196934729294483, 0.5953704422694938], ["Contrast", 0.13969334315069737, 0.19310569898434204]], [["Posterize", 0.5484346101051778, 0.7914140118600685], ["Brightness", 0.6428044691630473, 0.18811316670808076]], [["Invert", 0.22294834094984717, 0.05173157689962704], ["Cutout", 0.6091129168510456, 0.6280845506243643]], [["AutoContrast", 0.5726444076195267, 0.2799840903601295], ["Cutout", 0.3055752727786235, 0.591639807512993]], [["Brightness", 0.3707116723204462, 0.4049175910826627], ["Rotate", 0.4811601625588309, 0.2710760253723644]], [["ShearY", 0.627791719653608, 0.6877498291550205], ["TranslateX", 0.8751753308366824, 0.011164650018719358]], [["Posterize", 0.33832547954522263, 0.7087039872581657], ["Posterize", 0.6247474435007484, 0.7707784192114796]], [["Contrast", 0.17620186308493468, 0.9946224854942095], ["Solarize", 0.5431896088395964, 0.5867904203742308]], [["ShearX", 0.4667959516719652, 0.8938082224109446], ["TranslateY", 0.7311343008292865, 0.6829842246020277]], [["ShearX", 0.6130281467237769, 0.9924010909612302], ["Brightness", 0.41039241699696916, 0.9753218875311392]], [["TranslateY", 0.0747250386427123, 0.34602725521067534], ["Rotate", 0.5902597465515901, 0.361094672021087]], [["Invert", 0.05234890878959486, 0.36914978664919407], ["Sharpness", 0.42140532878231374, 0.19204058551048275]], [["ShearY", 0.11590485361909497, 0.6518540857972316], ["Invert", 0.6482444740361704, 0.48256237896163945]], [["Rotate", 0.4931329446923608, 0.037076242417301675], ["Contrast", 0.9097939772412852, 0.5619594905306389]], [["Posterize", 0.7311032479626216, 0.4796364593912915], ["Color", 0.13912123993932402, 0.03997286439663705]], [["AutoContrast", 0.6196602944085344, 0.2531430457527588], ["Rotate", 0.5583937060431972, 0.9893379795224023]], [["AutoContrast", 0.8847753125072959, 0.19123028952580057], ["TranslateY", 0.494361716097206, 0.14232297727461696]], [["Invert", 0.6212360716340707, 0.033898871473033165], ["AutoContrast", 0.30839896957008295, 0.23603569542166247]], [["Equalize", 0.8255583546605049, 0.613736933157845], ["AutoContrast", 0.6357166629525485, 0.7894617347709095]], [["Brightness", 0.33840706322846814, 0.07917167871493658], ["ShearY", 0.15693175752528676, 0.6282773652129153]], [["Cutout", 0.7550520024859294, 0.08982367300605598], ["ShearX", 0.5844942417320858, 0.36051195083380105]]]
    return p

def fa_reduced_svhn():
    p = [[["TranslateX", 0.001576965129744562, 0.43180488809874773], ["Invert", 0.7395307279252639, 0.7538444307982558]], [["Contrast", 0.5762062225409211, 0.7532431872873473], ["TranslateX", 0.45212523461624615, 0.02451684483019846]], [["Contrast", 0.18962433143225088, 0.29481185671147325], ["Contrast", 0.9998112218299271, 0.813015355163255]], [["Posterize", 0.9633391295905683, 0.4136786222304747], ["TranslateY", 0.8011655496664203, 0.44102126789970797]], [["Color", 0.8231185187716968, 0.4171602946893402], ["TranslateX", 0.8684965619113907, 0.36514568324909674]], [["Color", 0.904075230324581, 0.46319140331093767], ["Contrast", 0.4115196534764559, 0.7773329158740563]], [["Sharpness", 0.6600262774093967, 0.8045637700026345], ["TranslateY", 0.5917663766021198, 0.6844241908520602]], [["AutoContrast", 0.16223989311434306, 0.48169653554195924], ["ShearX", 0.5433173232860344, 0.7460278151912152]], [["ShearX", 0.4913604762760715, 0.83391837859561], ["Color", 0.5580367056511908, 0.2961512691312932]], [["Color", 0.18567091721211237, 0.9296983204905286], ["Cutout", 0.6074026199060156, 0.03303273406448193]], [["Invert", 0.8049054771963224, 0.1340792344927909], ["Color", 0.4208839940504979, 0.7096454840962345]], [["ShearX", 0.7997786664546294, 0.6492629575700173], ["AutoContrast", 0.3142777134084793, 0.6526010594925064]], [["TranslateX", 0.2581027144644976, 0.6997433332894101], ["Rotate", 0.45490480973606834, 0.238620570022944]], [["Solarize", 0.837397161027719, 0.9311141273136286], ["Contrast", 0.640364826293148, 0.6299761518677469]], [["Brightness", 0.3782457347141744, 0.7085036717054278], ["Brightness", 0.5346150083208507, 0.5858930737867671]], [["Invert", 0.48780391510474086, 0.610086407879722], ["Color", 0.5601999247616932, 0.5393836220423195]], [["Brightness", 0.00250086643283564, 0.5003355864896979], ["Brightness", 0.003922153283353616, 0.41107110154584925]], [["TranslateX", 0.4073069009685957, 0.9843435292693372], ["Invert", 0.38837085318721926, 0.9298542033875989]], [["ShearY", 0.05479740443795811, 0.9113983424872698], ["AutoContrast", 0.2181108114232728, 0.713996037012164]], [["Brightness", 0.27747508429413903, 0.3217467607288693], ["ShearX", 0.02715239061946995, 0.5430731635396449]], [["Sharpness", 0.08994432959374538, 0.004706443546453831], ["Posterize", 0.10768206853226996, 0.39020299239900236]], [["Cutout", 0.37498679037853905, 0.20784809761469553], ["Color", 0.9825516352194511, 0.7654155662756019]], [["Color", 0.8899349124453552, 0.7797700766409008], ["Rotate", 0.1370222187174981, 0.2622119295138398]], [["Cutout", 0.7088223332663685, 0.7884456023190028], ["Solarize", 0.5362257505160836, 0.6426837537811545]], [["Invert", 0.15686225694987552, 0.5500563899117913], ["Rotate", 0.16315224193260078, 0.4246854030170752]], [["Rotate", 0.005266247922433631, 0.06612026206223394], ["Contrast", 0.06494357829209037, 0.2738420319474947]], [["Cutout", 0.30200619566806275, 0.06558008068236942], ["Rotate", 0.2168576483823022, 0.878645566986328]], [["Color", 0.6358930679444622, 0.613404714161498], ["Rotate", 0.08733206733004326, 0.4348276574435751]], [["Cutout", 0.8834634887239585, 0.0006853845293474659], ["Solarize", 0.38132051231951847, 0.42558752668491195]], [["ShearY", 0.08830136548479937, 0.5522438878371283], ["Brightness", 0.23816560427834074, 0.3033709051157141]], [["Solarize", 0.9015331490756151, 0.9108788708847556], ["Contrast", 0.2057898014670072, 0.03260096030427456]], [["Equalize", 0.9455978685121174, 0.14850077333434056], ["TranslateY", 0.6888705996522545, 0.5300565492007543]], [["Cutout", 0.16942673959343585, 0.7294197201361826], ["TranslateX", 0.41184830642301534, 0.7060207449376135]], [["Color", 0.30133344118702166, 0.24384417956342314], ["Sharpness", 0.4640904544421743, 0.32431840288061864]], [["Sharpness", 0.5195055033472676, 0.9386677467005835], ["Color", 0.9536519432978372, 0.9624043444556467]], [["Rotate", 0.8689597230556101, 0.23955490826730633], ["Contrast", 0.050071600927462656, 0.1309891556004179]], [["Cutout", 0.5349421090878962, 0.08239510727779054], ["Rotate", 0.46064964710717216, 0.9037689320897339]], [["AutoContrast", 0.5625256909986802, 0.5358003783186498], ["Equalize", 0.09204330691163354, 0.4386906784850649]], [["ShearX", 0.0011061172864470226, 0.07150284682189278], ["AutoContrast", 0.6015956946553209, 0.4375362295530898]], [["ShearY", 0.25294276499800983, 0.7937560397859562], ["Brightness", 0.30834103299704474, 0.21960258701547009]], [["Posterize", 0.7423948904688074, 0.4598609935109695], ["Rotate", 0.5510348811675979, 0.26763724868985933]], [["TranslateY", 0.3208729319318745, 0.945513054853888], ["ShearX", 0.4916473963030882, 0.8743840560039451]], [["ShearY", 0.7557718687011286, 0.3125397104722828], ["Cutout", 0.5565359791865849, 0.5151359251135629]], [["AutoContrast", 0.16652786355571275, 0.1101575800958632], ["Rotate", 0.05108851703032641, 0.2612966401802814]], [["Brightness", 0.380296489835016, 0.0428162454174662], ["ShearX", 0.3911934083168285, 0.18933607362790178]], [["Color", 0.002476250465397678, 0.07795275305347571], ["Posterize", 0.08131841266654188, 0.14843363184306413]], [["Cutout", 0.36664558716104434, 0.20904484995063996], ["Cutout", 0.07986452057223141, 0.9287747671053432]], [["Color", 0.9296812469919231, 0.6634239915141935], ["Rotate", 0.07632463573240006, 0.408624029443747]], [["Cutout", 0.7594470171961278, 0.9834672124229463], ["Solarize", 0.4471371303745053, 0.5751101102286562]], [["Posterize", 0.051186719734032285, 0.5110941294710823], ["Sharpness", 0.040432522797391596, 0.42652298706992164]], [["Sharpness", 0.2645335264327221, 0.8844553189835457], ["Brightness", 0.7229600357932696, 0.16660749270785696]], [["Sharpness", 0.6296376086802589, 0.15564989758083458], ["Sharpness", 0.7913410481400365, 0.7022615408082826]], [["Cutout", 0.5517247347343883, 0.43794888517764674], ["ShearX", 0.6951051782530201, 0.6230992857867065]], [["ShearX", 0.9015708556331022, 0.6322135168527783], ["Contrast", 0.4285629283441831, 0.18158321019502988]], [["Brightness", 0.9014292329524769, 0.3660463325457713], ["Invert", 0.6700729097206592, 0.16502732071917703]], [["AutoContrast", 0.6432764477303431, 0.9998909112400834], ["Invert", 0.8124063975545761, 0.8149683327882365]], [["Cutout", 0.6023944009428617, 0.9630976951918225], ["ShearX", 0.2734723568803071, 0.3080911542121765]], [["Sharpness", 0.048949115014412806, 0.44497866256845164], ["Brightness", 0.5611832867244329, 0.12994217480426257]], [["TranslateY", 0.4619112333002525, 0.47317728091588396], ["Solarize", 0.618638784910472, 0.9508297099190338]], [["Sharpness", 0.9656274391147018, 0.3402622993963962], ["Cutout", 0.8452511174508919, 0.3094717093312621]], [["ShearX", 0.04942201651478659, 0.6910568465705691], ["AutoContrast", 0.7155342517619936, 0.8565418847743523]], [["Brightness", 0.5222290590721783, 0.6462675303633422], ["Sharpness", 0.7756317511341633, 0.05010730683866704]], [["Contrast", 0.17098396012942796, 0.9128908626236187], ["TranslateY", 0.1523815376677518, 0.4269909829886339]], [["Cutout", 0.7679024720089866, 0.22229116396644455], ["Sharpness", 0.47714827844878843, 0.8242815864830401]], [["Brightness", 0.9321772357292445, 0.11339758604001371], ["Invert", 0.7021078495093375, 0.27507749184928154]], [["ShearY", 0.7069449324510433, 0.07262757954730437], ["Cutout", 0.6298690227159313, 0.8866813664859028]], [["ShearX", 0.8153137620199989, 0.8478194179953927], ["ShearX", 0.7519451353411938, 0.3914579556959725]], [["Cutout", 0.07152574469472753, 0.2629935229222503], ["TranslateX", 0.43728405510089485, 0.2610201002449789]], [["AutoContrast", 0.5824529633013098, 0.5619551536261955], ["Rotate", 0.45434137552116965, 0.7567169855140041]], [["TranslateY", 0.9338431187142137, 0.14230481341042783], ["Cutout", 0.744797723251028, 0.4346601666787713]], [["ShearX", 0.3197252560289169, 0.8770408070016171], ["Color", 0.7657013088540465, 0.2685586719812284]], [["ShearY", 0.6542181749801549, 0.8148188744344297], ["Sharpness", 0.5108985661436543, 0.9926016115463769]], [["ShearY", 0.39218730620135694, 0.857769946478945], ["Color", 0.39588355914920886, 0.9910530523789284]], [["Invert", 0.4993610396803735, 0.08449723470758526], ["TranslateX", 0.46267456928508305, 0.46691125646493964]], [["Equalize", 0.8640576819821256, 0.3973808869887604], ["ShearY", 0.5491163877063172, 0.422429328786161]], [["Contrast", 0.6146206387722841, 0.8453559854684094], ["TranslateX", 0.7974333014574718, 0.47395476786951773]], [["Contrast", 0.6828704722015236, 0.6952755697785722], ["Brightness", 0.7903069452567497, 0.8350915035109574]], [["Rotate", 0.1211091761531299, 0.9667702562228727], ["Color", 0.47888534537103344, 0.8298620028065332]], [["Equalize", 0.20009722872711086, 0.21851235854853018], ["Invert", 0.4433641154198673, 0.41902203581091935]], [["AutoContrast", 0.6333190204577053, 0.23965630032835372], ["Color", 0.38651217030044804, 0.06447323778198723]], [["Brightness", 0.378274337541471, 0.5482593116308322], ["Cutout", 0.4856574442608347, 0.8889688535495244]], [["Rotate", 0.8201259323479384, 0.7404525573938633], ["Color", 0.28371236449364595, 0.7866003515933161]], [["Brightness", 0.10053196350009105, 0.18814037089411267], ["Sharpness", 0.5572102497672569, 0.04458217557977126]], [["AutoContrast", 0.6445330112376135, 0.48082049184921843], ["TranslateY", 0.378898917914949, 0.9338102625289362]], [["AutoContrast", 0.08482623401924708, 0.25199930695784384], ["Solarize", 0.5981823550521426, 0.19626357596662092]], [["Solarize", 0.4373030803918095, 0.22907881245285625], ["AutoContrast", 0.6383084635487905, 0.29517603235993883]], [["AutoContrast", 0.922112624726991, 0.29398098144910145], ["AutoContrast", 0.8550184811514672, 0.8030331582292343]], [["ShearX", 0.38761582800913896, 0.06304125015084923], ["Contrast", 0.3225758804984975, 0.7089696696094797]], [["TranslateY", 0.27499498563849206, 0.1917583097241206], ["Color", 0.5845853711746438, 0.5353520071667661]], [["ShearY", 0.530881951424285, 0.47961248148116453], ["ShearX", 0.04666387744533289, 0.275772822690165]], [["Solarize", 0.5727309318844802, 0.02889734544563341], ["AutoContrast", 0.638852434854615, 0.9819440776921611]], [["AutoContrast", 0.9766868312173507, 0.9651796447738792], ["AutoContrast", 0.3489760216898085, 0.3082182741354106]], [["Sharpness", 0.13693510871346704, 0.08297205456926067], ["Contrast", 0.3155812019005854, 0.031402991638917896]], [["TranslateY", 0.2664707540547008, 0.4838091910041236], ["ShearX", 0.5935665395229432, 0.7813088248538167]], [["ShearY", 0.7578577752251343, 0.5116014090216161], ["ShearX", 0.8332831240873545, 0.26781876290841017]], [["TranslateY", 0.473254381651761, 0.4203181582821155], ["ShearY", 0.732848696900726, 0.47895514793728433]], [["Solarize", 0.6922689176672292, 0.36403255869823725], ["AutoContrast", 0.910654040826914, 0.888651414068326]], [["ShearX", 0.37326536936166244, 0.47830923320699525], ["Equalize", 0.4724702976076929, 0.8176108279939023]], [["Contrast", 0.3839906424759326, 0.09109695563933692], ["Invert", 0.36305435543972325, 0.5701589223795499]], [["Invert", 0.5175591137387999, 0.38815675919253867], ["TranslateY", 0.1354848160153554, 0.41734106283245065]], [["Color", 0.829616006981199, 0.18631472346156963], ["Color", 0.2465115448326214, 0.9439365672808333]], [["Contrast", 0.18207939197942158, 0.39841173152850873], ["ShearX", 0.16723588254695632, 0.2868649619006758]], [["Posterize", 0.1941909136988733, 0.6322499882557473], ["Contrast", 0.6109060391509794, 0.27329598688783296]], [["AutoContrast", 0.9148775146158022, 0.09129288311923844], ["Sharpness", 0.4222442287436423, 0.847961820057229]], [["Color", 0.21084007475489852, 0.008218056412554131], ["Contrast", 0.43996934555301637, 0.500680146508504]], [["ShearY", 0.6745287915240038, 0.6120305524405164], ["Equalize", 0.467403794543269, 0.2207148995882467]], [["Color", 0.7712823974371379, 0.2839161885566902], ["Color", 0.8725368489709752, 0.3349470222415115]], [["Solarize", 0.5563976601161562, 0.540446614847802], ["Invert", 0.14228071175107454, 0.2242332811481905]], [["Contrast", 0.34596757983998383, 0.9158971503395041], ["Cutout", 0.6823724203724072, 0.5221518922863516]], [["Posterize", 0.3275475232882672, 0.6520033254468702], ["Color", 0.7434224109271398, 0.0824308188060544]], [["Cutout", 0.7295122229650082, 0.277887573018184], ["Brightness", 0.5303655506515258, 0.28628046739964497]], [["Color", 0.8533293996815943, 0.24909788223027743], ["Color", 0.6915962825167857, 0.33592561040195834]], [["TranslateX", 0.0761441550001345, 0.7043906245420134], ["Equalize", 0.670845297717783, 0.30986063097084215]], [["Contrast", 0.30592723366237995, 0.7365013059287382], ["Color", 0.6173835128817455, 0.6417028717640598]], [["Rotate", 0.05558240682703821, 0.7284722849011761], ["Color", 0.7814801133853666, 0.13335113981884217]], [["ShearY", 0.6521743070190724, 0.6272195913574455], ["Rotate", 0.36278432239870423, 0.2335623679787695]], [["Color", 0.6799351102482663, 0.3850250771244986], ["Brightness", 0.613901077818094, 0.2374900558949702]], [["Color", 0.551451255148252, 0.7284757153447965], ["Solarize", 0.4863815212982878, 0.3857941567681324]], [["Contrast", 0.32516343965159267, 0.689921852601276], ["Cutout", 0.5922142001124506, 0.7709605594115009]], [["Brightness", 0.23760063764495856, 0.6392077018854179], ["Brightness", 0.7288124083714078, 0.4487520490201095]], [["Sharpness", 0.5631112298553713, 0.6803534985114782], ["ShearX", 0.6743791169050775, 0.34039227245151127]], [["AutoContrast", 0.8260911840078349, 0.7705607269534767], ["Rotate", 0.8880749478363638, 0.8182460047684648]], [["ShearY", 0.7037620764408412, 0.5219573160970589], ["Posterize", 0.7186150466761102, 0.6187857686944253]], [["TranslateY", 0.2140494926702246, 0.9104233882669488], ["TranslateX", 0.4096039512896902, 0.9692703030784571]], [["Equalize", 0.5404313549028165, 0.04094078980738014], ["AutoContrast", 0.07870278300673744, 0.841020779977939]], [["ShearY", 0.2684638876128488, 0.5599793678740521], ["Cutout", 0.19537995362704022, 0.2400995206366768]], [["AutoContrast", 0.19366394417090382, 0.4130755503251951], ["Sharpness", 0.11735660606190662, 0.39276612830651914]], [["Cutout", 0.8313266945081518, 0.37171822186374703], ["Contrast", 0.5088549187459019, 0.2956405118511817]], [["Cutout", 0.28375485371479847, 0.37020183949342683], ["Posterize", 0.718761436947423, 0.2278804627251678]], [["ShearY", 0.6625840735667625, 0.5045065697748213], ["Rotate", 0.5175257698523389, 0.39496923901188824]], [["Color", 0.6498154010188212, 0.38674158604408604], ["Brightness", 0.8157804892728057, 0.05660118670560971]], [["Color", 0.5512855420254102, 0.7812054820692542], ["Solarize", 0.8851292984174468, 0.2808951606943277]], [["Contrast", 0.35258433539074363, 0.8085377169629859], ["Cutout", 0.5197965849563265, 0.8657111726930974]], [["Cutout", 0.23650925054419358, 0.746860862983295], ["Brightness", 0.8842190203336139, 0.4389347348156118]], [["Rotate", 0.8651460526861932, 0.0031372441327392753], ["Equalize", 0.3909498933963822, 0.6221687914603954]], [["TranslateX", 0.5793690303540427, 0.37939687327382987], ["Invert", 0.846172545690258, 0.36950442052945853]], [["Invert", 0.5151721602607067, 0.5860134277259832], ["Contrast", 0.6868708526377458, 0.2188104093363727]], [["Contrast", 0.28019632529718025, 0.8403553410328943], ["Cutout", 0.5238340355491738, 0.6948434115725599]], [["Rotate", 0.1592592617684533, 0.5212044951482974], ["Color", 0.42404215473874546, 0.45894052919059103]], [["AutoContrast", 0.21780978427851283, 0.11813011387113281], ["Contrast", 0.14557770349869537, 0.5468616480449002]], [["Cutout", 0.03573873600256905, 0.8747186430368771], ["AutoContrast", 0.4804465018567564, 0.3968185812087325]], [["ShearY", 0.027192162947493492, 0.35923750027515866], ["Sharpness", 0.03207302705814674, 0.25868625346023777]], [["AutoContrast", 0.9111793886013045, 0.33534571661592005], ["ShearY", 0.31365410004768934, 0.37055495208177025]], [["Color", 0.5119732811716222, 0.10635303813092001], ["Solarize", 0.9828759703639677, 0.33302532900783466]], [["Contrast", 0.9652840964645487, 0.9550826002089741], ["ShearY", 0.16934262075572262, 0.35893022906919625]], [["Invert", 0.21526903298837538, 0.5491812432380025], ["TranslateX", 0.27691575128765095, 0.9916365493500338]], [["AutoContrast", 0.7223428288831728, 0.3001506080569529], ["Posterize", 0.28280773693692957, 0.5630226986948541]], [["TranslateY", 0.5334698670580152, 0.4329627064903895], ["Solarize", 0.11621274404555687, 0.38564564358937725]], [["Brightness", 0.9001900081991266, 0.15453762529292236], ["Equalize", 0.6749827304986464, 0.2174408558291521]], [["TranslateY", 0.703293071780793, 0.20371204513522137], ["Invert", 0.7921926919880306, 0.2647654009616249]], [["AutoContrast", 0.32650519442680254, 0.5567514700913352], ["ShearY", 0.7627653627354407, 0.5363510886152073]], [["Rotate", 0.364293676091047, 0.4262321334071656], ["Posterize", 0.7284189361001443, 0.6052618047275847]], [["Contrast", 0.004679138490284229, 0.6985327823420937], ["Posterize", 0.25412559986607497, 0.969098825421215]], [["ShearY", 0.6831738973100172, 0.6916463366962687], ["TranslateY", 0.8744153159733203, 0.3667879549647143]], [["Posterize", 0.39138456188265913, 0.8617909225610128], ["TranslateX", 0.5198303654364824, 0.5518823068009463]], [["Invert", 0.6471155996761706, 0.4793957129423701], ["ShearX", 0.8046274258703997, 0.9711394307595065]], [["Solarize", 0.2442520851809611, 0.5518114414771629], ["Sharpness", 0.02324109511463257, 0.18216585433541427]], [["Cutout", 0.7004457278387007, 0.4904439660213413], ["Contrast", 0.6516622044646659, 0.7324290164242575]], [["Brightness", 0.594212018801632, 0.5624822682300464], ["ShearX", 0.47929863548325596, 0.5610640338380719]], [["TranslateX", 0.20863492063218445, 0.23761872077836552], ["Color", 0.9374148559524687, 0.06390809573246009]], [["AutoContrast", 0.5548946725094693, 0.40547561665765874], ["Equalize", 0.26341425401933344, 0.2763692089379619]], [["Invert", 0.8224614398122034, 0.15547159819315676], ["Rotate", 0.4915912924663281, 0.6995695827608112]], [["Equalize", 0.05752620481520809, 0.80230125774557], ["Rotate", 0.16338857010673558, 0.8066738989167762]], [["ShearY", 0.5437502855505825, 0.252101665309144], ["Contrast", 0.9268450172095902, 0.13437399256747992]], [["TranslateY", 0.6946438457089812, 0.35376889837139813], ["Sharpness", 0.15438234648960253, 0.2668696344562673]], [["Invert", 0.24506516252953542, 0.1939315433476327], ["Sharpness", 0.8921986990130818, 0.21478051316241717]], [["TranslateY", 0.5292829065905086, 0.6896826369723732], ["Invert", 0.4461047865540309, 0.9854416526561315]], [["Posterize", 0.8085062334285464, 0.4538963572040656], ["Brightness", 0.2623572045603854, 0.16723779221170698]], [["Solarize", 0.1618752496191097, 0.6007634864056693], ["TranslateY", 0.07808851801433346, 0.3951252736249746]], [["TranslateX", 0.35426056783145843, 0.8875451782909476], ["Brightness", 0.5537927990151869, 0.3042790536918476]], [["Cutout", 0.9051584028783342, 0.6050507821593669], ["ShearX", 0.31185875057627255, 0.39145181108334876]], [["Brightness", 0.43157388465566776, 0.45511767545129933], ["ShearY", 0.626464342187273, 0.5251031991594401]], [["Contrast", 0.7978520212540166, 0.45088491126800995], ["ShearY", 0.20415027867560143, 0.24369493783350643]], [["ShearX", 0.48152242363853065, 0.001652619381325604], ["Sharpness", 0.6154899720956758, 0.22465778944283568]], [["Posterize", 0.0008092255557418104, 0.8624848793450179], ["Solarize", 0.7580784903978838, 0.4141187863855049]], [["TranslateY", 0.4829597846471378, 0.6077028815706373], ["ShearX", 0.43316420981872894, 0.007119694447608018]], [["Equalize", 0.2914045973615852, 0.6298874433109889], ["Cutout", 0.18663096101056076, 0.20634383363149222]], [["TranslateX", 0.6909947340830737, 0.40843889682671003], ["ShearX", 0.3693105697811625, 0.070573833710386]], [["Rotate", 0.6184027722396339, 0.6483359499288176], ["AutoContrast", 0.8658233903089285, 0.31462524418660626]], [["Brightness", 0.8165837262133947, 0.38138221738335765], ["Contrast", 0.01566790570443702, 0.1250581265407818]], [["Equalize", 0.16745169701901802, 0.9239433721204139], ["ShearY", 0.5535908803004554, 0.35879199699526654]], [["Color", 0.9675880875486578, 0.19745998576077994], ["Posterize", 0.641736196661405, 0.5702363593336868]], [["ShearY", 0.27730895136251943, 0.4730273890919014], ["Posterize", 0.35829530316120517, 0.9040968539551122]], [["Cutout", 0.9989158254302966, 0.3210048366589035], ["Equalize", 0.9226385492886618, 0.21132010337062]], [["Posterize", 0.32861829410989934, 0.7608163668499222], ["TranslateY", 0.528381246453454, 0.6837459631017135]], [["ShearY", 0.6786278797045173, 0.49006792710382946], ["ShearX", 0.7860409944610941, 0.7960317025665418]], [["Solarize", 0.4420731874598513, 0.7163961196254427], ["Sharpness", 0.11927615232343353, 0.3649599343067734]], [["Cutout", 0.4606157449857542, 0.4682141505042986], ["Contrast", 0.8955528913735222, 0.8468556570983498]], [["Brightness", 0.5742349576881501, 0.5633914487991978], ["ShearX", 0.8288987143597276, 0.5937556836469728]], [["Posterize", 0.05362153577922808, 0.40072961361335696], ["Rotate", 0.6681795049585278, 0.5348470042353504]], [["TranslateY", 0.6190833866612555, 0.7338431624993972], ["Color", 0.5352400737236565, 0.1598194251940268]], [["Brightness", 0.9942846465176832, 0.11918348505217388], ["Brightness", 0.0659098729688602, 0.6558077481794591]], [["Equalize", 0.34089122700685126, 0.048940774058585546], ["ShearX", 0.5472987107071652, 0.2965222509150173]], [["Sharpness", 0.3660728361470086, 0.37607120931207433], ["Sharpness", 0.9974987257291261, 0.2483317486035219]], [["Posterize", 0.931283270966942, 0.7525022430475327], ["Cutout", 0.6299208568533524, 0.3313382622423058]], [["Invert", 0.5074998650080915, 0.9722820836624784], ["Solarize", 0.13997049847474802, 0.19340041815763026]], [["AutoContrast", 0.6804950477263457, 0.31675149536227815], ["Solarize", 0.800632422196852, 0.09054278636377117]], [["TranslateY", 0.6886579465517867, 0.549118383513461], ["Brightness", 0.7298771973550124, 0.59421647759784]], [["Equalize", 0.8117050130827859, 0.22494316766261946], ["AutoContrast", 0.5217061631918504, 0.6106946809838144]], [["Equalize", 0.4734718117645248, 0.7746036952254298], ["Posterize", 0.032049205574512685, 0.9681402692267316]], [["Brightness", 0.4724177066851541, 0.7969700024018729], ["Solarize", 0.6930049134926459, 0.3880086567038069]], [["TranslateX", 0.2833979092130342, 0.6873833799104118], ["Rotate", 0.37167767436617366, 0.03249352593350204]], [["Posterize", 0.7080588381354884, 0.03014586990329654], ["Posterize", 0.20883930954891392, 0.1328596635826556]], [["Cutout", 0.1992050307454733, 0.8079881690617468], ["ShearY", 0.3057279570820446, 0.34868823290010564]], [["TranslateY", 0.6204358851346782, 0.24978856155434062], ["ShearX", 0.2403059671388028, 0.6706906799258086]], [["Contrast", 0.5527380063918701, 0.27504242043334765], ["Rotate", 0.37361791978638376, 0.17818567121454373]], [["Cutout", 0.3368229687890997, 0.013512329226772313], ["Contrast", 0.18480406673028238, 0.21653280083721013]], [["AutoContrast", 0.13634047961070397, 0.5322441057075571], ["Posterize", 0.3409948654529233, 0.2562132228604077]], [["Invert", 0.3375636037272626, 0.5417577242453775], ["Sharpness", 0.10271458969925179, 0.5125859420868099]], [["Invert", 0.26465503753231256, 0.7386494688407392], ["AutoContrast", 0.5310106090963371, 0.14699248759273964]], [["Sharpness", 0.8494538270706318, 0.9524607358113082], ["Solarize", 0.21142978953773187, 0.10711867917080763]], [["Equalize", 0.5185117903942263, 0.06342404369282638], ["ShearY", 0.26812877371366156, 0.32386585917978056]], [["TranslateY", 0.42724471339053904, 0.5218262942425845], ["Brightness", 0.7618037699290332, 0.5773256674209075]], [["Solarize", 0.5683461491921462, 0.7988018975591509], ["AutoContrast", 0.21826664523938988, 0.4395073407383595]], [["Posterize", 0.2564295537162734, 0.6778150727248975], ["Equalize", 0.7571361164411801, 0.4281744623444925]], [["Invert", 0.5171620125994946, 0.8719074953677988], ["ShearX", 0.10216776728552601, 0.20888013515457593]], [["Equalize", 0.934033636879294, 0.7724470445507672], ["ShearX", 0.14671590364536757, 0.06500753170863127]], [["Cutout", 0.48433709681747783, 0.8989915985203363], ["ShearY", 0.5161346572684965, 0.3154078452465332]], [["AutoContrast", 0.4337913490682531, 0.8651407398083308], ["AutoContrast", 0.31402168607643444, 0.5001710653814162]], [["Brightness", 0.4805460794016203, 0.8182812769485313], ["Equalize", 0.6811585495672738, 0.25172380097389147]], [["TranslateX", 0.05384872718386273, 0.7854623644701991], ["Color", 0.12583336502656287, 0.08656304042059215]], [["TranslateX", 0.3949348949001942, 0.0668909826131569], ["ShearX", 0.2895255694762277, 0.23998090792480392]], [["TranslateY", 0.3183346601371876, 0.5869865305603826], ["Cutout", 0.38601500458347904, 0.37785641359408184]], [["Sharpness", 0.3676509660134142, 0.6370727445512337], ["Rotate", 0.17589815946040205, 0.912442427082365]], [["Equalize", 0.46427003979798154, 0.7771177715171392], ["Cutout", 0.6622980582423883, 0.47780927252115374]], [["TranslateX", 0.4535588156726688, 0.9548833090146791], ["ShearY", 0.18609208838268262, 0.034329918652624025]], [["Rotate", 0.4896172340987028, 0.4842683413051553], ["Brightness", 0.08416972178617699, 0.2946109607041465]], [["TranslateY", 0.1443363248914217, 0.7352253161146544], ["ShearX", 0.025210952382823004, 0.6249971039957651]], [["Brightness", 0.08771030702840285, 0.5926338109828604], ["Contrast", 0.629121304110493, 0.36114268164347396]], [["Cutout", 0.003318169533990778, 0.984234627407162], ["Color", 0.5656264894233379, 0.9913705503959709]], [["Cutout", 0.17582168928005226, 0.5163176285036686], ["Sharpness", 0.42976684239235224, 0.9936723374147685]], [["Rotate", 0.13343297511611085, 0.730719022391835], ["Cutout", 0.43419793455016154, 0.9802436121876401]], [["ShearX", 0.8761482122895571, 0.11688364945899332], ["Solarize", 0.6071032746712549, 0.9972373138154098]], [["Contrast", 0.2721995133325574, 0.9467839388553563], ["AutoContrast", 0.357368427575824, 0.6530359095247653]], [["Equalize", 0.5334298945812708, 0.7157629957411794], ["Brightness", 0.8885107405370157, 0.2909013041171791]], [["Equalize", 0.4907081744271751, 0.9999203497290372], ["ShearX", 0.0055186544890628575, 0.20501406304441697]], [["Color", 0.4865852751351166, 0.14717278223914915], ["TranslateX", 0.0492335566831905, 0.01654291587484527]], [["Contrast", 0.3753662301521211, 0.866484274102244], ["Color", 0.21148416029328898, 0.37861792266657684]], [["TranslateY", 0.03960047686663052, 0.9948086048192006], ["TranslateX", 0.5802633545422445, 0.7696464344779717]], [["Contrast", 0.6456791961464718, 0.6304663998505495], ["Sharpness", 0.594774521429873, 0.8024138008893688]], [["Equalize", 0.5326123709954759, 0.7361990154971826], ["Invert", 0.5337609996065145, 0.06826577456972233]], [["ShearY", 0.7177596430755101, 0.16672206074906565], ["Equalize", 0.1847132768987843, 0.16186121936769876]], [["ShearY", 0.037342495065949534, 0.7762322168034441], ["Rotate", 0.28731231550023495, 0.4605573565280328]], [["Contrast", 0.6815742688289678, 0.04073638022156048], ["Cutout", 0.20201133153964437, 0.048429819360450654]], [["Color", 0.5295323372448824, 0.8591352159356821], ["Posterize", 0.7743900815037675, 0.8308865010050488]], [["Solarize", 0.9325362059095493, 0.4070769736318192], ["Contrast", 0.09359008071252661, 0.2808191171337515]], [["Sharpness", 0.6413241263332543, 0.5493867784897841], ["Solarize", 0.021951790397463734, 0.1045868634597023]], [["Color", 0.006027943433085061, 0.698043169126901], ["TranslateX", 0.06672167045857719, 0.6096719632236709]], [["TranslateX", 0.42167004878865333, 0.8844171486107537], ["Color", 0.12383835252312375, 0.9559595374068695]], [["Posterize", 0.5382560989047361, 0.6014252438301297], ["Color", 0.26197040526014054, 0.3423981550778665]], [["Cutout", 0.33150268513579584, 0.40828564490879615], ["AutoContrast", 0.6907753092981255, 0.05779246756831708]], [["Equalize", 0.31608006376116865, 0.9958870759781376], ["TranslateY", 0.15842255624921547, 0.5764254535539765]], [["Contrast", 0.19859706438565994, 0.12680764238281503], ["TranslateY", 0.4694115475285127, 0.45831161348904836]], [["TranslateX", 0.18768081492494126, 0.7718605539481094], ["Cutout", 0.2340834739291012, 0.3290460999084155]], [["Posterize", 0.17300123510877463, 0.5276823821218432], ["AutoContrast", 0.5861008799330297, 0.31557924295308126]], [["TranslateX", 0.36140745478517367, 0.4172762477431993], ["Sharpness", 0.6518477061748665, 0.9033991248207786]], [["AutoContrast", 0.1757278990984992, 0.9562490311064124], ["Invert", 0.43712652497757065, 0.26925880337078234]], [["TranslateX", 0.38113274849599377, 0.35742156735271613], ["TranslateY", 0.47708889990018216, 0.7975974044609476]], [["Brightness", 0.39538470887490523, 0.09692156164771923], ["Equalize", 0.876825166573471, 0.0979346217138612]], [["Solarize", 0.07679586061933875, 0.45996163577975313], ["Invert", 0.039726680682847904, 0.23574574397443826]], [["ShearX", 0.9739648414905278, 0.5217986621319772], ["TranslateY", 0.21653455086845896, 0.30415852174016683]], [["TranslateY", 0.26965366633030263, 0.4355259497820251], ["Sharpness", 0.6343493801543757, 0.9337027079656623]], [["Rotate", 0.42301232492240126, 0.07813015342326983], ["AutoContrast", 0.28524730310382906, 0.24127293503900557]], [["Color", 0.826300213905907, 0.008451115447607682], ["Equalize", 0.6770124607838715, 0.2889698349030014]], [["Cutout", 0.3461911530045792, 0.7481322146924341], ["Brightness", 0.1831459184570124, 0.5487074846857195]], [["Brightness", 0.8455429603962046, 0.4838335496721761], ["Cutout", 0.5778222397066808, 0.7789798279724414]], [["Brightness", 0.7859388330361665, 0.5907006126719181], ["Brightness", 0.5299842953874527, 0.008670514958094622]], [["Rotate", 0.9584331504536162, 0.7242692977964363], ["TranslateY", 0.46941406313257866, 0.748911298847083]], [["AutoContrast", 0.5878130357161462, 0.25218818797390996], ["Solarize", 0.815466142337258, 0.20231731395730107]], [["ShearX", 0.15594838773787617, 0.9764784874102524], ["TranslateY", 0.5805369037495945, 0.1412009058745196]], [["Sharpness", 0.7936370935749524, 0.5142489498674206], ["Sharpness", 0.1544307510097193, 0.3678451501088748]], [["TranslateY", 0.29391437860633873, 0.3520843012638746], ["Brightness", 0.5885278199370352, 0.04915265122854349]], [["AutoContrast", 0.3329771519033218, 0.2459852352278583], ["Equalize", 0.8674782697650298, 0.2900192232303214]], [["Cutout", 0.58997726901359, 0.9910393463442352], ["Contrast", 0.09792234559792412, 0.23341828880112486]], [["Cutout", 0.4643317809492098, 0.3224299097542076], ["TranslateY", 0.7998033586490294, 0.27086436352896565]], [["AutoContrast", 0.13138317155414905, 0.3419742927322439], ["TranslateY", 0.05413070060788905, 0.5504283113763994]], [["Posterize", 0.3645493423712921, 0.10684861674653627], ["Color", 0.6343589365592908, 0.9712261380583729]], [["Color", 0.06539862123316142, 0.34370535435837324], ["Equalize", 0.8098077629435421, 0.1272416658849032]], [["Invert", 0.3600258964493429, 0.7455698641930473], ["Color", 0.4118102215241555, 0.4489347750419333]], [["Sharpness", 0.2230673636976691, 0.2240713255305713], ["AutoContrast", 0.5039292091174429, 0.033700713206763835]], [["ShearX", 0.10611028325684749, 0.4235430688519599], ["Brightness", 0.354597328722803, 0.6835155193055997]], [["ShearX", 0.101313662029975, 0.3048854771395032], ["ShearX", 0.39832929626318425, 0.5569152062399838]], [["ShearX", 0.46033087857932264, 0.5976525683159943], ["Color", 0.8117411866929898, 0.22950658046373415]], [["Cutout", 0.04125062306390376, 0.5021647863925347], ["TranslateY", 0.4949139091550513, 0.40234738545601595]], [["TranslateX", 0.9982425877241792, 0.3912268450702254], ["Cutout", 0.8094853705295444, 0.4628037417520003]], [["Contrast", 0.47154787535001147, 0.5116549800625204], ["Invert", 0.4929108509901112, 0.713690694626014]], [["ShearX", 0.3073913369156325, 0.5912409524756753], ["Equalize", 0.5603975982699875, 0.12046838435247365]], [["TranslateY", 0.8622939212850868, 0.057802109037417344], ["Invert", 0.7577173459800602, 0.33727019024447835]], [["Cutout", 0.3646694663986778, 0.6285264075514656], ["Color", 0.5589259087346165, 0.6650676195317845]], [["Invert", 0.8563008117600374, 0.6216056385231019], ["AutoContrast", 0.7575002303510038, 0.6906934785154547]], [["ShearX", 0.4415411885102101, 0.301535484182858], ["TranslateY", 0.779716145113622, 0.5792057745092073]], [["Invert", 0.10736083594024397, 0.10640910911300788], ["Posterize", 0.5923391813408784, 0.5437447559328059]], [["Color", 0.4745215286268124, 0.08046291318852558], ["Rotate", 0.1642897827127771, 0.20754337935267492]], [["Invert", 0.3141086213412405, 0.5865422721808763], ["AutoContrast", 0.7551954144793225, 0.5588044000850431]], [["Equalize", 0.979500405577596, 0.6846916489547885], ["Rotate", 0.11257616752512875, 0.8137724117751907]], [["Equalize", 0.6315666801659133, 0.71548254701219], ["Cutout", 0.38805635642306224, 0.29282906744304604]], [["Posterize", 0.022485702859896456, 0.2794994040845844], ["Color", 0.4554990465860552, 0.5842888808848151]], [["Invert", 0.15787502346886398, 0.5137397924063724], ["TranslateY", 0.487638703473969, 0.6428121360825987]], [["Rotate", 0.20473927977443407, 0.6090899892067203], ["Contrast", 0.3794752343740154, 0.8056548374185936]], [["AutoContrast", 0.35889225269685354, 0.7311496777471619], ["Sharpness", 0.10152796686794396, 0.34768639850633193]], [["Rotate", 0.6298704242033275, 0.09649334401126405], ["Solarize", 0.24713244934163017, 0.4292117526982358]], [["Contrast", 0.9851015107131748, 0.30895068679118054], ["Sharpness", 0.7167845732283787, 0.36269175386392893]], [["Equalize", 0.49699932368219435, 0.21262924430159158], ["Contrast", 0.8497731498354579, 0.672321242252727]], [["ShearX", 0.18955591368056923, 0.47178691165954034], ["Sharpness", 0.17732805705271348, 0.5486957094984023]], [["ShearY", 0.5087926728214892, 0.8236809302978783], ["AutoContrast", 0.9661195881001936, 0.1309360428195535]], [["Rotate", 0.7825835251082691, 0.8292427086033229], ["TranslateX", 0.2034110174253454, 0.4073091408820304]], [["Cutout", 0.33457316681888716, 0.480098511703719], ["Sharpness", 0.8686004956803908, 0.21719357589897192]], [["ShearX", 0.30750577846813, 0.6349236735519613], ["Color", 0.5096781256213182, 0.5367289796478476]], [["Rotate", 0.7881847986981432, 0.846966895144323], ["Posterize", 0.33955649631388407, 0.9484449471562024]], [["Posterize", 0.5154127791998345, 0.8765287012129974], ["Posterize", 0.09621562708431097, 0.42108077474553995]], [["ShearX", 0.5513772653411826, 0.27285892893658015], ["AutoContrast", 0.027608088485522986, 0.1738173285576814]], [["Equalize", 0.7950881609822011, 0.05938388811616446], ["ShearX", 0.7864733097562856, 0.5928584864954718]], [["Equalize", 0.03401947599579436, 0.4936643525799874], ["Solarize", 0.8445332527647407, 0.4695434980914176]], [["AutoContrast", 0.9656295942383031, 0.6330670076537706], ["Brightness", 0.303859679517296, 0.8882002295195086]], [["ShearY", 0.5242765280639856, 0.7977406809732712], ["Rotate", 0.24810823616083127, 0.41392557985700773]], [["Posterize", 0.6824268148168342, 0.21831492475831715], ["ShearY", 0.0008811906288737209, 0.1939566265644924]], [["ShearY", 0.8413370823124643, 0.7075999817793881], ["Brightness", 0.7942266192900009, 0.0384845738170444]], [["ShearY", 0.9003919463843213, 0.5068340457708402], ["AutoContrast", 0.9990937631537938, 0.35323621376481695]], [["Contrast", 0.3266913024108897, 0.5470774782762176], ["Contrast", 0.31235464476196995, 0.5723334696204473]], [["AutoContrast", 0.40137522654585955, 0.4274859892417776], ["Sharpness", 0.6173858127038773, 0.9629236289042568]], [["Sharpness", 0.3728210261025356, 0.7873518787942092], ["Solarize", 0.4319848902062112, 0.799524274852396]], [["Sharpness", 0.009379857090624758, 0.3143858944787348], ["ShearY", 0.20273037650420184, 0.3501104740582885]], [["Color", 0.1837135820716444, 0.5709648984713641], ["Solarize", 0.36312838060628455, 0.3753448575775562]], [["Cutout", 0.3400431457353702, 0.6871688775988243], ["ShearX", 0.42524570507364123, 0.7108865889616602]], [["Sharpness", 0.30703348499729893, 0.885278643437672], ["Cutout", 0.04407034125935705, 0.6821013415071144]], [["Brightness", 0.7164362367177879, 0.3383891625406651], ["Posterize", 0.002136409392137939, 0.5744439712876557]], [["Rotate", 0.757566991428807, 0.41351586654059386], ["TranslateY", 0.6716670812367449, 0.45381701497377025]], [["Color", 0.29554345831738604, 0.5747484938203239], ["Brightness", 0.6495565535422139, 0.38353714282675055]], [["Color", 0.6552239827844064, 0.6396684879350223], ["Rotate", 0.4078437959841622, 0.8229364582618871]], [["ShearX", 0.3325165311431108, 0.99875651917317], ["Cutout", 0.060614087173980605, 0.8655206968462149]], [["ShearY", 0.8591223614020521, 0.47375809606391645], ["ShearY", 0.09964216351993155, 0.7076762087109618]], [["Color", 0.9353968383925787, 0.5171703648813921], ["Cutout", 0.7542267059402566, 0.4591488152776885]], [["ShearX", 0.6832456179177027, 0.6798505733549863], ["Color", 0.7408439718746301, 0.5061967673457707]], [["Equalize", 0.4451729339243929, 0.9242958562575693], ["Posterize", 0.2426742903818478, 0.7914731845374992]], [["Posterize", 0.6241497285503436, 0.6800650930438693], ["Rotate", 0.8212761169895445, 0.42470879405266637]], [["Sharpness", 0.35467334577635123, 0.4150922293649909], ["Color", 0.38988011871489925, 0.08762395748275534]], [["Invert", 0.20231176261188386, 0.34300045056881756], ["Color", 0.6311643386438919, 0.4311911861691113]], [["Contrast", 0.2892223327756343, 0.533349670629816], ["ShearY", 0.6483243327679983, 0.37584367848303185]], [["Contrast", 0.6516401043089397, 0.3801387361685983], ["Contrast", 0.38470661862567795, 0.994720698440467]], [["Contrast", 0.44558087160644655, 0.4234506152228727], ["AutoContrast", 0.30132391715441104, 0.7758068064149011]], [["ShearY", 0.8336612877669443, 0.6961881064757953], ["TranslateX", 0.111182606133131, 0.7138593872015647]], [["Brightness", 0.7252053408816349, 0.6883715819669095], ["Cutout", 0.6664014893052573, 0.5118622737562747]], [["TranslateX", 0.04294623433241698, 0.4737274091618545], ["Solarize", 0.15848056715239178, 0.436678451116009]], [["ShearX", 0.41843604414439584, 0.5571669083243844], ["Solarize", 0.31754187268874345, 0.643294796216908]], [["Cutout", 0.308644829376876, 0.9455913104658791], ["Cutout", 0.04221174396591258, 0.8004389485099825]], [["Invert", 0.7644819805649288, 0.393641460630097], ["Posterize", 0.20832144467525543, 0.6449709932505365]], [["ShearY", 0.60954354330238, 0.45193814135157406], ["Rotate", 0.07564178568434804, 0.5700158941616946]], [["Color", 0.47993653910354905, 0.18770437256254732], ["Equalize", 0.16540989366253533, 0.3295832145751728]], [["Sharpness", 0.773656112445468, 0.899183686347773], ["AutoContrast", 0.6225833171499476, 0.8375805811436356]], [["Brightness", 0.3119630413126101, 0.21694186245727698], ["Cutout", 0.08263220622864997, 0.9910421137289533]], [["TranslateY", 0.5200200210314198, 0.44467464167817444], ["Cutout", 0.3466375681433383, 0.22385957813397142]], [["ShearY", 0.4445374219718209, 0.23917745675733915], ["Equalize", 0.32094329607540717, 0.6286388268054685]], [["Invert", 0.6194633221674505, 0.6219326801360905], ["Color", 0.43219405413154555, 0.5463431710956901]], [["ShearX", 0.5491808798436206, 0.4485147269153593], ["ShearX", 0.9624243432991532, 0.581319457926692]], [["Cutout", 0.8486066390061917, 0.48538785811340557], ["Cutout", 0.15945182827781573, 0.4114259503742423]], [["TranslateX", 0.9845485123667319, 0.7590166645874611], ["Solarize", 0.9920857955871512, 0.33259831689209834]], [["Brightness", 0.3985764491687188, 0.3516086190155328], ["Cutout", 0.13907765098725244, 0.42430309616193995]], [["Color", 0.35877942890428727, 0.363294622757879], ["Equalize", 0.4997709941984466, 0.34475754120666147]], [["Sharpness", 0.5234916035905941, 0.8988480410886609], ["AutoContrast", 0.793554237802939, 0.2575758806963965]], [["Brightness", 0.36998588693418133, 0.24144652775222428], ["Cutout", 0.06610767765334377, 0.9979246311006975]], [["TranslateY", 0.6132425595571164, 0.43952345951359123], ["Cutout", 0.361849532200793, 0.8462247954545264]], [["Posterize", 0.36953849915949677, 0.3144747463577223], ["Equalize", 0.3258985378881982, 0.6314053736452068]], [["TranslateY", 0.35835648104981205, 0.08075066564380576], ["TranslateX", 0.5242389109555177, 0.11959330395816647]], [["ShearX", 0.32773751079554303, 0.9307864751586945], ["Sharpness", 0.006921805496030664, 0.8736511230672348]], [["TranslateY", 0.48202000226401526, 0.7058919195136056], ["ShearY", 0.6998308555145181, 0.21074360071080764]], [["AutoContrast", 0.7615852152325713, 0.24914859158079972], ["Cutout", 0.8270894478252626, 0.5804285538051077]], [["AutoContrast", 0.5391662421077847, 0.5233969710179517], ["Brightness", 0.04205906143049083, 0.382677139318253]], [["Brightness", 0.6904817357054526, 0.9116378156160974], ["Invert", 0.24305250280628815, 0.2384731852843838]], [["TranslateX", 0.2661235046256291, 0.9705982948874188], ["Sharpness", 0.35821873293899625, 0.0030835471296858444]], [["Posterize", 0.39029991982997647, 0.4286238191447004], ["TranslateX", 0.08954883207184736, 0.7263973533121859]], [["Cutout", 0.040284118298638344, 0.0388330236482832], ["Posterize", 0.7807814946471116, 0.5238352731112299]], [["ShearY", 0.43556653451802413, 0.6924037743225071], ["Contrast", 0.001081515338562919, 0.7340363920548519]], [["Sharpness", 0.6966467544442373, 0.10202517317137291], ["Color", 0.18836344735972566, 0.31736252662501935]], [["Contrast", 0.6460000689193517, 0.16242196500430484], ["AutoContrast", 0.6003831047484897, 0.8612141912778188]], [["Brightness", 0.9172874494072921, 0.292364504408795], ["Solarize", 0.344602582555059, 0.7054248176903991]], [["Brightness", 0.020940469451794064, 0.5051042440134866], ["Cutout", 0.569500058123745, 0.9091247933460598]], [["Invert", 0.7367715506799225, 0.636137024500329], ["TranslateY", 0.6186960283294023, 0.37626001619073624]], [["TranslateX", 0.2863246154089121, 0.7454318730628517], ["ShearY", 0.6649909124084395, 0.37639265910774133]], [["Equalize", 0.34603376919062656, 0.9324026002997775], ["Sharpness", 0.8481669261233902, 0.14545759197862507]], [["Contrast", 0.6184370038862784, 0.8074198580702933], ["TranslateX", 0.07036135693949985, 0.46222686847401306]], [["Invert", 0.9304884364616345, 0.26298808050002387], ["Color", 0.8027813156985396, 0.7748486756116594]], [["Posterize", 0.2887993806199106, 0.9576118517235523], ["Contrast", 0.07498577510121784, 0.09131727137211232]], [["Contrast", 0.8110536569461197, 0.051038215841138386], ["Solarize", 0.8799018446258887, 0.25028365826721977]], [["Cutout", 0.006954733791187662, 0.030507696587206496], ["Brightness", 0.45329597160103124, 0.9623148451520953]], [["TranslateX", 0.7436227980344521, 0.45996857241163086], ["Solarize", 0.9682234479355196, 0.70777684485634]], [["Brightness", 0.2080557865889058, 0.025557286020371328], ["AutoContrast", 0.4786039197123853, 0.9271157120589375]], [["Solarize", 0.1822930503108656, 0.8448222682426465], ["ShearX", 0.6221001240196488, 0.207994745014715]], [["Color", 0.27879201870553094, 0.9112278219836276], ["Color", 0.7508664408516654, 0.14885798940641318]], [["ShearX", 0.5496326925552889, 0.7643918760952656], ["AutoContrast", 0.7887459433195374, 0.5993900500657054]], [["ShearY", 0.7182376017241904, 0.7470412126724141], ["Rotate", 0.7644845975844854, 0.38510752407409893]], [["Contrast", 0.7984591239416293, 0.054767400038152704], ["Posterize", 0.7324315466290486, 0.41749946919991243]], [["Contrast", 0.596887781894766, 0.14832691232456097], ["Contrast", 0.05140651977459313, 0.14459348285712803]], [["TranslateX", 0.32766681876233766, 0.5291103977440215], ["Color", 0.6039423443931029, 0.6280077043167083]], [["Invert", 0.5267106136816635, 0.9429838545064784], ["Sharpness", 0.9999053422304087, 0.24764251340211074]], [["Contrast", 0.495767451313242, 0.6744720418896594], ["Brightness", 0.2220993631062378, 0.023842431692152832]], [["Invert", 0.7609399278201697, 0.38010826932678554], ["Color", 0.8454251931688355, 0.5876680099851194]], [["Posterize", 0.24967505238473384, 0.3801835337368412], ["Contrast", 0.15106121477353399, 0.6785384814310887]], [["Invert", 0.49594153211743874, 0.32307787492774986], ["Contrast", 0.46822075688054793, 0.7106858486805577]], [["Sharpness", 0.7204076261101202, 0.5928585438185809], ["Rotate", 0.2922878012111486, 0.2742491027179961]], [["Solarize", 0.2866813728691532, 0.2856363754608978], ["TranslateY", 0.7817609208793659, 0.17156048740523572]], [["Cutout", 0.03345540659323987, 0.30068271036485605], ["ShearY", 0.2556603044234358, 0.32397855468866993]], [["TranslateY", 0.20032231858163152, 0.4577561841994639], ["Cutout", 0.8063563515601337, 0.9224365467344459]], [["TranslateY", 0.27130034613023113, 0.7446375583249849], ["ShearX", 0.8254766023480402, 0.4187078898038131]], [["ShearX", 0.2937536068210411, 0.3864492533047109], ["Contrast", 0.7069611463424469, 0.686695922492015]], [["TranslateX", 0.5869084659063555, 0.7866008068031776], ["Invert", 0.289041613918004, 0.5774431720429087]], [["Posterize", 0.6199250263408456, 0.36010044446077893], ["Color", 0.7216853388297056, 0.18586684958836489]], [["Posterize", 0.16831615585406814, 0.08052519983493259], ["Cutout", 0.7325882891023244, 0.77416439921321]], [["Posterize", 0.3000961100422498, 0.5181759282337892], ["Contrast", 0.40376073196794304, 0.613724714153924]], [["ShearX", 0.32203193464136226, 0.037459860897434916], ["Solarize", 0.961542785512965, 0.5176575408248285]], [["Posterize", 0.8986732529036036, 0.7773257927223327], ["AutoContrast", 0.9765986969928243, 0.2092264330225745]], [["Posterize", 0.7463386563644007, 0.7086671048242543], ["Posterize", 0.6433819807034994, 0.00541136425219968]], [["Contrast", 0.8810746688690078, 0.4821029611474963], ["Invert", 0.5121169325265204, 0.6360694878582249]], [["AutoContrast", 0.457606735372388, 0.6104794570624505], ["Color", 0.0020511991982608124, 0.6488142202778011]], [["Invert", 0.01744463899367027, 0.9799156424364703], ["ShearY", 0.3448213456605478, 0.04437356383800711]], [["Solarize", 0.28511589596283315, 0.283465265528744], ["Rotate", 0.6831807199089897, 0.0617176467316177]], [["Sharpness", 0.329148970281285, 0.398397318402924], ["Color", 0.9125837011914073, 0.4724426676489746]], [["Posterize", 0.05701522811381192, 0.17109014518445975], ["Cutout", 0.785885656821686, 0.39072624694455804]], [["TranslateY", 0.36644251447248277, 0.5818480868136134], ["Equalize", 0.06162286852923926, 0.710929848709861]], [["ShearY", 0.8667124241442813, 0.7556246528256454], ["ShearY", 0.505190335528531, 0.2935701441277698]], [["Brightness", 0.6369570015916268, 0.5131486964430919], ["Color", 0.4887119711633827, 0.9364572089679907]], [["Equalize", 0.06596702627228657, 0.42632445412423303], ["Equalize", 0.583434672187985, 0.045592788478947655]], [["ShearY", 0.12701084021549092, 0.501622939075192], ["Cutout", 0.7948319202684251, 0.5662618207034569]], [["Posterize", 0.24586808377061664, 0.5178008194277262], ["Contrast", 0.1647040530405073, 0.7459410952796975]], [["Solarize", 0.346601298126444, 0.02933266448415553], ["ShearY", 0.9571781647031095, 0.4992610484566735]], [["Brightness", 0.5174960605130408, 0.4387498174634591], ["AutoContrast", 0.6327403754086753, 0.8279630556620247]], [["Posterize", 0.7591448754183128, 0.6265369743070788], ["Posterize", 0.5030300462943854, 0.00401699185532868]], [["Contrast", 0.02643254602183477, 0.44677741300429646], ["Invert", 0.2921779546234399, 0.732876182854368]], [["TranslateY", 0.3516821152310867, 0.7142224211142528], ["Brightness", 0.07382104862245475, 0.45368581543623165]], [["Invert", 0.21382474908836685, 0.8413922690356168], ["Invert", 0.4082563426777157, 0.17018243778787834]], [["Brightness", 0.9533955059573749, 0.8279651051553477], ["Cutout", 0.6730769221406385, 0.07780554260470988]], [["Brightness", 0.6022173063382547, 0.6008500678386571], ["Sharpness", 0.5051909719558138, 0.002298383273851839]], [["Contrast", 0.03373395758348563, 0.3343918835437655], ["Sharpness", 0.8933651164916847, 0.21738300404986516]], [["TranslateX", 0.7095755408419822, 0.26445508146225394], ["Equalize", 0.18255527363432034, 0.38857557766574147]], [["Solarize", 0.4045911117686074, 0.009106925727519921], ["Posterize", 0.9380296936271705, 0.5485821516085955]], [["Posterize", 0.20361995432403968, 0.45378735898242406], ["AutoContrast", 0.9020357653982511, 0.7880592087609304]], [["AutoContrast", 0.9921550787672145, 0.7396130723399785], ["Cutout", 0.4203609896071977, 0.13000504717682415]], [["Equalize", 0.1917806394805356, 0.5549114911941102], ["Posterize", 0.27636900597148506, 0.02953514963949344]], [["AutoContrast", 0.5427071893197213, 0.6650127340685553], ["Color", 0.011762461060904839, 0.3793508738225649]], [["Invert", 0.18495006059896424, 0.8561476625981166], ["ShearY", 0.6417068692813954, 0.9908751019535517]], [["Solarize", 0.2992385431633619, 0.33622162977907644], ["Rotate", 0.6070550252540432, 0.010205544695142064]], [["Sharpness", 0.33292787606841845, 0.549446566149951], ["Color", 0.9097665730481233, 0.9947658451503181]], [["Posterize", 0.11207465085954937, 0.23296263754645155], ["Cutout", 0.6159972426858633, 0.38289684517298556]], [["TranslateX", 0.7343689718523805, 0.16303049089087485], ["Equalize", 0.3138385390145809, 0.6096356352129273]], [["Solarize", 0.4807269891506887, 0.28116279654856363], ["Posterize", 0.9753467973380021, 0.6327025372916857]], [["Posterize", 0.837244997106023, 0.5586046483574153], ["AutoContrast", 0.9005775602024721, 0.7983389828641411]], [["AutoContrast", 0.8347112949943837, 0.7321850307727004], ["Cutout", 0.3322676575657192, 0.14409873524237032]], [["Equalize", 0.12285967262649124, 0.5368519477089722], ["Posterize", 0.2693593445898034, 0.15098267759162076]], [["Invert", 0.331021587020619, 0.3140868578915853], ["Cutout", 0.48268387543799884, 0.7642598986625201]], [["Equalize", 0.47573794714622175, 0.8628185952549363], ["Solarize", 0.14860046214144496, 0.3739284346347912]], [["AutoContrast", 0.6747373196190459, 0.2912917979635714], ["Posterize", 0.27259573208358623, 0.9643671211873469]], [["Sharpness", 0.15019788105901233, 0.7289238028242861], ["ShearY", 0.7998448015985137, 0.5924798900807636]], [["Brightness", 0.7874052186079156, 0.9446398428550358], ["Equalize", 0.5105557539139616, 0.6719808885741001]], [["ShearX", 0.783252331899515, 0.74960184771181], ["ShearX", 0.4327935527932927, 0.29980994764698565]], [["Rotate", 0.03892023906368644, 0.24868635699639904], ["Cutout", 0.6408903979315637, 0.32135851733523907]], [["Invert", 0.9972802027590713, 0.9374194642823106], ["ShearX", 0.20016463162924894, 0.0052278586143255645]], [["AutoContrast", 0.9328687102578992, 0.44280614999256235], ["Color", 0.05637751621265141, 0.26921974769786455]], [["AutoContrast", 0.2798532308065416, 0.5283914274806746], ["Cutout", 0.12930089032151, 0.25624459046884057]], [["Invert", 0.2397428994839993, 0.31011715409282065], ["Cutout", 0.5875151915473042, 0.7454458580264322]], [["Equalize", 0.374815667651982, 0.9502053862625081], ["Solarize", 0.10100323698574426, 0.5124939317648691]], [["AutoContrast", 0.6009889057852652, 0.3080148907275367], ["Posterize", 0.6543352447742621, 0.17498668744492413]], [["Sharpness", 0.14402909409016001, 0.9239239955843186], ["ShearY", 0.8959818090635513, 0.7258262803413784]], [["Brightness", 0.8672271320432974, 0.8241439816189235], ["Equalize", 0.4954433852960082, 0.6687050430971254]], [["Solarize", 0.47813402689782114, 0.9447222576804901], ["TranslateY", 0.32546974113401694, 0.8367777573080345]], [["Sharpness", 0.48098022972519927, 0.2731904819197933], ["Rotate", 0.14601550238940067, 0.3955290089346866]], [["AutoContrast", 0.3777442613874327, 0.9991495158709968], ["TranslateY", 0.2951496731751222, 0.6276755696126608]], [["Cutout", 0.487150344941835, 0.7976642551725155], ["Solarize", 0.643407733524025, 0.6313641977306543]], [["Rotate", 0.35017053741686033, 0.23960877779589906], ["Sharpness", 0.8741761196478873, 0.12362019972427862]], [["Invert", 0.8849459784626776, 0.48532144354199647], ["Invert", 0.702430443380318, 0.924655906426149]], [["Equalize", 0.6324140359298986, 0.9780539325897597], ["AutoContrast", 0.39105074227907843, 0.3636856607173081]], [["AutoContrast", 0.8049993541952016, 0.3231157206314408], ["ShearY", 0.6675686366141409, 0.7345332792455934]], [["Sharpness", 0.12332351413693327, 0.9345179453120547], ["Solarize", 0.1594280186083361, 0.422049311332906]], [["Rotate", 0.38227253679386375, 0.7664364038099101], ["AutoContrast", 0.5725492572719726, 0.21049701651094446]], [["Brightness", 0.6432891832524184, 0.8243948738979008], ["Equalize", 0.20355899618080098, 0.7983877568044979]], [["ShearY", 0.694393675204811, 0.3686964692262895], ["TranslateX", 0.5593122846101599, 0.3378904046390629]], [["Invert", 0.9139730140623171, 0.7183505086140822], ["Posterize", 0.2675839177893596, 0.21399738931234905]], [["TranslateX", 0.05309461965184896, 0.032983777975422554], ["Sharpness", 0.412621944330688, 0.4752089612268503]], [["Equalize", 0.06901149860261116, 0.27405796188385945], ["AutoContrast", 0.7710451977604326, 0.20474249114426807]], [["ShearX", 0.47416427531072325, 0.2738614239087857], ["Cutout", 0.2820106413231565, 0.6295219975308107]], [["Cutout", 0.19984489885141582, 0.7019895950299546], ["ShearX", 0.4264722378410729, 0.8483962467724536]], [["ShearY", 0.42111446850243256, 0.1837626718066795], ["Brightness", 0.9187856196205942, 0.07478292286531767]], [["Solarize", 0.2832036589192868, 0.8253473638854684], ["Cutout", 0.7279303826662196, 0.615420010694839]], [["ShearX", 0.963251873356884, 0.5625577053738846], ["Color", 0.9637046840298858, 0.9992644813427337]], [["Invert", 0.7976502716811696, 0.43330238739921956], ["ShearY", 0.9113181667853614, 0.9066729024232627]], [["Posterize", 0.5750620807485399, 0.7729691927432935], ["Contrast", 0.4527879467651071, 0.9647739595774402]], [["Posterize", 0.5918751472569104, 0.26467375535556653], ["Posterize", 0.6347402742279589, 0.7476940787143674]], [["Invert", 0.16552404612306285, 0.9829939598708993], ["Solarize", 0.29886553921638087, 0.22487098773064948]], [["Cutout", 0.24209211313246753, 0.5522928952260516], ["AutoContrast", 0.6212831649673523, 0.4191071063984261]], [["ShearX", 0.4726406722647257, 0.26783614257572447], ["TranslateY", 0.251078162624763, 0.26103450676044304]], [["Cutout", 0.8721775527314426, 0.6284108541347894], ["ShearX", 0.7063325779145683, 0.8467168866724094]], [["ShearY", 0.42226987564279606, 0.18012694533480308], ["Brightness", 0.858499853702629, 0.4738929353785444]], [["Solarize", 0.30039851082582764, 0.8151511479162529], ["Cutout", 0.7228873804059033, 0.6174351379837011]], [["ShearX", 0.4921198221896609, 0.5678998037958154], ["Color", 0.7865298825314806, 0.9309020966406338]], [["Invert", 0.8077821007916464, 0.7375015762124386], ["Cutout", 0.032464574567796195, 0.25405044477004846]], [["Color", 0.6061325441870133, 0.2813794250571565], ["TranslateY", 0.5882949270385848, 0.33262043078220227]], [["ShearX", 0.7877331864215293, 0.8001131937448647], ["Cutout", 0.19828215489868783, 0.5949317580743655]], [["Contrast", 0.529508728421701, 0.36477855845285007], ["Color", 0.7145481740509138, 0.2950794787786947]], [["Contrast", 0.9932891064746089, 0.46930062926732646], ["Posterize", 0.9033014136780437, 0.5745902253320527]]]
    return p

class Augmentation(object):
    def __init__(self, policies):
        self.policies = policies

    def __call__(self, img):
        for _ in range(1):
            policy = random.choice(self.policies)
            for name, pr, level in policy:
                if random.random() > pr:
                    continue
                img = apply_augment(img, name, level)
        return img

cifar10_faa = Augmentation(fa_reduced_cifar10())
svhn_faa = Augmentation(fa_reduced_svhn())