import numpy as np


def safe_clip(x: np.ndarray) -> np.ndarray:
    """
    Clip input to be positive (>= 1e-9) to avoid math errors
    in log, power, or division-based utilities.
    """
    return np.clip(np.asarray(x, dtype=np.float64), 1e-9, None)



def linear_utility(x: np.ndarray) -> np.ndarray:
    """Linear utility: U(x) = x."""
    return np.asarray(x, dtype=np.float64)


def power_utility(x: np.ndarray, alpha: float) -> np.ndarray:
    """Power utility: U(x) = x^alpha."""
    return np.power(safe_clip(x), alpha)


def quadratic_utility(x: np.ndarray, b: float) -> np.ndarray:
    """Quadratic utility: U(x) = x - b*x^2."""
    x_safe = safe_clip(x)
    return x_safe - b * np.power(x_safe, 2)


# --- CRRA / CARA ---

def crra_utility(x: np.ndarray, theta: float) -> np.ndarray:
    """
    Constant Relative Risk Aversion (CRRA).
    U(x) = (x^(1-θ) - 1) / (1 - θ),   if θ != 1
         = log(x),                    if θ == 1
    """
    x_safe = safe_clip(x)
    if np.isclose(theta, 1.0):
        return np.log(x_safe)
    return (np.power(x_safe, 1.0 - theta) - 1.0) / (1.0 - theta)


def cara_utility(x: np.ndarray, alpha: float) -> np.ndarray:
    """
    Constant Absolute Risk Aversion (CARA).
    U(x) = (1 - exp(-αx)) / α,   if α != 0
         = x,                   if α == 0
    """
    x_safe = safe_clip(x)
    if np.isclose(alpha, 0.0):
        return x_safe
    return (1 - np.exp(-alpha * x_safe)) / alpha


# --- Variants ---

def isoelastic_utility(x: np.ndarray, theta: float, lam: float) -> np.ndarray:
    """
    Isoelastic utility with scale λ.
    U(x) = ( (λx)^(1-θ) - 1 ) / (1 - θ),   if θ != 1
         = log(λx),                        if θ == 1
    """
    x_safe = safe_clip(x)
    if np.isclose(theta, 1.0):
        return np.log(lam * x_safe)
    return (np.power(lam * x_safe, 1 - theta) - 1) / (1 - theta)


def prospect_theory_value(x: np.ndarray, alpha: float, beta: float, lam: float, reference_point: float) -> np.ndarray:
    """
    Prospect Theory value function:
    v(x) = (x - r)^α   if x >= r
         = -λ * (r - x)^β   if x < r
    """
    x = np.asarray(x, dtype=np.float64)
    d = x - reference_point
    gains = np.power(np.maximum(d, 0.0), alpha)
    losses = -lam * np.power(np.maximum(-d, 0.0), beta)
    return gains + losses


def hara_utility(x: np.ndarray, alpha: float, beta: float, gamma: float) -> np.ndarray:
    """
    Hyperbolic Absolute Risk Aversion (HARA).
    U(x) = ((1 - γ)/γ) * (αx + β)^γ
    """
    x_safe = safe_clip(x)
    gamma_safe = gamma if not np.isclose(gamma, 0) else 1e-9
    term = alpha * x_safe + beta
    term = np.clip(term, 1e-9, None)  # avoid negatives
    return ((1 - gamma_safe) / gamma_safe) * np.power(term, gamma_safe)


def expo_power_utility_saha(x: np.ndarray, alpha: float, theta: float) -> np.ndarray:
    """
    Expo-Power (Saha) utility.
    U(x) = (1 - exp(-α * x^(1-θ))) / α,   if α != 0
         = x^(1-θ),                      if α == 0
    """
    x_safe = safe_clip(x)
    if np.isclose(alpha, 0.0):
        return np.power(x_safe, 1 - theta)
    exponent_term = -alpha * np.power(x_safe, 1 - theta)
    return (1 - np.exp(exponent_term)) / alpha
