# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved

import argparse
import collections
import json
import os
import random
import sys
import time
import uuid

import numpy as np
import PIL
import torch
import torchvision
import torch.utils.data
from domainbed import datasets
from domainbed import hparams_registry
from domainbed import algorithms
from domainbed.lib import misc
from domainbed.lib.fast_data_loader import InfiniteDataLoader, FastDataLoader
import datetime
import re

if __name__ == "__main__":
    torch.cuda.empty_cache()
    parser = argparse.ArgumentParser(description='Domain generalization')
    parser.add_argument('--data_dir', type=str)
    parser.add_argument('--dataset', type=str, default="RotatedMNIST")
    parser.add_argument('--algorithm', type=str, default="ERM")
    parser.add_argument('--task', type=str, default="domain_generalization",
        choices=["domain_generalization", "domain_adaptation"])
    parser.add_argument('--hparams', type=str,
        help='JSON-serialized hparams dict')
    parser.add_argument('--hparams_seed', type=int, default=0,
        help='Seed for random hparams (0 means "default hparams")')
    parser.add_argument('--trial_seed', type=int, default=0,
        help='Trial number (used for seeding split_dataset and '
        'random_hparams).')
    parser.add_argument('--seed', type=int, default=0,
        help='Seed for everything else')
    parser.add_argument('--cuda', type=int, default=0,
        help='cuda')
    parser.add_argument('--steps', type=int, default=None,
        help='Number of steps. Default is dataset-dependent.')
    parser.add_argument('--checkpoint_freq', type=int, default=None,
        help='Checkpoint every N steps. Default is dataset-dependent.')
    parser.add_argument('--test_envs', type=int, nargs='+', default=[0])
    parser.add_argument('--output_dir', type=str, default=None)
    parser.add_argument('--qat_config_path', type=str, default=None, help="qat config path")
    parser.add_argument('--local_model_path', type=str, default=None, help="full precision path")
    parser.add_argument('--algorithm_dict', type=str, default=None, help="")
    parser.add_argument('--holdout_fraction', type=float, default=0.2)
    parser.add_argument('--uda_holdout_fraction', type=float, default=0,
        help="For domain adaptation, % of test to use unlabeled for training.")
    parser.add_argument('--skip_model_save', action='store_true')
    parser.add_argument('--save_model_every_checkpoint', action='store_true')
    parser.add_argument('--s_grad', type=str, default='both', help="first, both, second, random")
    parser.add_argument('--fix_layer', type=int, default=0, help='0,1,2,3,4')
    parser.add_argument('--clip_range', type=int, default=0, help='[-range, range], -1 mean low,mid,high')
    parser.add_argument('--fix_5000', type=str, default='layer2.0.conv1.quan_a_fn.s')
    parser.add_argument('--fix_reverse', action='store_true', help='reverse fix')
    parser.add_argument('--fix_alter_steps', type=int, default=0, help='alter train')
    parser.add_argument('--sam_rho', type=float, default=0.05)
    parser.add_argument('--freeze_steps', type=int, default=2111) # 2111表示这项没有
    parser.add_argument('--freeze_old_ratio', type=float, default=-1) # 小于freeze_old_ratio的冻结old
    parser.add_argument('--sagm_alpha', type=float, default=0.001)
    parser.add_argument('--sort_steps', type=int, default=400)
    parser.add_argument('--freeze_new_ratio', type=float, default=1.1)# 大于freeze_new_ratio的冻结new
    parser.add_argument('--st_freeze_steps', type=int, default=0)# 大于freeze_new_ratio的冻结new
    parser.add_argument('--lsq_lr', type=float, default=1e-5)# 大于freeze_new_ratio的冻结new
    parser.add_argument('--continue_freeze',  action='store_false') # 默认True, 传入为False
    parser.add_argument('--reverse_freeze',  action='store_true') # 默认False 传入为True
    parser.add_argument('--no_unfreeze',  action='store_true') # 默认False, 传入为True
    
    args = parser.parse_args()

    # If we ever want to implement checkpointing, just persist these values
    # every once in a while, and then load them from disk here.
    start_step = 0
    algorithm_dict = None


    if args.hparams_seed == 0:
        hparams = hparams_registry.default_hparams(args.algorithm, args.dataset)
    else:
        hparams = hparams_registry.random_hparams(args.algorithm, args.dataset,
            misc.seed_hash(args.hparams_seed, args.trial_seed))
    if args.hparams:
        with open(args.hparams, 'r') as file:
            data = file.read()
            hparams.update(json.loads(data))
    hparams.update({'steps': args.steps})
    hparams.update({'qat_config_path': args.qat_config_path})
    hparams.update({'local_model_path': args.local_model_path})
    hparams.update({'s_grad': args.s_grad})
    hparams.update({'fix_layer': args.fix_layer})
    hparams.update({'clip_range': args.clip_range})
    hparams.update({'fix_5000': args.fix_5000})
    hparams.update({'fix_reverse': args.fix_reverse})
    hparams.update({'fix_alter_steps': args.fix_alter_steps})
    hparams.update({'sam_rho': args.sam_rho})
    hparams.update({'freeze_steps': args.freeze_steps})
    hparams.update({'st_freeze_steps': args.st_freeze_steps})
    hparams.update({'freeze_old_ratio': args.freeze_old_ratio})
    hparams.update({'freeze_new_ratio': args.freeze_new_ratio})
    hparams.update({'sagm_alpha': args.sagm_alpha})
    hparams.update({'sort_steps': args.sort_steps})
    hparams.update({'lsq_lr': args.lsq_lr})
    hparams.update({'continue_freeze': args.continue_freeze})
    hparams.update({'reverse_freeze': args.reverse_freeze})
    hparams.update({'no_unfreeze': args.no_unfreeze})
    
    # output_dir修改
    current_time = str(datetime.datetime.now().timestamp())
    test_envs_str = '+'.join(str(num) for num in args.test_envs)
    if args.output_dir is None:
        if args.qat_config_path != None:
            bit = re.findall(r'\d+', args.qat_config_path)
            bit = ''.join(bit)
            args.output_dir = f"output/{args.dataset}/{args.algorithm}/{test_envs_str}/bit{bit}/seed{args.seed}/s_grad{args.s_grad}_/clip_range{args.clip_range}_/lr{hparams['lr']}_wd{hparams['weight_decay']}_freeze_old_ratio{args.freeze_old_ratio}_freeze_steps{args.freeze_steps}_st{args.st_freeze_steps}_continue{args.continue_freeze}_reverse_freeze{args.reverse_freeze}_no_unfreeze{args.no_unfreeze}/{current_time}"
        else:
            args.output_dir = f"output/{args.dataset}/{args.algorithm}/{test_envs_str}/full/seed{args.seed}_{current_time}"
        
    
    
    os.makedirs(args.output_dir, exist_ok=True)
    sys.stdout = misc.Tee(os.path.join(args.output_dir, 'out.txt'))
    sys.stderr = misc.Tee(os.path.join(args.output_dir, 'err.txt'))

    print("Environment:")
    print("\tPython: {}".format(sys.version.split(" ")[0]))
    print("\tPyTorch: {}".format(torch.__version__))
    print("\tTorchvision: {}".format(torchvision.__version__))
    print("\tCUDA: {}".format(torch.version.cuda))
    print("\tCUDNN: {}".format(torch.backends.cudnn.version()))
    print("\tNumPy: {}".format(np.__version__))
    print("\tPIL: {}".format(PIL.__version__))

    print('Args:')
    for k, v in sorted(vars(args).items()):
        print('\t{}: {}'.format(k, v))

    print('HParams:')
    for k, v in sorted(hparams.items()):
        print('\t{}: {}'.format(k, v))

    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

    if torch.cuda.is_available():
        device = "cuda:" + str(args.cuda)
    else:
        device = "cpu"
    torch.cuda.set_device(device)
    hparams.update({'cuda': device})
    if args.dataset in vars(datasets):
        dataset = vars(datasets)[args.dataset](args.data_dir,
            args.test_envs, hparams)
    else:
        raise NotImplementedError

    # Split each env into an 'in-split' and an 'out-split'. We'll train on
    # each in-split except the test envs, and evaluate on all splits.

    # To allow unsupervised domain adaptation experiments, we split each test
    # env into 'in-split', 'uda-split' and 'out-split'. The 'in-split' is used
    # by collect_results.py to compute classification accuracies.  The
    # 'out-split' is used by the Oracle model selectino method. The unlabeled
    # samples in 'uda-split' are passed to the algorithm at training time if
    # args.task == "domain_adaptation". If we are interested in comparing
    # domain generalization and domain adaptation results, then domain
    # generalization algorithms should create the same 'uda-splits', which will
    # be discared at training.
    in_splits = []
    out_splits = []
    uda_splits = []
    for env_i, env in enumerate(dataset):
        uda = []

        out, in_ = misc.split_dataset(env,
            int(len(env)*args.holdout_fraction),
            misc.seed_hash(args.trial_seed, env_i))

        if env_i in args.test_envs:
            uda, in_ = misc.split_dataset(in_,
                int(len(in_)*args.uda_holdout_fraction),
                misc.seed_hash(args.trial_seed, env_i))

        if hparams['class_balanced']:
            in_weights = misc.make_weights_for_balanced_classes(in_)
            out_weights = misc.make_weights_for_balanced_classes(out)
            if uda is not None:
                uda_weights = misc.make_weights_for_balanced_classes(uda)
        else:
            in_weights, out_weights, uda_weights = None, None, None
        in_splits.append((in_, in_weights))
        out_splits.append((out, out_weights))
        if len(uda):
            uda_splits.append((uda, uda_weights))

    if args.task == "domain_adaptation" and len(uda_splits) == 0:
        raise ValueError("Not enough unlabeled samples for domain adaptation.")

    train_loaders = [InfiniteDataLoader(
        dataset=env,
        weights=env_weights,
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS)
        for i, (env, env_weights) in enumerate(in_splits)
        if i not in args.test_envs]

    uda_loaders = [InfiniteDataLoader(
        dataset=env,
        weights=env_weights,
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS)
        for i, (env, env_weights) in enumerate(uda_splits)]

    eval_loaders = [FastDataLoader(
        dataset=env,
        batch_size=64,
        num_workers=dataset.N_WORKERS)
        for env, _ in (in_splits + out_splits + uda_splits)]
    eval_weights = [None for _, weights in (in_splits + out_splits + uda_splits)]
    eval_loader_names = ['env{}_in'.format(i)
        for i in range(len(in_splits))]
    eval_loader_names += ['env{}_out'.format(i)
        for i in range(len(out_splits))]
    eval_loader_names += ['env{}_uda'.format(i)
        for i in range(len(uda_splits))]

    algorithm_class = algorithms.get_algorithm_class(args.algorithm)
    algorithm = algorithm_class(dataset.input_shape, dataset.num_classes,
        len(dataset) - len(args.test_envs), hparams)

    if algorithm_dict is not None:
        algorithm.load_state_dict(algorithm_dict)

    algorithm.to(device)

    train_minibatches_iterator = zip(*train_loaders)
    uda_minibatches_iterator = zip(*uda_loaders)
    checkpoint_vals = collections.defaultdict(lambda: [])

    steps_per_epoch = min([len(env)/hparams['batch_size'] for env,_ in in_splits])

    n_steps = args.steps or dataset.N_STEPS
    checkpoint_freq = args.checkpoint_freq or dataset.CHECKPOINT_FREQ

    def save_checkpoint(filename):
        if args.skip_model_save:
            return
        save_dict = {
            "args": vars(args),
            "model_input_shape": dataset.input_shape,
            "model_num_classes": dataset.num_classes,
            "model_num_domains": len(dataset) - len(args.test_envs),
            "model_hparams": hparams,
            "model_dict": algorithm.state_dict()
        }
        torch.save(save_dict, os.path.join(args.output_dir, filename))

    layer_names = ['0.network.layer1.0.conv1', '0.network.layer2.0.conv1', '0.network.layer4.0.conv3']
    last_results_keys = None
    step_start_time = time.time()
    for step in range(start_step, n_steps):
        if 'QAT' in args.algorithm and step % 100 == 0:
            for name, p in algorithm.network.named_parameters():
                if '.s' in name:
                    print( "step", step, name, p.data)
                # if 'layer1.0.conv1.quan_w_fn.s' in name:
                #     print( "step", step, 'layer1.0.conv1.quan_w_fn.s', p.data)
                # elif 'layer2.0.conv1.quan_w_fn.s' in name:
                #     print( "step", step, 'layer2.0.conv1.quan_w_fn.s', p.data)
                # elif 'layer4.0.conv3.quan_w_fn.s' in name:
                #     print( "step", step, 'layer4.0.conv3.quan_w_fn.s', p.data)
                # if 'layer1.0.conv1.quan_a_fn.s' in name:
                #     print( "step", step, 'layer1.0.conv1.quan_a_fn.s', p.data)
                # elif 'layer2.0.conv1.quan_a_fn.s' in name:
                #     print( "step", step, 'layer2.0.conv1.quan_a_fn.s', p.data)
                # elif 'layer4.0.conv3.quan_a_fn.s' in name:
                #     print( "step", step, 'layer4.0.conv3.quan_a_fn.s', p.data)
            
            
            for name, p in algorithm.network.named_modules():
                for layer_name in layer_names:
                    if layer_name  == name:
                        dis = torch.norm(p.weight - p.quan_w_fn(p.weight), p='fro')
                        print("step", step, name, "weight & quantized dis", dis)
        minibatches_device = [(x.to(device), y.to(device))
            for x,y in next(train_minibatches_iterator)]
        if args.task == "domain_adaptation":
            uda_device = [x.to(device)
                for x,_ in next(uda_minibatches_iterator)]
        else:
            uda_device = None
        step_vals = algorithm.update(minibatches_device, uda_device)
        checkpoint_vals['step_time'].append(time.time() - step_start_time)

        for key, val in step_vals.items():
            checkpoint_vals[key].append(val)
        # if (step % checkpoint_freq == 0) or (step == n_steps - 1):
        #     results = {
        #         'step': step,
        #         'epoch': step / steps_per_epoch,
        #     }

        #     for key, val in checkpoint_vals.items():
        #         results[key] = np.mean(val)

        #     evals = zip(eval_loader_names, eval_loaders, eval_weights)
        #     for name, loader, weights in evals:
        #         acc = misc.accuracy(algorithm, loader, weights, device)
        #         results[name+'_acc'] = acc

        #     results['mem_gb'] = torch.cuda.max_memory_allocated() / (1024.*1024.*1024.)

        #     results_keys = sorted(results.keys())
        #     if results_keys != last_results_keys:
        #         misc.print_row(results_keys, colwidth=12)
        #         last_results_keys = results_keys
        #     misc.print_row([results[key] for key in results_keys],
        #         colwidth=12)

        #     results.update({
        #         'hparams': hparams,
        #         'args': vars(args)
        #     })

        #     epochs_path = os.path.join(args.output_dir, 'results.jsonl')
        #     with open(epochs_path, 'a') as f:
        #         f.write(json.dumps(results, sort_keys=True) + "\n")

        #     algorithm_dict = algorithm.state_dict()
        #     start_step = step + 1
        #     checkpoint_vals = collections.defaultdict(lambda: [])

        #     if args.save_model_every_checkpoint:
        #         save_checkpoint(f'model_step{step}.pkl')
        if args.dataset == 'DomainNet':
            if hparams['steps'] - step > 5000:
                if step % (checkpoint_freq) == 0:
                    print('step', step)
                # if step % (1 * checkpoint_freq) == 0:
                #     results = {
                #         'step': step,
                #         'epoch': step / steps_per_epoch,
                #     }

                #     for key, val in checkpoint_vals.items():
                #         results[key] = np.mean(val)

                #     evals = zip(eval_loader_names, eval_loaders, eval_weights)
                #     for name, loader, weights in evals:
                #         acc = misc.accuracy(algorithm, loader, weights, device)
                #         results[name+'_acc'] = acc

                #     results['mem_gb'] = torch.cuda.max_memory_allocated() / (1024.*1024.*1024.)

                #     results_keys = sorted(results.keys())
                #     if results_keys != last_results_keys:
                #         misc.print_row(results_keys, colwidth=12)
                #         last_results_keys = results_keys
                #     misc.print_row([results[key] for key in results_keys],
                #         colwidth=12)

                #     results.update({
                #         'hparams': hparams,
                #         'args': vars(args)
                #     })

                #     epochs_path = os.path.join(args.output_dir, 'results.jsonl')
                #     with open(epochs_path, 'a') as f:
                #         f.write(json.dumps(results, sort_keys=True) + "\n")

                #     algorithm_dict = algorithm.state_dict()
                #     start_step = step + 1
                #     checkpoint_vals = collections.defaultdict(lambda: [])

                #     if args.save_model_every_checkpoint:
                #         save_checkpoint(f'model_step{step}.pkl')
            else:
                if (step % (checkpoint_freq) == 0 or (step == n_steps - 1)):
                    results = {
                    'step': step,
                    'epoch': step / steps_per_epoch,
                    }

                    for key, val in checkpoint_vals.items():
                        results[key] = np.mean(val)

                    evals = zip(eval_loader_names, eval_loaders, eval_weights)
                    for name, loader, weights in evals:
                        acc = misc.accuracy(algorithm, loader, weights, device)
                        results[name+'_acc'] = acc

                    results['mem_gb'] = torch.cuda.max_memory_allocated() / (1024.*1024.*1024.)

                    results_keys = sorted(results.keys())
                    if results_keys != last_results_keys:
                        misc.print_row(results_keys, colwidth=12)
                        last_results_keys = results_keys
                    misc.print_row([results[key] for key in results_keys],
                        colwidth=12)

                    results.update({
                        'hparams': hparams,
                        'args': vars(args)
                    })

                    epochs_path = os.path.join(args.output_dir, 'results.jsonl')
                    with open(epochs_path, 'a') as f:
                        f.write(json.dumps(results, sort_keys=True) + "\n")

                    algorithm_dict = algorithm.state_dict()
                    start_step = step + 1
                    checkpoint_vals = collections.defaultdict(lambda: [])

                    if args.save_model_every_checkpoint:
                        save_checkpoint(f'model_step{step}.pkl')
        else:
            if step % (checkpoint_freq) == 0 or (step == n_steps - 1):
                results = {
                'step': step,
                'epoch': step / steps_per_epoch,
                }

                for key, val in checkpoint_vals.items():
                    results[key] = np.mean(val)

                evals = zip(eval_loader_names, eval_loaders, eval_weights)
                for name, loader, weights in evals:
                    acc = misc.accuracy(algorithm, loader, weights, device)
                    results[name+'_acc'] = acc

                results['mem_gb'] = torch.cuda.max_memory_allocated() / (1024.*1024.*1024.)

                results_keys = sorted(results.keys())
                if results_keys != last_results_keys:
                    misc.print_row(results_keys, colwidth=12)
                    last_results_keys = results_keys
                misc.print_row([results[key] for key in results_keys],
                    colwidth=12)

                results.update({
                    'hparams': hparams,
                    'args': vars(args)
                })

                epochs_path = os.path.join(args.output_dir, 'results.jsonl')
                with open(epochs_path, 'a') as f:
                    f.write(json.dumps(results, sort_keys=True) + "\n")

                algorithm_dict = algorithm.state_dict()
                start_step = step + 1
                checkpoint_vals = collections.defaultdict(lambda: [])

                if args.save_model_every_checkpoint:
                    save_checkpoint(f'model_step{step}.pkl')
    save_checkpoint('model.pkl')

    with open(os.path.join(args.output_dir, 'done'), 'w') as f:
        f.write('done')
