'''ResNet in PyTorch.
For Pre-activation ResNet, see 'preact_resnet.py'.
Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385

where this version deletes the final classifier layer and use the output from the penutimate layer
'''
import torch
import torch.nn as nn
import torch.nn.functional as F
from base import BaseModel


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, 
                conv_layer=None,
                norm_layer=None,
                activation_layer=None,
                remove_relu=False):
        super(BasicBlock, self).__init__()
        print(activation_layer)
        self.conv1 = conv_layer(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.norm1 = norm_layer(planes)
        self.relu = activation_layer(planes)
        self.conv2 = conv_layer(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.norm2 = norm_layer(planes)
        self.remove_relu = remove_relu
        if self.remove_relu:
            print("Basicblock, Removing the last relu")

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                conv_layer(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                norm_layer(self.expansion*planes)
            )

    def forward(self, x):
        x, out_list = x
        out = self.relu(self.norm1(self.conv1(x)))
        out_list.append(out) # 1 3 5 7 9 11 13 15
        out = self.norm2(self.conv2(out))
        out += self.shortcut(x)
        if not self.remove_relu:
            out = self.relu(out)
        out_list.append(out) # 2 4 6 8 10 12 14 16
        return [out, out_list]


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1,
                conv_layer=None,
                norm_layer=None,
                activation_layer=None,
                remove_relu=False):
        super(Bottleneck, self).__init__()
        self.conv1 = conv_layer(in_planes, planes, kernel_size=1, bias=False)
        self.norm1 = norm_layer(planes)
        self.conv2 = conv_layer(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.norm2 = norm_layer(planes)
        self.conv3 = conv_layer(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.norm3 = norm_layer(self.expansion*planes)
        self.relu = activation_layer(planes)
        self.remove_relu = remove_relu
        if self.remove_relu:
            print("Bottleneck, Removing the last relu")

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                conv_layer(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                norm_layer(self.expansion*planes)
            )

    def forward(self, x):
        out = self.relu(self.norm1(self.conv1(x)))
        out = self.relu(self.norm2(self.conv2(out)))
        out = self.norm3(self.conv3(out))
        out += self.shortcut(x)
        if not self.remove_relu:
            out = self.relu(out)
        return out


class ResNet(BaseModel):
    def __init__(self, block, num_blocks,
                norm_layer_type = 'bn',
                conv_layer_type = 'conv',
                linear_layer_type = 'linear',
                activation_layer_type = 'relu',
                etf_fc = False,
                remove_last_relu = False):
        super().__init__(norm_layer_type, conv_layer_type, linear_layer_type, 
                        activation_layer_type)
        
        self.in_planes = 64

        self.conv1 = self.conv_layer(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.norm1 = self.norm_layer(64)
        self.remove_last_relu = remove_last_relu
        self.relu = self.activation_layer(self.in_planes)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2, last=True)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
                    
    def _make_layer(self, block, planes, num_blocks, stride, last=False):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for i, stride in enumerate(strides):
            if last and i == len(strides) - 1:
                layers.append(block(self.in_planes, planes, stride,
                                    self.conv_layer, self.norm_layer, 
                                    self.activation_layer, 
                                    remove_relu = self.remove_last_relu))
                self.in_planes = planes * block.expansion
            else:
                layers.append(block(self.in_planes, planes, stride,
                                    self.conv_layer, self.norm_layer, 
                                    self.activation_layer))
                self.in_planes = planes * block.expansion
                
        return nn.Sequential(*layers)

    def forward(self, x):
        all_layer_out = []
        out = self.relu(self.norm1(self.conv1(x)))
        all_layer_out.append(out) # 0
        #print(out, all_layer_out)
        out, all_layer_out = self.layer1([out, all_layer_out])
        out, all_layer_out = self.layer2([out, all_layer_out])
        out, all_layer_out = self.layer3([out, all_layer_out])
        out, all_layer_out = self.layer4([out, all_layer_out])
        
        out = self.avgpool(out)
        out = torch.flatten(out, 1)
        return out, all_layer_out


def ResNet18(**kwargs):
    return ResNet(BasicBlock, [2, 2, 2, 2], **kwargs)


def ResNet34(**kwargs):
    return ResNet(BasicBlock, [3, 4, 6, 3], **kwargs)


def ResNet50(**kwargs):
    return ResNet(Bottleneck, [3, 4, 6, 3], **kwargs)


def ResNet101(**kwargs):
    return ResNet(Bottleneck, [3, 4, 23, 3], **kwargs)


def ResNet152(**kwargs):
    return ResNet(Bottleneck, [3, 8, 36, 3], **kwargs)


# Added, copy from https://github.com/HobbitLong/SupContrast/blob/master/networks/resnet_big.py
model_dict = {
    'resnet18': [ResNet18, 512],
    'resnet34': [ResNet34, 512],
    'resnet50': [ResNet50, 2048],
    'resnet101': [ResNet101, 2048],
    'mlp': [None,3072]
}


class LinearBatchNorm(nn.Module):
    """Implements BatchNorm1d by BatchNorm2d, for SyncBN purpose"""
    def __init__(self, dim, affine=True):
        super(LinearBatchNorm, self).__init__()
        self.dim = dim
        self.bn = nn.BatchNorm2d(dim, affine=affine)

    def forward(self, x):
        x = x.view(-1, self.dim, 1, 1)
        x = self.bn(x)
        x = x.view(-1, self.dim)
        return x


class SupConResNet(nn.Module):
    """backbone + projection head"""
    def __init__(self, name='resnet50', head='mlp', feat_dim=128, remove_last_relu=False):
        super(SupConResNet, self).__init__()
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun(remove_last_relu=remove_last_relu)
        self.loss_type = "SupCon"
        self.head_type = head
        if head == 'linear':
            self.head = nn.Linear(dim_in, feat_dim)
        elif head == 'mlp':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, feat_dim)
            )
        elif head == 'mlp2':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, feat_dim)
            )
        elif head == 'mlp3':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, feat_dim)
            )
        else:
            raise NotImplementedError(
                'head not supported: {}'.format(head))

    def forward(self, x):
        feat = self.encoder(x)
        # Normalize feature from encoder
        #feat = F.normalize(feat, dim=1)
        # default supcon normalizes feature after projection head before pass to loss
        feat = F.normalize(self.head(feat), dim=1)
        return feat

class SupCEResNet(nn.Module):
    """encoder + classifier"""
    def __init__(self, name='resnet50', head='mlp', feat_dim=128, remove_last_relu=False): # ,num_classes=10
        super(SupCEResNet, self).__init__()
        print(f"use {name} model")
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun(remove_last_relu=remove_last_relu)
        self.loss_type = "CE"
        self.head_type = head
    
        if head == "linear":
            self.head = nn.Sequential()
        elif head == 'mlp':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True)
            )
        elif head == 'mlp2':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True)
            )
        elif head == 'mlp3':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True)
            )
        
        else:
            raise NotImplementedError(
                'head not supported: {}'.format(head))
            
        #self.fc = nn.Linear(dim_in, num_classes)

    def forward(self, x):
        fea, all_layer_out = self.encoder(x)
        # Normalize feature from encoder
        #fea = F.normalize(fea, dim=1)
        
        head = self.head(fea)
        return head, all_layer_out


# class SupCEResNet(nn.Module):
#     """encoder + classifier"""
#     def __init__(self, name='resnet50', head='mlp'): # ,num_classes=10
#         super(SupCEResNet, self).__init__()
#         model_fun, dim_in = model_dict[name]
#         self.encoder = model_fun()
#         self.loss_type = "CE"
#         self.head_type = head
#         if head == 'mlp2':
#             self.head = nn.Sequential(
#                 nn.Linear(dim_in, dim_in),
#                 nn.ReLU(inplace=True),
#                 nn.Linear(dim_in, dim_in),
#                 nn.ReLU(inplace=True)
#             )
#         elif head == 'mlp':
#             self.head = nn.Sequential(
#                 nn.Linear(dim_in, dim_in),
#                 nn.ReLU(inplace=True)
#             )
#         elif head == "linear":
#             self.head = nn.Sequential()
#         else:
#             raise NotImplementedError(
#                 'head not supported: {}'.format(head))
            
#         #self.fc = nn.Linear(dim_in, num_classes)

#     def forward(self, x):
#         fea = self.encoder(x)
#         head = self.head(fea)
#         #return self.fc(head)
#         return head


class LinearClassifier(nn.Module):
    """Linear classifier"""
    def __init__(self, name='resnet50', num_classes=10, loss = "ce"):
        super(LinearClassifier, self).__init__()
        self.loss = loss
        print(f"Now initializing Linear Classifier, the loss used is {self.loss}")
        #print("Removing bias term in the linear Classifier")
        _, feat_dim = model_dict[name]
        self.linear = nn.Linear(feat_dim, num_classes, bias = True)

    def forward(self, features):
        if self.loss == "ce":
            # Normalize weight
            #self.linear.weight.data = F.normalize(self.linear.weight.data, dim = 1)
            # Normalize feature
            #features = F.normalize(features, dim=1)
            return self.linear(features)# * 10
        elif self.loss == "supcon":
            return self.linear(features)
        else:
            raise ValueError(f"loss: {self.loss} not supported")