#!/usr/bin/env bash
set -euo pipefail


###############################################################################
# CONFIGURATION
###############################################################################

# Set MODE to 'experiments' or 'switch'
MODE="${1:-experiments}"

echo "Running in MODE=$MODE"


# Models to evaluate (add or remove entries as needed
models=(
    # "o4-mini-2025-04-16"
    # "gemini-2.5-flash-preview-04-17"
    "Qwen/Qwen2.5-VL-72B-Instruct"
    # "gemini-2.0-flash"
)

# Datasets to run experiments on
datasets=(
    "clevr" 
    "omnimath-1" "bbeh_dyck_languages"
    "bbeh_object_counting" "bbeh_object_properties" "bbeh_boardgame_qa" "bbeh_boolean_expressions"
    "bbeh_zebra_puzzles" "bbeh_buggy_tables" "bbeh_spatial_reasoning" "bbeh_multistep_arithmetic"
    "bbeh_geometric_shapes" "bbeh_shuffled_objects" "bbeh_temporal_sequence" "bbeh_disambiguation_qa"
    "bbeh_causal_understanding" "bbeh_time_arithmetic" "bbeh_web_of_lies" "bbeh_sarc_triples"
    "bbeh_nycc" "bbeh_sportqa" "bbeh_linguini" "bbeh_movie_recommendation"
    "bbeh_hyperbaton" "bbeh_word_sorting"
    "leaf" "clutrr"
)

# Methods to use
methods=(
    "gen_sym_reason_prog_checks"
    "code"
    "zs_cot"
)



###############################################################################
# UTILITY FUNCTIONS
###############################################################################

# Control parallel jobs
wait_if_needed() {
    while [ "$(jobs -p | wc -l)" -ge 128 ]; do
        sleep 1
    done
}

# Check whether the given (model, method, dataset) experiment
# is **already running**; if so, skip launching it again.
is_running() {
    local _model="$1"
    local _method="$2"
    local _dataset="$3"

    # We look for an existing python process that contains all three flags.
    # The grep -F strings are deliberately separate so that order doesn’t matter.
    ps aux \
        | grep -F "scripts/run_experiments.py" \
        | grep -F -- "--model ${_model}" \
        | grep -F -- "--method ${_method}" \
        | grep -F -- "--dataset ${_dataset}" \
        | grep -v grep \
        >/dev/null
}

# Absolute or relative path to the results file for an experiment
results_file() {
    local _model="$1" _method="$2" _dataset="$3"
    echo "logs/${_model}/${_dataset}/${_method}/results.txt"
}


###############################################################################
# ACTIVATE ENVIRONMENT
###############################################################################
if [[ -z "${CONDA_DEFAULT_ENV:-}" ]]; then
    if command -v conda &> /dev/null; then
        echo "Activating conda environment 'unesy'..."
        eval "$(conda shell.bash hook)"
        conda activate unesy
    else
        echo "⚠️  Conda not found, and no conda environment is active. Please activate 'unesy' manually."
        exit 1
    fi
else
    echo "Conda environment '$CONDA_DEFAULT_ENV' already active."
fi

###############################################################################
# MAIN SCRIPT LOGIC
###############################################################################

case "$MODE" in
    experiments)
        echo "Starting full experiment sweep..."

        for model in "${models[@]}"; do
            for method in "${methods[@]}"; do
                for dataset in "${datasets[@]}"; do

                    # Skip if another identical run is still alive
                    if is_running "$model" "$method" "$dataset"; then
                        echo "⏩ Already running: model=$model, method=$method, dataset=$dataset"
                        continue
                    fi

                    # 2️⃣ Skip if finished results exist
                    result_path="$(results_file "$model" "$method" "$dataset")"
                    if [[ -f "$result_path" ]]; then
                        echo "✅ Completed earlier: $result_path"
                        continue
                    fi

                    wait_if_needed
                    echo "→ Starting: model=$model, method=$method, dataset=$dataset"
                    sleep 10
                    python scripts/run_experiments.py \
                        --method "$method" \
                        --model "$model" \
                        --log \
                        --dataset "$dataset" &
                done
            done
        done
        ;;

    switch)
        echo "Starting 'switch' mode (algorithmic_eval.py --method 7)..."
        method="7"

        for model in "${models[@]}"; do
            for dataset in "${datasets[@]}"; do
                result_path="logs/${model}/${dataset}/algorithmic_eval_results_${method}.json"

                if [[ -f "$result_path" ]]; then
                    echo "⏩ Skipping existing result: $result_path"
                    continue
                fi

                wait_if_needed
                echo "→ Starting: model=$model, method=$method, dataset=$dataset"
                sleep 10
                python scripts/algorithmic_eval.py \
                    --method "$method" \
                    --model "$model" \
                    --dataset "$dataset" &
            done
        done
        ;;

    *)
        echo "Unknown MODE '$MODE'. Use: experiments | switch"
        exit 1
        ;;
esac

###############################################################################
# WAIT FOR JOBS COMPLETION
###############################################################################

wait
echo "All jobs completed successfully."