#define _CFFI_

/* We try to define Py_LIMITED_API before including Python.h.

   Mess: we can only define it if Py_DEBUG, Py_TRACE_REFS and
   Py_REF_DEBUG are not defined.  This is a best-effort approximation:
   we can learn about Py_DEBUG from pyconfig.h, but it is unclear if
   the same works for the other two macros.  Py_DEBUG implies them,
   but not the other way around.

   The implementation is messy (issue #350): on Windows, with _MSC_VER,
   we have to define Py_LIMITED_API even before including pyconfig.h.
   In that case, we guess what pyconfig.h will do to the macros above,
   and check our guess after the #include.

   Note that on Windows, with CPython 3.x, you need >= 3.5 and virtualenv
   version >= 16.0.0.  With older versions of either, you don't get a
   copy of PYTHON3.DLL in the virtualenv.  We can't check the version of
   CPython *before* we even include pyconfig.h.  ffi.set_source() puts
   a ``#define _CFFI_NO_LIMITED_API'' at the start of this file if it is
   running on Windows < 3.5, as an attempt at fixing it, but that's
   arguably wrong because it may not be the target version of Python.
   Still better than nothing I guess.  As another workaround, you can
   remove the definition of Py_LIMITED_API here.

   See also 'py_limited_api' in cffi/setuptools_ext.py.
*/
#if !defined(_CFFI_USE_EMBEDDING) && !defined(Py_LIMITED_API)
#  ifdef _MSC_VER
#    if !defined(_DEBUG) && !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#    include <pyconfig.h>
     /* sanity-check: Py_LIMITED_API will cause crashes if any of these
        are also defined.  Normally, the Python file PC/pyconfig.h does not
        cause any of these to be defined, with the exception that _DEBUG
        causes Py_DEBUG.  Double-check that. */
#    ifdef Py_LIMITED_API
#      if defined(Py_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_DEBUG, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_TRACE_REFS)
#        error "pyconfig.h unexpectedly defines Py_TRACE_REFS, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_REF_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_REF_DEBUG, but Py_LIMITED_API is set"
#      endif
#    endif
#  else
#    include <pyconfig.h>
#    if !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#  endif
#endif

#include <Python.h>
#ifdef __cplusplus
extern "C" {
#endif
#include <stddef.h>

/* This part is from file 'cffi/parse_c_type.h'.  It is copied at the
   beginning of C sources generated by CFFI's ffi.set_source(). */

typedef void *_cffi_opcode_t;

#define _CFFI_OP(opcode, arg)   (_cffi_opcode_t)(opcode | (((uintptr_t)(arg)) << 8))
#define _CFFI_GETOP(cffi_opcode)    ((unsigned char)(uintptr_t)cffi_opcode)
#define _CFFI_GETARG(cffi_opcode)   (((intptr_t)cffi_opcode) >> 8)

#define _CFFI_OP_PRIMITIVE       1
#define _CFFI_OP_POINTER         3
#define _CFFI_OP_ARRAY           5
#define _CFFI_OP_OPEN_ARRAY      7
#define _CFFI_OP_STRUCT_UNION    9
#define _CFFI_OP_ENUM           11
#define _CFFI_OP_FUNCTION       13
#define _CFFI_OP_FUNCTION_END   15
#define _CFFI_OP_NOOP           17
#define _CFFI_OP_BITFIELD       19
#define _CFFI_OP_TYPENAME       21
#define _CFFI_OP_CPYTHON_BLTN_V 23   // varargs
#define _CFFI_OP_CPYTHON_BLTN_N 25   // noargs
#define _CFFI_OP_CPYTHON_BLTN_O 27   // O  (i.e. a single arg)
#define _CFFI_OP_CONSTANT       29
#define _CFFI_OP_CONSTANT_INT   31
#define _CFFI_OP_GLOBAL_VAR     33
#define _CFFI_OP_DLOPEN_FUNC    35
#define _CFFI_OP_DLOPEN_CONST   37
#define _CFFI_OP_GLOBAL_VAR_F   39
#define _CFFI_OP_EXTERN_PYTHON  41

#define _CFFI_PRIM_VOID          0
#define _CFFI_PRIM_BOOL          1
#define _CFFI_PRIM_CHAR          2
#define _CFFI_PRIM_SCHAR         3
#define _CFFI_PRIM_UCHAR         4
#define _CFFI_PRIM_SHORT         5
#define _CFFI_PRIM_USHORT        6
#define _CFFI_PRIM_INT           7
#define _CFFI_PRIM_UINT          8
#define _CFFI_PRIM_LONG          9
#define _CFFI_PRIM_ULONG        10
#define _CFFI_PRIM_LONGLONG     11
#define _CFFI_PRIM_ULONGLONG    12
#define _CFFI_PRIM_FLOAT        13
#define _CFFI_PRIM_DOUBLE       14
#define _CFFI_PRIM_LONGDOUBLE   15

#define _CFFI_PRIM_WCHAR        16
#define _CFFI_PRIM_INT8         17
#define _CFFI_PRIM_UINT8        18
#define _CFFI_PRIM_INT16        19
#define _CFFI_PRIM_UINT16       20
#define _CFFI_PRIM_INT32        21
#define _CFFI_PRIM_UINT32       22
#define _CFFI_PRIM_INT64        23
#define _CFFI_PRIM_UINT64       24
#define _CFFI_PRIM_INTPTR       25
#define _CFFI_PRIM_UINTPTR      26
#define _CFFI_PRIM_PTRDIFF      27
#define _CFFI_PRIM_SIZE         28
#define _CFFI_PRIM_SSIZE        29
#define _CFFI_PRIM_INT_LEAST8   30
#define _CFFI_PRIM_UINT_LEAST8  31
#define _CFFI_PRIM_INT_LEAST16  32
#define _CFFI_PRIM_UINT_LEAST16 33
#define _CFFI_PRIM_INT_LEAST32  34
#define _CFFI_PRIM_UINT_LEAST32 35
#define _CFFI_PRIM_INT_LEAST64  36
#define _CFFI_PRIM_UINT_LEAST64 37
#define _CFFI_PRIM_INT_FAST8    38
#define _CFFI_PRIM_UINT_FAST8   39
#define _CFFI_PRIM_INT_FAST16   40
#define _CFFI_PRIM_UINT_FAST16  41
#define _CFFI_PRIM_INT_FAST32   42
#define _CFFI_PRIM_UINT_FAST32  43
#define _CFFI_PRIM_INT_FAST64   44
#define _CFFI_PRIM_UINT_FAST64  45
#define _CFFI_PRIM_INTMAX       46
#define _CFFI_PRIM_UINTMAX      47
#define _CFFI_PRIM_FLOATCOMPLEX 48
#define _CFFI_PRIM_DOUBLECOMPLEX 49
#define _CFFI_PRIM_CHAR16       50
#define _CFFI_PRIM_CHAR32       51

#define _CFFI__NUM_PRIM         52
#define _CFFI__UNKNOWN_PRIM           (-1)
#define _CFFI__UNKNOWN_FLOAT_PRIM     (-2)
#define _CFFI__UNKNOWN_LONG_DOUBLE    (-3)

#define _CFFI__IO_FILE_STRUCT         (-1)


struct _cffi_global_s {
    const char *name;
    void *address;
    _cffi_opcode_t type_op;
    void *size_or_direct_fn;  // OP_GLOBAL_VAR: size, or 0 if unknown
                              // OP_CPYTHON_BLTN_*: addr of direct function
};

struct _cffi_getconst_s {
    unsigned long long value;
    const struct _cffi_type_context_s *ctx;
    int gindex;
};

struct _cffi_struct_union_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_STRUCT_UNION
    int flags;               // _CFFI_F_* flags below
    size_t size;
    int alignment;
    int first_field_index;   // -> _cffi_fields array
    int num_fields;
};
#define _CFFI_F_UNION         0x01   // is a union, not a struct
#define _CFFI_F_CHECK_FIELDS  0x02   // complain if fields are not in the
                                     // "standard layout" or if some are missing
#define _CFFI_F_PACKED        0x04   // for CHECK_FIELDS, assume a packed struct
#define _CFFI_F_EXTERNAL      0x08   // in some other ffi.include()
#define _CFFI_F_OPAQUE        0x10   // opaque

struct _cffi_field_s {
    const char *name;
    size_t field_offset;
    size_t field_size;
    _cffi_opcode_t field_type_op;
};

struct _cffi_enum_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_ENUM
    int type_prim;           // _CFFI_PRIM_xxx
    const char *enumerators; // comma-delimited string
};

struct _cffi_typename_s {
    const char *name;
    int type_index;   /* if opaque, points to a possibly artificial
                         OP_STRUCT which is itself opaque */
};

struct _cffi_type_context_s {
    _cffi_opcode_t *types;
    const struct _cffi_global_s *globals;
    const struct _cffi_field_s *fields;
    const struct _cffi_struct_union_s *struct_unions;
    const struct _cffi_enum_s *enums;
    const struct _cffi_typename_s *typenames;
    int num_globals;
    int num_struct_unions;
    int num_enums;
    int num_typenames;
    const char *const *includes;
    int num_types;
    int flags;      /* future extension */
};

struct _cffi_parse_info_s {
    const struct _cffi_type_context_s *ctx;
    _cffi_opcode_t *output;
    unsigned int output_size;
    size_t error_location;
    const char *error_message;
};

struct _cffi_externpy_s {
    const char *name;
    size_t size_of_result;
    void *reserved1, *reserved2;
};

#ifdef _CFFI_INTERNAL
static int parse_c_type(struct _cffi_parse_info_s *info, const char *input);
static int search_in_globals(const struct _cffi_type_context_s *ctx,
                             const char *search, size_t search_len);
static int search_in_struct_unions(const struct _cffi_type_context_s *ctx,
                                   const char *search, size_t search_len);
#endif

/* this block of #ifs should be kept exactly identical between
   c/_cffi_backend.c, cffi/vengine_cpy.py, cffi/vengine_gen.py
   and cffi/_cffi_include.h */
#if defined(_MSC_VER)
# include <malloc.h>   /* for alloca() */
# if _MSC_VER < 1600   /* MSVC < 2010 */
   typedef __int8 int8_t;
   typedef __int16 int16_t;
   typedef __int32 int32_t;
   typedef __int64 int64_t;
   typedef unsigned __int8 uint8_t;
   typedef unsigned __int16 uint16_t;
   typedef unsigned __int32 uint32_t;
   typedef unsigned __int64 uint64_t;
   typedef __int8 int_least8_t;
   typedef __int16 int_least16_t;
   typedef __int32 int_least32_t;
   typedef __int64 int_least64_t;
   typedef unsigned __int8 uint_least8_t;
   typedef unsigned __int16 uint_least16_t;
   typedef unsigned __int32 uint_least32_t;
   typedef unsigned __int64 uint_least64_t;
   typedef __int8 int_fast8_t;
   typedef __int16 int_fast16_t;
   typedef __int32 int_fast32_t;
   typedef __int64 int_fast64_t;
   typedef unsigned __int8 uint_fast8_t;
   typedef unsigned __int16 uint_fast16_t;
   typedef unsigned __int32 uint_fast32_t;
   typedef unsigned __int64 uint_fast64_t;
   typedef __int64 intmax_t;
   typedef unsigned __int64 uintmax_t;
# else
#  include <stdint.h>
# endif
# if _MSC_VER < 1800   /* MSVC < 2013 */
#  ifndef __cplusplus
    typedef unsigned char _Bool;
#  endif
# endif
# define _cffi_float_complex_t   _Fcomplex    /* include <complex.h> for it */
# define _cffi_double_complex_t  _Dcomplex    /* include <complex.h> for it */
#else
# include <stdint.h>
# if (defined (__SVR4) && defined (__sun)) || defined(_AIX) || defined(__hpux)
#  include <alloca.h>
# endif
# define _cffi_float_complex_t   float _Complex
# define _cffi_double_complex_t  double _Complex
#endif

#ifdef __GNUC__
# define _CFFI_UNUSED_FN  __attribute__((unused))
#else
# define _CFFI_UNUSED_FN  /* nothing */
#endif

#ifdef __cplusplus
# ifndef _Bool
   typedef bool _Bool;   /* semi-hackish: C++ has no _Bool; bool is builtin */
# endif
#endif

/**********  CPython-specific section  **********/
#ifndef PYPY_VERSION


#if PY_MAJOR_VERSION >= 3
# define PyInt_FromLong PyLong_FromLong
#endif

#define _cffi_from_c_double PyFloat_FromDouble
#define _cffi_from_c_float PyFloat_FromDouble
#define _cffi_from_c_long PyInt_FromLong
#define _cffi_from_c_ulong PyLong_FromUnsignedLong
#define _cffi_from_c_longlong PyLong_FromLongLong
#define _cffi_from_c_ulonglong PyLong_FromUnsignedLongLong
#define _cffi_from_c__Bool PyBool_FromLong

#define _cffi_to_c_double PyFloat_AsDouble
#define _cffi_to_c_float PyFloat_AsDouble

#define _cffi_from_c_int(x, type)                                        \
    (((type)-1) > 0 ? /* unsigned */                                     \
        (sizeof(type) < sizeof(long) ?                                   \
            PyInt_FromLong((long)x) :                                    \
         sizeof(type) == sizeof(long) ?                                  \
            PyLong_FromUnsignedLong((unsigned long)x) :                  \
            PyLong_FromUnsignedLongLong((unsigned long long)x)) :        \
        (sizeof(type) <= sizeof(long) ?                                  \
            PyInt_FromLong((long)x) :                                    \
            PyLong_FromLongLong((long long)x)))

#define _cffi_to_c_int(o, type)                                          \
    ((type)(                                                             \
     sizeof(type) == 1 ? (((type)-1) > 0 ? (type)_cffi_to_c_u8(o)        \
                                         : (type)_cffi_to_c_i8(o)) :     \
     sizeof(type) == 2 ? (((type)-1) > 0 ? (type)_cffi_to_c_u16(o)       \
                                         : (type)_cffi_to_c_i16(o)) :    \
     sizeof(type) == 4 ? (((type)-1) > 0 ? (type)_cffi_to_c_u32(o)       \
                                         : (type)_cffi_to_c_i32(o)) :    \
     sizeof(type) == 8 ? (((type)-1) > 0 ? (type)_cffi_to_c_u64(o)       \
                                         : (type)_cffi_to_c_i64(o)) :    \
     (Py_FatalError("unsupported size for type " #type), (type)0)))

#define _cffi_to_c_i8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[1])
#define _cffi_to_c_u8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[2])
#define _cffi_to_c_i16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[3])
#define _cffi_to_c_u16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[4])
#define _cffi_to_c_i32                                                   \
                 ((int(*)(PyObject *))_cffi_exports[5])
#define _cffi_to_c_u32                                                   \
                 ((unsigned int(*)(PyObject *))_cffi_exports[6])
#define _cffi_to_c_i64                                                   \
                 ((long long(*)(PyObject *))_cffi_exports[7])
#define _cffi_to_c_u64                                                   \
                 ((unsigned long long(*)(PyObject *))_cffi_exports[8])
#define _cffi_to_c_char                                                  \
                 ((int(*)(PyObject *))_cffi_exports[9])
#define _cffi_from_c_pointer                                             \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[10])
#define _cffi_to_c_pointer                                               \
    ((char *(*)(PyObject *, struct _cffi_ctypedescr *))_cffi_exports[11])
#define _cffi_get_struct_layout                                          \
    not used any more
#define _cffi_restore_errno                                              \
    ((void(*)(void))_cffi_exports[13])
#define _cffi_save_errno                                                 \
    ((void(*)(void))_cffi_exports[14])
#define _cffi_from_c_char                                                \
    ((PyObject *(*)(char))_cffi_exports[15])
#define _cffi_from_c_deref                                               \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[16])
#define _cffi_to_c                                                       \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[17])
#define _cffi_from_c_struct                                              \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[18])
#define _cffi_to_c_wchar_t                                               \
    ((_cffi_wchar_t(*)(PyObject *))_cffi_exports[19])
#define _cffi_from_c_wchar_t                                             \
    ((PyObject *(*)(_cffi_wchar_t))_cffi_exports[20])
#define _cffi_to_c_long_double                                           \
    ((long double(*)(PyObject *))_cffi_exports[21])
#define _cffi_to_c__Bool                                                 \
    ((_Bool(*)(PyObject *))_cffi_exports[22])
#define _cffi_prepare_pointer_call_argument                              \
    ((Py_ssize_t(*)(struct _cffi_ctypedescr *,                           \
                    PyObject *, char **))_cffi_exports[23])
#define _cffi_convert_array_from_object                                  \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[24])
#define _CFFI_CPIDX  25
#define _cffi_call_python                                                \
    ((void(*)(struct _cffi_externpy_s *, char *))_cffi_exports[_CFFI_CPIDX])
#define _cffi_to_c_wchar3216_t                                           \
    ((int(*)(PyObject *))_cffi_exports[26])
#define _cffi_from_c_wchar3216_t                                         \
    ((PyObject *(*)(int))_cffi_exports[27])
#define _CFFI_NUM_EXPORTS 28

struct _cffi_ctypedescr;

static void *_cffi_exports[_CFFI_NUM_EXPORTS];

#define _cffi_type(index)   (                           \
    assert((((uintptr_t)_cffi_types[index]) & 1) == 0), \
    (struct _cffi_ctypedescr *)_cffi_types[index])

static PyObject *_cffi_init(const char *module_name, Py_ssize_t version,
                            const struct _cffi_type_context_s *ctx)
{
    PyObject *module, *o_arg, *new_module;
    void *raw[] = {
        (void *)module_name,
        (void *)version,
        (void *)_cffi_exports,
        (void *)ctx,
    };

    module = PyImport_ImportModule("_cffi_backend");
    if (module == NULL)
        goto failure;

    o_arg = PyLong_FromVoidPtr((void *)raw);
    if (o_arg == NULL)
        goto failure;

    new_module = PyObject_CallMethod(
        module, (char *)"_init_cffi_1_0_external_module", (char *)"O", o_arg);

    Py_DECREF(o_arg);
    Py_DECREF(module);
    return new_module;

  failure:
    Py_XDECREF(module);
    return NULL;
}


#ifdef HAVE_WCHAR_H
typedef wchar_t _cffi_wchar_t;
#else
typedef uint16_t _cffi_wchar_t;   /* same random pick as _cffi_backend.c */
#endif

_CFFI_UNUSED_FN static uint16_t _cffi_to_c_char16_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return (uint16_t)_cffi_to_c_wchar_t(o);
    else
        return (uint16_t)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char16_t(uint16_t x)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

_CFFI_UNUSED_FN static int _cffi_to_c_char32_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return (int)_cffi_to_c_wchar_t(o);
    else
        return (int)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char32_t(unsigned int x)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

union _cffi_union_alignment_u {
    unsigned char m_char;
    unsigned short m_short;
    unsigned int m_int;
    unsigned long m_long;
    unsigned long long m_longlong;
    float m_float;
    double m_double;
    long double m_longdouble;
};

struct _cffi_freeme_s {
    struct _cffi_freeme_s *next;
    union _cffi_union_alignment_u alignment;
};

_CFFI_UNUSED_FN static int
_cffi_convert_array_argument(struct _cffi_ctypedescr *ctptr, PyObject *arg,
                             char **output_data, Py_ssize_t datasize,
                             struct _cffi_freeme_s **freeme)
{
    char *p;
    if (datasize < 0)
        return -1;

    p = *output_data;
    if (p == NULL) {
        struct _cffi_freeme_s *fp = (struct _cffi_freeme_s *)PyObject_Malloc(
            offsetof(struct _cffi_freeme_s, alignment) + (size_t)datasize);
        if (fp == NULL)
            return -1;
        fp->next = *freeme;
        *freeme = fp;
        p = *output_data = (char *)&fp->alignment;
    }
    memset((void *)p, 0, (size_t)datasize);
    return _cffi_convert_array_from_object(p, ctptr, arg);
}

_CFFI_UNUSED_FN static void
_cffi_free_array_arguments(struct _cffi_freeme_s *freeme)
{
    do {
        void *p = (void *)freeme;
        freeme = freeme->next;
        PyObject_Free(p);
    } while (freeme != NULL);
}

/**********  end CPython-specific section  **********/
#else
_CFFI_UNUSED_FN
static void (*_cffi_call_python_org)(struct _cffi_externpy_s *, char *);
# define _cffi_call_python  _cffi_call_python_org
#endif


#define _cffi_array_len(array)   (sizeof(array) / sizeof((array)[0]))

#define _cffi_prim_int(size, sign)                                      \
    ((size) == 1 ? ((sign) ? _CFFI_PRIM_INT8  : _CFFI_PRIM_UINT8)  :    \
     (size) == 2 ? ((sign) ? _CFFI_PRIM_INT16 : _CFFI_PRIM_UINT16) :    \
     (size) == 4 ? ((sign) ? _CFFI_PRIM_INT32 : _CFFI_PRIM_UINT32) :    \
     (size) == 8 ? ((sign) ? _CFFI_PRIM_INT64 : _CFFI_PRIM_UINT64) :    \
     _CFFI__UNKNOWN_PRIM)

#define _cffi_prim_float(size)                                          \
    ((size) == sizeof(float) ? _CFFI_PRIM_FLOAT :                       \
     (size) == sizeof(double) ? _CFFI_PRIM_DOUBLE :                     \
     (size) == sizeof(long double) ? _CFFI__UNKNOWN_LONG_DOUBLE :       \
     _CFFI__UNKNOWN_FLOAT_PRIM)

#define _cffi_check_int(got, got_nonpos, expected)      \
    ((got_nonpos) == (expected <= 0) &&                 \
     (got) == (unsigned long long)expected)

#ifdef MS_WIN32
# define _cffi_stdcall  __stdcall
#else
# define _cffi_stdcall  /* nothing */
#endif

#ifdef __cplusplus
}
#endif

/************************************************************/

/*

Copyright (c) 2022 Ole-Christoffer Granmo

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This code implements the Convolutional Tsetlin Machine from paper arXiv:1905.09688
https://arxiv.org/abs/1905.09688

*/

#ifdef _MSC_VER
#  include <intrin.h>
#  define __builtin_popcount __popcnt
#endif

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <string.h>
#include "fast_rand.h"

#include "ClauseBank.h"

static inline void cb_initialize_random_streams(unsigned int *feedback_to_ta, int number_of_literals, int number_of_ta_chunks, float s)
{
	// Initialize all bits to zero	
	memset(feedback_to_ta, 0, number_of_ta_chunks*sizeof(unsigned int));

	int n = number_of_literals;
	float p = 1.0 / s;

	int active = normal(n * p, n * p * (1 - p));
	active = active >= n ? n : active;
	active = active < 0 ? 0 : active;
	while (active--) {
		int f = fast_rand() % (number_of_literals);
		while (feedback_to_ta[f / 32] & (1 << (f % 32))) {
			f = fast_rand() % (number_of_literals);
	    }
		feedback_to_ta[f / 32] |= 1 << (f % 32);
	}
}

// Increment the states of each of those 32 Tsetlin Automata flagged in the active bit vector.
static inline void cb_inc(unsigned int *ta_state, unsigned int active, int number_of_state_bits)
{
	unsigned int carry, carry_next;

	carry = active;
	for (int b = 0; b < number_of_state_bits; ++b) {
		if (carry == 0)
			break;

		carry_next = ta_state[b] & carry; // Sets carry bits (overflow) passing on to next bit
		ta_state[b] = ta_state[b] ^ carry; // Performs increments with XOR
		carry = carry_next;
	}

	if (carry > 0) {
		for (int b = 0; b < number_of_state_bits; ++b) {
			ta_state[b] |= carry;
		}
	} 	
}

// Decrement the states of each of those 32 Tsetlin Automata flagged in the active bit vector.
static inline void cb_dec(unsigned int *ta_state, unsigned int active, int number_of_state_bits)
{
	unsigned int carry, carry_next;

	carry = active;
	for (int b = 0; b < number_of_state_bits; ++b) {
		if (carry == 0)
			break;

		carry_next = (~ta_state[b]) & carry; // Sets carry bits (overflow) passing on to next bit
		ta_state[b] = ta_state[b] ^ carry; // Performs increments with XOR
		carry = carry_next;
	}

	if (carry > 0) {
		for (int b = 0; b < number_of_state_bits; ++b) {
			ta_state[b] &= ~carry;
		}
	} 
}

/* Calculate the output of each clause using the actions of each Tsetline Automaton. */
static inline void cb_calculate_clause_output_feedback(unsigned int *ta_state, unsigned int *output_one_patches, unsigned int *clause_output, unsigned int *clause_patch, int number_of_ta_chunks, int number_of_state_bits, unsigned int filter, int number_of_patches, unsigned int *literal_active, unsigned int *Xi)
{
	int output_one_patches_count = 0;
	for (int patch = 0; patch < number_of_patches; ++patch) {
		unsigned int output = 1;
		for (int k = 0; k < number_of_ta_chunks-1; k++) {
			unsigned int pos = k*number_of_state_bits + number_of_state_bits-1;
			output = output && (ta_state[pos] & (Xi[patch*number_of_ta_chunks + k] | (~literal_active[k]))) == ta_state[pos];

			if (!output) {
				break;
			}
		}

		unsigned int pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
		output = output &&
			(ta_state[pos] & (Xi[patch*number_of_ta_chunks + number_of_ta_chunks - 1] | (~literal_active[number_of_ta_chunks - 1])) & filter) ==
			(ta_state[pos] & filter);

		if (output) {
			output_one_patches[output_one_patches_count] = patch;
			output_one_patches_count++;
		}
	}

	if (output_one_patches_count > 0) {
		*clause_output = 1;

		int patch_id = fast_rand() % output_one_patches_count;
 		*clause_patch = output_one_patches[patch_id];
	} else {
		*clause_output = 0;
	}
}

/* Calculate the output of each clause using the actions of each Tsetline Automaton. */
static inline int cb_calculate_clause_output_single_false_literal(unsigned int *ta_state, unsigned int *candidate_offending_literals, int number_of_ta_chunks, int number_of_state_bits, unsigned int filter, int number_of_patches, unsigned int *literal_active, unsigned int *Xi)
{
	int offending_literals_count = 0;
	int offending_literal_id = 0;
	for (int patch = 0; patch < number_of_patches; ++patch) {
		unsigned int max_one_offending_literal = 1;
		unsigned int already_one_offending_literal = 0;

		for (int k = 0; k < number_of_ta_chunks-1; k++) {
			unsigned int pos = k*number_of_state_bits + number_of_state_bits-1;
			unsigned int offending_literals = (ta_state[pos] & (Xi[patch*number_of_ta_chunks + k] | (~literal_active[k]))) ^ ta_state[pos];
			if ((offending_literals & (offending_literals - 1)) > 0) {
				max_one_offending_literal = 0;
				break;
			} else if (offending_literals != 0) {
				if (!already_one_offending_literal) {
					already_one_offending_literal = 1;
					offending_literal_id = log2(offending_literals);
				} else {
					max_one_offending_literal = 0;
					break;
				}
			}
		}

		unsigned int pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
		unsigned int offending_literals = (ta_state[pos] & (Xi[patch*number_of_ta_chunks + number_of_ta_chunks - 1] | (~literal_active[number_of_ta_chunks - 1])) & filter) ^ (ta_state[pos] & filter);
		if ((offending_literals & (offending_literals - 1)) > 0) {
			max_one_offending_literal = 0;
			break;
		} else if (offending_literals != 0) {
			if (!already_one_offending_literal) {
				already_one_offending_literal = 1;
				offending_literal_id = log2(offending_literals);
			} else {
				max_one_offending_literal = 0;
				break;
			}
		}

		if (max_one_offending_literal && already_one_offending_literal) {
			candidate_offending_literals[offending_literals_count] = offending_literal_id;
			offending_literals_count++;
		}
	}

	if (offending_literals_count > 0) {
		int offending_literal_pos = fast_rand() % offending_literals_count;
 		return(candidate_offending_literals[offending_literal_pos]);
	} else {
		return(-1);
	}
}

static inline unsigned int cb_calculate_clause_output_update(unsigned int *ta_state, int number_of_ta_chunks, int number_of_state_bits, unsigned int filter, int number_of_patches, unsigned int *literal_active, unsigned int *Xi)
{
	for (int patch = 0; patch < number_of_patches; ++patch) {
		unsigned int output = 1;
		for (int k = 0; k < number_of_ta_chunks-1; k++) {
			unsigned int pos = k*number_of_state_bits + number_of_state_bits-1;
			output = output && (ta_state[pos] & (Xi[patch*number_of_ta_chunks + k] | (~literal_active[k]))) == ta_state[pos];

			if (!output) {
				break;
			}
		}

		unsigned int pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
		output = output &&
			(ta_state[pos] & (Xi[patch*number_of_ta_chunks + number_of_ta_chunks - 1] | (~literal_active[number_of_ta_chunks - 1])) & filter) ==
			(ta_state[pos] & filter);

		if (output) {
			return(1);
		}
	}

	return(0);
}

static inline void cb_calculate_clause_output_patchwise(unsigned int *ta_state, int number_of_ta_chunks, int number_of_state_bits, unsigned int filter, int number_of_patches, unsigned int *output, unsigned int *Xi)
{
	for (int patch = 0; patch < number_of_patches; ++patch) {
		output[patch] = 1;
		for (int k = 0; k < number_of_ta_chunks-1; k++) {
			unsigned int pos = k*number_of_state_bits + number_of_state_bits-1;
			output[patch] = output[patch] && (ta_state[pos] & Xi[patch*number_of_ta_chunks + k]) == ta_state[pos];

			if (!output[patch]) {
				break;
			}
		}

		unsigned int pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
		output[patch] = output[patch] &&
			(ta_state[pos] & Xi[patch*number_of_ta_chunks + number_of_ta_chunks - 1] & filter) ==
			(ta_state[pos] & filter);
	}

	return;
}

static inline unsigned int cb_calculate_clause_output_predict(unsigned int *ta_state, int number_of_ta_chunks, int number_of_state_bits, unsigned int filter, int number_of_patches, unsigned int *Xi)
{
	for (int patch = 0; patch < number_of_patches; ++patch) {
		unsigned int output = 1;
		unsigned int all_exclude = 1;
		for (int k = 0; k < number_of_ta_chunks-1; k++) {
			unsigned int pos = k*number_of_state_bits + number_of_state_bits-1;
			output = output && (ta_state[pos] & Xi[patch*number_of_ta_chunks + k]) == ta_state[pos];

			if (!output) {
				break;
			}
			all_exclude = all_exclude && (ta_state[pos] == 0);
		}

		unsigned int pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
		output = output &&
			(ta_state[pos] & Xi[patch*number_of_ta_chunks + number_of_ta_chunks - 1] & filter) ==
			(ta_state[pos] & filter);

		all_exclude = all_exclude && ((ta_state[pos] & filter) == 0);

		if (output && all_exclude == 0) {
			return(1);
		}
	}

	return(0);
}


void cb_type_i_feedback(
        unsigned int *ta_state,
        unsigned int *feedback_to_ta,
        unsigned int *output_one_patches,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        int number_of_patches,
        float update_p,
        float s,
        unsigned int boost_true_positive_feedback,
        unsigned int reuse_random_feedback,
        unsigned int max_included_literals,
        unsigned int *clause_active,
        unsigned int *literal_active,
        unsigned int *Xi
)
{
    // Lage mask/filter
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	if (reuse_random_feedback) {
		cb_initialize_random_streams(feedback_to_ta, number_of_literals, number_of_ta_chunks, s);
	}

	for (int j = 0; j < number_of_clauses; ++j) {
		if ((((float)fast_rand())/((float)FAST_RAND_MAX) > update_p) || (!clause_active[j])) {
			continue;
		}

		unsigned int clause_pos = j*number_of_ta_chunks*number_of_state_bits;

		unsigned int clause_output;
		unsigned int clause_patch;

		cb_calculate_clause_output_feedback(&ta_state[clause_pos], output_one_patches, &clause_output, &clause_patch, number_of_ta_chunks, number_of_state_bits, filter, number_of_patches, literal_active, Xi);

		if (!reuse_random_feedback) {
			cb_initialize_random_streams(feedback_to_ta, number_of_literals, number_of_ta_chunks, s);
		}

		if (clause_output && cb_number_of_include_actions(ta_state, j, number_of_literals, number_of_state_bits) <= max_included_literals) {
			// Type Ia Feedback
			for (int k = 0; k < number_of_ta_chunks; ++k) {
				unsigned int ta_pos = k*number_of_state_bits;

				if (boost_true_positive_feedback == 1) {
	 				cb_inc(&ta_state[clause_pos + ta_pos], literal_active[k] & Xi[clause_patch*number_of_ta_chunks + k], number_of_state_bits);
				} else {
					cb_inc(&ta_state[clause_pos + ta_pos], literal_active[k] & Xi[clause_patch*number_of_ta_chunks + k] & (~feedback_to_ta[k]), number_of_state_bits);
				}

	 			cb_dec(&ta_state[clause_pos + ta_pos], literal_active[k] & (~Xi[clause_patch*number_of_ta_chunks + k]) & feedback_to_ta[k], number_of_state_bits);
			}
		} else {
			// Type Ib Feedback
				
			for (int k = 0; k < number_of_ta_chunks; ++k) {
				unsigned int ta_pos = k*number_of_state_bits;

				cb_dec(&ta_state[clause_pos + ta_pos], literal_active[k] & feedback_to_ta[k], number_of_state_bits);
			}
		}
	}
}

void cb_type_ii_feedback(
        unsigned int *ta_state,
        unsigned int *output_one_patches,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        int number_of_patches,
        float update_p,
        unsigned int *clause_active,
        unsigned int *literal_active,
        unsigned int *Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int j = 0; j < number_of_clauses; j++) {
		if ((((float)fast_rand())/((float)FAST_RAND_MAX) > update_p) || (!clause_active[j])) {
			continue;
		}

		unsigned int clause_pos = j*number_of_ta_chunks*number_of_state_bits;

		unsigned int clause_output;
		unsigned int clause_patch;
		cb_calculate_clause_output_feedback(&ta_state[clause_pos], output_one_patches, &clause_output, &clause_patch, number_of_ta_chunks, number_of_state_bits, filter, number_of_patches, literal_active, Xi);

		if (clause_output) {				
			for (int k = 0; k < number_of_ta_chunks; ++k) {
				unsigned int ta_pos = k*number_of_state_bits;
				cb_inc(&ta_state[clause_pos + ta_pos], literal_active[k] & (~Xi[clause_patch*number_of_ta_chunks + k]), number_of_state_bits);
			}
		}
	}
}

void cb_type_iii_feedback(
        unsigned int *ta_state,
        unsigned int *ind_state,
        unsigned int *clause_and_target,
        unsigned int *output_one_patches,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits_ta,
        int number_of_state_bits_ind,
        int number_of_patches,
        float update_p,
        float d,
        unsigned int *clause_active,
        unsigned int *literal_active,
        unsigned int *Xi,
        unsigned int target
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int j = 0; j < number_of_clauses; ++j) {
		if ((!clause_active[j])) {
			continue;
		}

		unsigned int clause_pos_ta = j*number_of_ta_chunks*number_of_state_bits_ta;
		unsigned int clause_pos_ind = j*number_of_ta_chunks*number_of_state_bits_ind;

		unsigned int clause_output;
		unsigned int clause_patch;
		cb_calculate_clause_output_feedback(
		    &ta_state[clause_pos_ta],
		    output_one_patches,
		    &clause_output,
		    &clause_patch,
		    number_of_ta_chunks,
		    number_of_state_bits_ta,
		    filter,
		    number_of_patches,
		    literal_active,
		    Xi
        );

		if (clause_output) {
			if (target) {
				if (((float)fast_rand())/((float)FAST_RAND_MAX) <= (1.0 - 1.0/d)) {
					for (int k = 0; k < number_of_ta_chunks; ++k) {

						unsigned int ind_pos = k*number_of_state_bits_ind;
						cb_inc(
						    &ind_state[clause_pos_ind + ind_pos],
						    literal_active[k] & clause_and_target[j * number_of_ta_chunks + k] & Xi[clause_patch * number_of_ta_chunks + k],
						    number_of_state_bits_ind
                        );
					}
				}
			}

			for (int k = 0; k < number_of_ta_chunks; ++k) {
				unsigned int ind_pos = k*number_of_state_bits_ind;
				// Decrease if clause is true and literal is true
				cb_dec(
                    &ind_state[clause_pos_ind + ind_pos],
                    literal_active[k] & (~clause_and_target[j * number_of_ta_chunks + k]) & Xi[clause_patch*number_of_ta_chunks + k],
                    number_of_state_bits_ind);
			}

			// Invert literals
			for (int k = 0; k < number_of_ta_chunks; ++k) {
				unsigned int remove;
				if (target) {
				 	remove = clause_and_target[j*number_of_ta_chunks + k];
				} else {
					remove = 0;
				}
				unsigned int add = ~clause_and_target[j*number_of_ta_chunks + k];
				clause_and_target[j*number_of_ta_chunks + k] |= add;
				clause_and_target[j*number_of_ta_chunks + k] &= (~remove);
			}
		}

		// Included
		if (!clause_output) {
			int offending_literal = cb_calculate_clause_output_single_false_literal(&ta_state[clause_pos_ta], output_one_patches, number_of_ta_chunks, number_of_state_bits_ta, filter, number_of_patches, literal_active, Xi);
			if (offending_literal != - 1) {
				unsigned int ta_chunk = offending_literal / 32;
				unsigned int ta_pos = offending_literal % 32;

				if ((clause_and_target[j*number_of_ta_chunks + ta_chunk] & (1 << ta_pos)) == 0) {
					clause_and_target[j*number_of_ta_chunks + ta_chunk] |= (1 << ta_pos);
				} else if (target) {
					clause_and_target[j*number_of_ta_chunks + ta_chunk] &= (~(1 << ta_pos));
				}
			}
		}

		if ((((float)fast_rand())/((float)FAST_RAND_MAX) > update_p) || (!clause_active[j])) {
			continue;
		}

		for (int k = 0; k < number_of_ta_chunks; ++k) {
			unsigned int ta_pos = k*number_of_state_bits_ta;
			unsigned int ind_pos = k*number_of_state_bits_ind;

			cb_dec(
			    &ta_state[clause_pos_ta + ta_pos],
			    literal_active[k] & (~ind_state[clause_pos_ind + ind_pos + number_of_state_bits_ind - 1]),
			    number_of_state_bits_ta
            );
		}


	}
}

void cb_calculate_clause_outputs_predict(
        unsigned int *ta_state,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        int number_of_patches,
        unsigned int *clause_output,
        unsigned int *Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int j = 0; j < number_of_clauses; j++) {
		unsigned int clause_pos = j*number_of_ta_chunks*number_of_state_bits;
		clause_output[j] = cb_calculate_clause_output_predict(&ta_state[clause_pos], number_of_ta_chunks, number_of_state_bits, filter, number_of_patches, Xi);
	}
}

void cb_initialize_incremental_clause_calculation(
        unsigned int *ta_state,
        unsigned int *literal_clause_map,
        unsigned int *literal_clause_map_pos,
        unsigned int *false_literals_per_clause,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        unsigned int *previous_Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	// Initialize all literals as false for the previous example per patch
	for (int k = 0; k < number_of_ta_chunks; ++k) {
		previous_Xi[k] = 0;
	}

	// Initialize all false literal counters to 0 per patch
	for (int j = 0; j < number_of_clauses; ++j) {
		false_literals_per_clause[j] = 0;
	}

	// Build the literal clause map, and update the false literal counters
	// Start filling out the map from position 0
	unsigned int pos = 0;
	for (int k = 0; k < number_of_literals; ++k) {
		unsigned int ta_chunk = k / 32;
		unsigned int chunk_pos = k % 32;

		// For each literal, find out which clauses includes it
		for (int j = 0; j < number_of_clauses; ++j) {	
			// Obtain the clause ta chunk containing the literal decision (exclude/include)
			unsigned int clause_ta_chunk = j * number_of_ta_chunks * number_of_state_bits + ta_chunk * number_of_state_bits + number_of_state_bits - 1;
			if (ta_state[clause_ta_chunk] & (1 << chunk_pos)) {
				// Literal k included in clause j
				literal_clause_map[pos] = j;

				++false_literals_per_clause[j];
				++pos;
			}
		}
		literal_clause_map_pos[k] = pos;
	}

	// Make empty clauses false
	for (int j = 0; j < number_of_clauses; ++j) {
		if (false_literals_per_clause[j] == 0) {
			false_literals_per_clause[j] = 1;
		}
	}
}



// This function retrieves the count of literals from the given Tsetlin Automaton state.
// ta_state: an array representing the state of the Tsetlin Automaton.
// number_of_clauses: the total number of clauses in the TA.
// number_of_literals: the total number of literals in the TA.
// number_of_state_bits: the number of bits used to represent each state in the TA.
// result: an array to store the count of each literal.
void cb_get_literals(
    const unsigned int *ta_state,
    unsigned int number_of_clauses,
    unsigned int number_of_literals,
    unsigned int number_of_state_bits,
    unsigned int *result
){
    // Calculate the number of chunks required to represent all literals.
    unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

    // Iterate through all the clauses.
    for (unsigned int j = 0; j < number_of_clauses; j++) {
        // Iterate through all the literals.
        for (unsigned int k = 0; k < number_of_literals; k++) {

            // Determine which chunk the literal is in and its position within the chunk.
            unsigned int ta_chunk = k / 32;
            unsigned int chunk_pos = k % 32;

            // Calculate the position of the literal in the TA state array.
            unsigned int pos = j * number_of_ta_chunks * number_of_state_bits + ta_chunk * number_of_state_bits + number_of_state_bits-1;

            // Check if the literal is present (bit is set) in the TA state array.
            if ((ta_state[pos] & (1 << chunk_pos)) > 0) {
                // Increment the count of the literal in the result array.
                unsigned int result_pos = j * number_of_literals + k;
                result[result_pos] = 1;
            }
        }
    }
}

void cb_calculate_clause_outputs_incremental_batch(
        unsigned int * literal_clause_map,
        unsigned int *literal_clause_map_pos,
        unsigned int *false_literals_per_clause,
        int number_of_clauses,
        int number_of_literals,
        int number_of_patches,
        unsigned int *clause_output,
        unsigned int *previous_Xi,
        unsigned int *Xi,
        int batch_size
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	unsigned int *current_Xi = Xi;
	for (int b = 0; b < batch_size; ++b) {
		for (int j = 0; j < number_of_clauses; ++j) {
			clause_output[b*number_of_clauses + j] = 0;
		}

		for (int patch = 0; patch < number_of_patches; ++patch) {
			cb_calculate_clause_outputs_incremental(literal_clause_map, literal_clause_map_pos, false_literals_per_clause, number_of_clauses, number_of_literals, previous_Xi, current_Xi);
			for (int j = 0; j < number_of_clauses; ++j) {
				if (false_literals_per_clause[j] == 0) {
					clause_output[b*number_of_clauses + j] = 1;
				}
			}
			current_Xi += number_of_ta_chunks;
		}
	}
}

void cb_calculate_clause_outputs_incremental(
        unsigned int * literal_clause_map,
        unsigned int *literal_clause_map_pos,
        unsigned int *false_literals_per_clause,
        int number_of_clauses,
        int number_of_literals,
        unsigned int *previous_Xi,
        unsigned int *Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	// Look up each in literal clause map
	unsigned int start_pos = 0;
	for (int k = 0; k < number_of_literals; ++k) {
		unsigned int ta_chunk = k / 32;
		unsigned int chunk_pos = k % 32;

		// Check which literals have changed
		if ((Xi[ta_chunk] & (1 << chunk_pos)) && !(previous_Xi[ta_chunk] & (1 << chunk_pos))) {
			// If the literal now is True, decrement the false literal counter of all clauses including the literal
			for (int j = 0; j < literal_clause_map_pos[k] - start_pos; ++j) {
				--false_literals_per_clause[literal_clause_map[start_pos + j]];
			}
		} else if (!(Xi[ta_chunk] & (1 << chunk_pos)) && (previous_Xi[ta_chunk] & (1 << chunk_pos))) {
			// If the literal now is False, increment the false counter of all clauses including literal
			for (int j = 0; j < literal_clause_map_pos[k] - start_pos; ++j) {
				++false_literals_per_clause[literal_clause_map[start_pos + j]];
			}
		}

		start_pos = literal_clause_map_pos[k];
	}

	// Copy current Xi to previous_Xi
	for (int k = 0; k < number_of_ta_chunks; ++k) {
		previous_Xi[k] = Xi[k];
	}
}

void cb_calculate_clause_outputs_update(
        unsigned int *ta_state,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        int number_of_patches,
        unsigned int *clause_output,
        unsigned int *literal_active,
        unsigned int *Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}

	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int j = 0; j < number_of_clauses; j++) {
		unsigned int clause_pos = j*number_of_ta_chunks*number_of_state_bits;
		clause_output[j] = cb_calculate_clause_output_update(&ta_state[clause_pos], number_of_ta_chunks, number_of_state_bits, filter, number_of_patches, literal_active, Xi);
	}
}

void cb_calculate_clause_outputs_patchwise(
        unsigned int *ta_state,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        int number_of_patches,
        unsigned int *clause_output,
        unsigned int *Xi
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}

	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int j = 0; j < number_of_clauses; j++) {
		unsigned int clause_pos = j*number_of_ta_chunks*number_of_state_bits;
		cb_calculate_clause_output_patchwise(&ta_state[clause_pos], number_of_ta_chunks, number_of_state_bits, filter, number_of_patches, &clause_output[j*number_of_patches], Xi);
	}
}

void cb_calculate_literal_frequency(
        unsigned int *ta_state,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        unsigned int *clause_active,
        unsigned int *literal_count
)
{
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int k = 0; k < number_of_literals; k++) {
		literal_count[k] = 0;
	}
	
	for (int j = 0; j < number_of_clauses; j++) {
		if ((!clause_active[j])) {
			continue;
		}

		for (int k = 0; k < number_of_literals; k++) {
			unsigned int ta_chunk = k / 32;
			unsigned int chunk_pos = k % 32;
			unsigned int pos = j * number_of_ta_chunks * number_of_state_bits + ta_chunk * number_of_state_bits + number_of_state_bits-1;
			if ((ta_state[pos] & (1 << chunk_pos)) > 0) {
				literal_count[k] += 1;
			}
		}
	}
}

void cb_included_literals(
        unsigned int *ta_state,
        int number_of_clauses,
        int number_of_literals,
        int number_of_state_bits,
        unsigned int *actions
)
{
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

	for (int k = 0; k < number_of_ta_chunks; k++) {
		actions[k] = 0;
	}
	
	for (int j = 0; j < number_of_clauses; j++) {	
		for (int k = 0; k < number_of_ta_chunks; k++) {
			unsigned int pos = j * number_of_ta_chunks * number_of_state_bits + k * number_of_state_bits + number_of_state_bits-1;
			actions[k] |= ta_state[pos];
		}
	}
}

int cb_number_of_include_actions(
        unsigned int *ta_state,
        int clause,
        int number_of_literals,
        int number_of_state_bits
)
{
	unsigned int filter;
	if (((number_of_literals) % 32) != 0) {
		filter  = (~(0xffffffff << ((number_of_literals) % 32)));
	} else {
		filter = 0xffffffff;
	}
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;
	
	unsigned int clause_pos = clause*number_of_ta_chunks*number_of_state_bits;

	int number_of_include_actions = 0;
	for (int k = 0; k < number_of_ta_chunks-1; ++k) {
		unsigned int ta_pos = k*number_of_state_bits + number_of_state_bits-1;
		number_of_include_actions += __builtin_popcount(ta_state[clause_pos + ta_pos]);
	}
	unsigned int ta_pos = (number_of_ta_chunks-1)*number_of_state_bits + number_of_state_bits-1;
	number_of_include_actions += __builtin_popcount(ta_state[clause_pos + ta_pos] & filter);

	return(number_of_include_actions);
}
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdbool.h>
#include "Tools.h"

int enable_printing = 0;

void myPrint(FILE *file, const char* format, ...)
{
	if (enable_printing == 1)
	{
    	va_list args;
		va_start(args, format);
		vfprintf(file, format, args);

		va_end(args);
	}
}

int generate_random(int rows){
	return rand() % rows;
}

unsigned int compareints(const void * a, const void * b)
{
  return(*(unsigned int*)a- *(unsigned int*)b);
}

void store_feature_to_X(int feature, int number_of_features, unsigned int *X, int output_pos)
{
    int chunk_nr = feature / 32;
    int chunk_pos = feature % 32;
    X[output_pos + chunk_nr] |= (1U << chunk_pos);

    chunk_nr = (feature + number_of_features) / 32;
    chunk_pos = (feature + number_of_features) % 32;
    X[output_pos + chunk_nr] &= ~(1U << chunk_pos);
}
/*

Copyright (c) 2023 Ole-Christoffer Granmo

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This code implements the Convolutional Tsetlin Machine from paper arXiv:1905.09688
https://arxiv.org/abs/1905.09688

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "AutoencoderDocuments.h"
#include "Tools.h"
#include <stdarg.h>
#include <stdbool.h>

typedef struct {
    unsigned int value;
    unsigned int index;
} IndexedValue;

unsigned int compareIndexedValues(const void *a, const void *b) {
    return ((IndexedValue *)a)->value - ((IndexedValue *)b)->value;
}

void store_document_to_X(int row, int output_pos, unsigned int *indptr_row, unsigned int *indices_row, int number_of_cols, unsigned int *X){
	for (int k = indptr_row[row]; k < indptr_row[row+1]; ++k) {	
		store_feature_to_X(indices_row[k], number_of_cols, X, output_pos);
	}
}

void produce_example_by_documents(
        unsigned int *active_output,
        int number_of_active_outputs,
        unsigned int *indptr_row,
        unsigned int *indices_row,
        int number_of_rows,
        unsigned int *indptr_col,
        unsigned int *indices_col,
        int number_of_cols,
        unsigned int *X,
		unsigned int *Y, 
        int accumulation,
		unsigned int *data_col,
		int categories,
		int random_per_category,
		int expert_start_index,
		int expert_end_index,
		int enable_log
)
{
	FILE* file = fopen("result/output.txt", "a");
	if (file != NULL) {
		enable_printing = enable_log;
		myPrint(file, "\nStart new example with expert_start_index = %d, and expert_end_index = %d\n",expert_start_index,expert_end_index);
		
		int row;

		int number_of_features = number_of_cols;
		int number_of_literals = 2*number_of_features;
		unsigned int number_of_literal_chunks = (number_of_literals-1)/32 + 1;

		// Loop over active outputs, producing one example per output
		for (int o = 0; o < number_of_active_outputs; ++o) {
			int output_pos = o*number_of_literal_chunks;
			myPrint(file, "This loop for target word: %d\n", active_output[o]);

			// Initialize example with false features
			int	number_of_feature_chunks = (((number_of_literals-1)/32 + 1));
			for (int k = 0; k < number_of_feature_chunks - 1; ++k) {
				X[output_pos + k] = 0U;
			}

			for (int k = number_of_feature_chunks - 1; k < number_of_literal_chunks; ++k) {
				X[output_pos + k] = ~0U;
			}

			for (int k = (number_of_feature_chunks-1)*32; k < number_of_cols; ++k) {
				int chunk_nr = k / 32;
				int chunk_pos = k % 32;
				X[output_pos + chunk_nr] &= ~(1U << chunk_pos);
			}

			if ((indptr_col[active_output[o]+1] - indptr_col[active_output[o]] == 0) || (indptr_col[active_output[o]+1] - indptr_col[active_output[o]] == number_of_rows)) {
				// If no positive/negative examples, produce a random example
				for (int a = 0; a < accumulation; ++a) {
					row = generate_random(number_of_rows);
					store_document_to_X(row, output_pos, indptr_row,indices_row,number_of_cols,X);
				}
			}

			if (indptr_col[active_output[o]+1] - indptr_col[active_output[o]] == 0) {
				// If no positive examples, produce a negative output value
				Y[o] = 0;
				continue;
			} else if (indptr_col[active_output[o]+1] - indptr_col[active_output[o]] == number_of_rows) {
				// If no negative examples, produce a positive output value
				Y[o] = 1;
				continue;
			} 
			
			// Randomly select either positive or negative example
			Y[o] = rand() % 2;
		
			if (Y[o]) {
				int start_index = indptr_col[active_output[o]];
				int end_index = indptr_col[active_output[o]+1];
				int total_rows = (end_index - start_index);

				if (categories > 0 && total_rows >= accumulation)
				{
					IndexedValue *indexed_data = malloc(total_rows * sizeof(IndexedValue));

					for (int i = 0; i < total_rows; i++) {
						indexed_data[i].value = data_col[start_index + i];
						indexed_data[i].index = start_index + i;
					}

					qsort(indexed_data, total_rows, sizeof(IndexedValue), compareIndexedValues);

					int size_per_category = accumulation / categories;
					// last category with most frequent words
					int category_start_index = size_per_category * (categories - 1);
					for (int a = 0; a < accumulation; ++a) {
						if (random_per_category)
						{
							//pick one by one without rondomize inside each category
							int random_index_data = category_start_index + (rand() % accumulation);
							int random_index = indexed_data[random_index_data].index;
							row = indices_col[random_index];	
						}
						else{
							row = indices_col[indexed_data[a + category_start_index].index];
						}
						store_document_to_X(row,output_pos,indptr_row,indices_row,number_of_features,X);
					}
					// for (int category = 1; category <= categories; category++) {
					// 	category_start_index = category_start_index + size_per_category;
					// }			
					free(indexed_data);
				}
				else
				{
                    myPrint(file, "No Categories and Experts enabled.\n");
                    int expert_size = expert_end_index - expert_start_index;
					if (expert_size > 0)
					{
						int *expert_rows = (int *) malloc(expert_size * sizeof(int));
						if (expert_rows == NULL) {
							myPrint(file, "Memory allocation failed.\n");
							return;
						}
						//counter for document within particular expert
						int expert_rows_index = 0;
						for (int i = start_index; i < end_index; i++) {
							int target_row = indices_col[i];
							if (target_row >= expert_start_index && target_row < expert_end_index) {
								expert_rows[expert_rows_index] = target_row;
								expert_rows_index++;
							}
						}
						if (expert_rows_index == 0) {
							myPrint(file, "No valid target rows found.\n");
							free(expert_rows);
							fclose(file);
							return;
						}
						else{
							myPrint(file, "Number of experts rows founded: %d\n", expert_rows_index);
						}
						for (int a = 0; a < accumulation; ++a) {
							int random_index = rand() % expert_rows_index;
							row = expert_rows[random_index];
							myPrint(file, "will take document %d from my expert_rows whcih is row number %d in X_train\n", random_index,row);
							store_document_to_X(row, output_pos, indptr_row,indices_row,number_of_cols,X);
						}
						free(expert_rows);
					}
					else{
						myPrint(file, "start accumlation randomly\n");
						for (int a = 0; a < accumulation; ++a) {
							// Pick example randomly among positive examples
							int random_index = start_index + (rand() % (end_index - start_index));
							row = indices_col[random_index];
							store_document_to_X(row, output_pos, indptr_row,indices_row,number_of_cols,X);
						}
						
					}
				}
			} else {
				int a = 0;
				while (a < accumulation) {
					row = rand() % number_of_rows;

					if (bsearch(&row, &indices_col[indptr_col[active_output[o]]], indptr_col[active_output[o]+1] - indptr_col[active_output[o]], sizeof(unsigned int), compareints) == NULL) {
						store_document_to_X(row, output_pos, indptr_row,indices_row,number_of_cols,X);
						a++;
					}
				}
			}
		}
		fclose(file);
	}
}

void produce_example_per_target_by_document(
        unsigned int *active_output,
        int number_of_active_outputs,
        unsigned int *indptr_row,
        unsigned int *indices_row,
        int number_of_rows,
        unsigned int *indptr_col,
        unsigned int *indices_col,
        int number_of_cols,
        unsigned int *X,
        int target,
        int target_value,
        int accumulation
)
{
	int row;

	int number_of_features = number_of_cols;
	int number_of_literals = 2*number_of_features;

	unsigned int number_of_literal_chunks = (number_of_literals-1)/32 + 1;

	// Initialize example vector X
	memset(X, 0, number_of_literal_chunks * sizeof(unsigned int));
	for (int k = number_of_features; k < number_of_literals; ++k) {
		int chunk_nr = k / 32;
		int chunk_pos = k % 32;
		X[chunk_nr] |= (1U << chunk_pos);
	}
	
	if ((indptr_col[active_output[target]+1] - indptr_col[active_output[target]] == 0) || (indptr_col[active_output[target]+1] - indptr_col[active_output[target]] == number_of_rows)) {
		// If no positive/negative examples, produce a random example
		for (int a = 0; a < accumulation; ++a) {
			row = rand() % number_of_rows;
			store_document_to_X(row,0,indptr_row,indices_row,number_of_features,X);
		}
		return;
	}

	if (target_value) {
		for (int a = 0; a < accumulation; ++a) {
			// Pick example randomly among positive examples
			int random_index = indptr_col[active_output[target]] + (rand() % (indptr_col[active_output[target]+1] - indptr_col[active_output[target]]));
			row = indices_col[random_index];
			store_document_to_X(row,0,indptr_row,indices_row,number_of_features,X);
		}
	} else {
		int a = 0;
		while (a < accumulation) {
			row = rand() % number_of_rows;

			if (bsearch(&row, &indices_col[indptr_col[active_output[target]]], indptr_col[active_output[target]+1] - indptr_col[active_output[target]], sizeof(unsigned int), compareints) == NULL) {
				store_document_to_X(row,0,indptr_row,indices_row,number_of_features,X);
				a++;
			}
		}
	}
}


void tmu_encode(
        unsigned int *X,
        unsigned int *encoded_X,
        int number_of_examples,
        int dim_x,
        int dim_y,
        int dim_z,
        int patch_dim_x,
        int patch_dim_y,
        int append_negated,
        int class_features
)
{
	int global_number_of_features = dim_x * dim_y * dim_z;
	int number_of_features = class_features + patch_dim_x * patch_dim_y * dim_z + (dim_x - patch_dim_x) + (dim_y - patch_dim_y);
	int number_of_patches = (dim_x - patch_dim_x + 1) * (dim_y - patch_dim_y + 1);

	int number_of_literal_chunks;
	if (append_negated) {
		number_of_literal_chunks= (((2*number_of_features-1)/32 + 1));
	} else {
		number_of_literal_chunks= (((number_of_features-1)/32 + 1));
	}

	unsigned int *Xi;
	unsigned int *encoded_Xi;

	unsigned int input_pos = 0;
	unsigned int input_step_size = global_number_of_features;

	// Fill encoded_X with zeros

	memset(encoded_X, 0, number_of_examples * number_of_patches * number_of_literal_chunks * sizeof(unsigned int));

	unsigned int encoded_pos = 0;
	for (int i = 0; i < number_of_examples; ++i) {
		//printf("%d\n", i);

		int patch_nr = 0;
		// Produce the patches of the current image
		for (int y = 0; y < dim_y - patch_dim_y + 1; ++y) {
			for (int x = 0; x < dim_x - patch_dim_x + 1; ++x) {
				Xi = &X[input_pos];
				encoded_Xi = &encoded_X[encoded_pos];

				// Encode class into feature vector 
				for (int class_feature = 0; class_feature < class_features; ++class_feature) {

					int chunk_nr = (class_feature + number_of_features) / 32;
					int chunk_pos = (class_feature + number_of_features) % 32;
					encoded_Xi[chunk_nr] |= (1 << chunk_pos);
				}

				// Encode y coordinate of patch into feature vector 
				for (int y_threshold = 0; y_threshold < dim_y - patch_dim_y; ++y_threshold) {
					int patch_pos = class_features + y_threshold;

					if (y > y_threshold) {
						int chunk_nr = patch_pos / 32;
						int chunk_pos = patch_pos % 32;
						encoded_Xi[chunk_nr] |= (1 << chunk_pos);
					} else if (append_negated) {
						int chunk_nr = (patch_pos + number_of_features) / 32;
						int chunk_pos = (patch_pos + number_of_features) % 32;
						encoded_Xi[chunk_nr] |= (1 << chunk_pos);
					}
				}

				// Encode x coordinate of patch into feature vector
				for (int x_threshold = 0; x_threshold < dim_x - patch_dim_x; ++x_threshold) {
					int patch_pos = class_features + (dim_y - patch_dim_y) + x_threshold;

					if (x > x_threshold) {
						int chunk_nr = patch_pos / 32;
						int chunk_pos = patch_pos % 32;

						encoded_Xi[chunk_nr] |= (1 << chunk_pos);
					} else if (append_negated) {
						int chunk_nr = (patch_pos + number_of_features) / 32;
						int chunk_pos = (patch_pos + number_of_features) % 32;
						encoded_Xi[chunk_nr] |= (1 << chunk_pos);
					}
				} 

				// Encode patch content into feature vector
				for (int p_y = 0; p_y < patch_dim_y; ++p_y) {
					for (int p_x = 0; p_x < patch_dim_x; ++p_x) {
						for (int z = 0; z < dim_z; ++z) {
							int image_pos = (y + p_y)*dim_x*dim_z + (x + p_x)*dim_z + z;
							int patch_pos = class_features + (dim_y - patch_dim_y) + (dim_x - patch_dim_x) + p_y * patch_dim_x * dim_z + p_x * dim_z + z;

							if (Xi[image_pos] == 1) {
								int chunk_nr = patch_pos / 32;
								int chunk_pos = patch_pos % 32;
								encoded_Xi[chunk_nr] |= (1 << chunk_pos);
							} else if (append_negated) {
								int chunk_nr = (patch_pos + number_of_features) / 32;
								int chunk_pos = (patch_pos + number_of_features) % 32;
								encoded_Xi[chunk_nr] |= (1 << chunk_pos);
							}
						}
					}
				}
				encoded_pos += number_of_literal_chunks;
				patch_nr++;
			}
		}
		input_pos += input_step_size;
	}
}

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "AutoencoderClauses.h"
#include "Tools.h"
#include <stdarg.h>
#include <stdbool.h>

void store_clause_to_X(int index,int columns, unsigned int *clauses, int number_of_features, unsigned int *X){
	int start_index = index * columns;
	int end_index = (index + 1) * columns;
	for (int k = start_index; k < end_index; ++k) {
		int feature = clauses[k];
		if (feature > 0 && feature < number_of_features)
		{
			store_feature_to_X(feature, number_of_features, X, 0);
		}
    }
}

void produce_example_by_combined_clauses(
        int number_of_cols,
        unsigned int *X,
        int target_value,
        int accumulation,
		unsigned int *source_clauses,
		int *source_clauses_weights,
		int source_rows,
		int source_columns,
		unsigned int *destination_clauses,
		int *destination_clauses_weights,
		int destination_rows,
		int destination_columns,
		int negative_weight_clause,
		int enable_log
)
{
	FILE* file = fopen("result/output.txt", "a");
	if (file != NULL) {
		enable_printing = enable_log;
		myPrint(file, "\nStart new accumulation (%d) for target_value (%d) ",accumulation,target_value);
		// myPrint(file, "number_of_X_cols = %d\n",number_of_X_cols);
		// myPrint(file, "target_value = %d\n",target_value);
		// myPrint(file, "source_columns = %d\n",source_columns);
		// myPrint(file, "destination_columns = %d\n",destination_columns);
		// myPrint(file, "source_rows = %d\n",source_rows);
		// myPrint(file, "destination_rows = %d\n",destination_rows);
		
		int row;
		int length_of_source = source_rows * source_columns;
		int length_of_destination = destination_rows * destination_columns;

		int number_of_features = number_of_cols;
		int number_of_literals = 2*number_of_features;

		unsigned int number_of_literal_chunks = (number_of_literals-1)/32 + 1;

		memset(X, 0, number_of_literal_chunks * sizeof(unsigned int));
		for (int k = number_of_features; k < number_of_literals; ++k) {
			int chunk_nr = k / 32;
			int chunk_pos = k % 32;
			X[chunk_nr] |= (1U << chunk_pos);
		}
		
		if (source_columns == 0 || destination_columns == 0 || source_rows == 0 || destination_rows == 0) {
			return;
		}

		if (target_value) {
			myPrint(file, "and selected positive clauses is ");
			int a = 0;
			while (a < accumulation) {
				int random_source_index = (rand() % source_rows);
				int random_destination_index = (rand() % destination_rows);
				if (source_clauses_weights[random_source_index] > 0 
					&& destination_clauses_weights[random_destination_index] > 0)
				{
					myPrint(file, "S(%d) ",random_source_index);
					myPrint(file, "D(%d) -- ",random_destination_index);
					store_clause_to_X(random_source_index, source_columns, source_clauses,number_of_cols,X);
					store_clause_to_X(random_destination_index, destination_columns, destination_clauses,number_of_cols,X);
					a++;
				}
			}
		} else {
			myPrint(file, "and selected negative clauses is ");
			if(negative_weight_clause){
				int a = 0;
				while (a < accumulation) {
					int random_source_index = (rand() % source_rows);
					int random_destination_index = (rand() % destination_rows);
					if (source_clauses_weights[random_source_index] < 0 
						&& destination_clauses_weights[random_destination_index] < 0)
					{
						store_clause_to_X(random_source_index, source_columns, source_clauses,number_of_cols,X);
						store_clause_to_X(random_destination_index, destination_columns, destination_clauses,number_of_cols,X);
						myPrint(file, "S(%d) ",random_source_index);
						myPrint(file, "D(%d) -- ",random_destination_index);
						a++;
					}
				}
			}
			else{
				int a = 0;
				while (a < accumulation) {
					int r = 0;
					int total_features = source_columns + destination_columns;
					while (r < total_features){
						int feature;
						bool featureExists;
						do {
							featureExists = false;
							feature = rand() % number_of_cols;
							for (int i = 0; i < length_of_source; i++) {
								if (source_clauses[i] == feature) {
									featureExists = true;
								}
							}
							for (int i = 0; i < length_of_destination; i++) {
								if (destination_clauses[i] == feature) {
									featureExists = true;
								}
							}
						} while (featureExists);
						myPrint(file, "F(%d) ",feature);
						store_feature_to_X(feature, number_of_cols, X,0);
						r++;
					}
					a++;
				}
			}
		}
		fclose(file);
	}
}

void produce_example_by_clauses(
        int number_of_features,
        unsigned int *X,
        int target_value,
        int accumulation,
		unsigned int *clauses,
		int *clauses_weights,
		int rows,
		int columns,
		int negative_weight_clause,
		int enable_log
)
{
	FILE* file = fopen("result/output.txt", "a");
	if (file != NULL) {
		enable_printing = enable_log;
		myPrint(file, "\nStart new accumulation (%d) for target_value (%d) ",accumulation,target_value);
		
		int row;
		int length_of_source = rows * columns;
		int number_of_literals = 2*number_of_features;

		unsigned int number_of_literal_chunks = (number_of_literals-1)/32 + 1;

		memset(X, 0, number_of_literal_chunks * sizeof(unsigned int));
		for (int k = number_of_features; k < number_of_literals; ++k) {
			int chunk_nr = k / 32;
			int chunk_pos = k % 32;
			X[chunk_nr] |= (1U << chunk_pos);
		}
		
		if (columns == 0 || rows == 0) {
			return;
		}

		if (target_value) {
			myPrint(file, "and selected positive clauses is ");
			// for (int i = 0; i < rows; i++)
			// {
			// 	if (clauses_weights[i] > 0)
			// 	{
			// 		store_clause_to_X(i, source_columns, source_clauses,number_of_features,X);
			// 		myPrint(file, "S(%d) ",i);
			// 	}
			// }
			int a = 0;
			while (a < accumulation) {
				int random_index = (rand() % rows);
				if (clauses_weights[random_index] > 0)
				{
					store_clause_to_X(random_index, columns, clauses,number_of_features,X);
					myPrint(file, "+R(%d) ",random_index);
					a++;
				}
			}
		} else {
			myPrint(file, "and selected negative clauses is ");
			if(negative_weight_clause){
				// for (int i = 0; i < source_rows; i++)
				// {
				// 	if (source_clauses_weights[i] < 0)
				// 	{
				// 		store_clause_to_X(i, source_columns, source_clauses,number_of_features,X);
				// 		myPrint(file, "S(%d) ",i);
				// 	}
				// }
				int a = 0;
				while (a < accumulation) {
					int random_index = (rand() % rows);
					if (clauses_weights[random_index] < 0)
					{
						store_clause_to_X(random_index, columns, clauses,number_of_features,X);
						myPrint(file, "-R(%d) ",random_index);
						a++;
					}
				}
			}
			else{
				int a = 0;
				while (a < accumulation) {
					int r = 0;
					int total_features = columns;
					while (r < total_features){
						int feature;
						bool featureExists;
						do {
							featureExists = false;
							feature = rand() % number_of_features;
							for (int i = 0; i < length_of_source; i++) {
								if (clauses[i] == feature) {
									featureExists = true;
								}
							}
						} while (featureExists);
						myPrint(file, "F(%d) ",feature);
						store_feature_to_X(feature, number_of_features, X,0);
						r++;
					}
					a++;
				}
			}
		}
		fclose(file);
	}
}

void produce_example_by_features(
	int number_of_features,
	unsigned int *X,
	int *features,
	int involved_features_number,
	int enable_log
){
	FILE* file = fopen("result/output.txt", "a");
	if (file != NULL) {
		enable_printing = enable_log;
		myPrint(file, "\nStart saving to X with features:(%d) ",involved_features_number);
		for (int i = 0; i < involved_features_number; i++)
		{
			store_feature_to_X(features[i], number_of_features, X, 0);
		}
		fclose(file);
	}
}
/*

Copyright (c) 2021 Ole-Christoffer Granmo

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This code implements the Convolutional Tsetlin Machine from paper arXiv:1905.09688
https://arxiv.org/abs/1905.09688

*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <string.h>
#include "fast_rand.h"

void wb_increment(
        int *clause_weights,
        int number_of_clauses,
        unsigned int *clause_output,
        float update_p,
        unsigned int *clause_active,
        unsigned int positive_weights
)
{
	for (int j = 0; j < number_of_clauses; ++j) {
		if (clause_active[j] && clause_output[j] && (positive_weights || (clause_weights[j] != -1)) && (((float)fast_rand())/((float)FAST_RAND_MAX) <= update_p)) {
			clause_weights[j]++;
		}
	}
}

void wb_decrement(
        int *clause_weights,
        int number_of_clauses,
        unsigned int *clause_output,
        float update_p,
        unsigned int *clause_active,
        unsigned int negative_weights
)
{
	for (int j = 0; j < number_of_clauses; j++) {
		if (clause_active[j] && clause_output[j] && (negative_weights || (clause_weights[j] != 1)) && (((float)fast_rand())/((float)FAST_RAND_MAX) <= update_p)) {
			clause_weights[j]--;
		}
	}
}
/*

Copyright (c) 2023 Ole-Christoffer Granmo

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This code implements the Convolutional Tsetlin Machine from paper arXiv:1905.09688
https://arxiv.org/abs/1905.09688

*/

#ifdef _MSC_VER
#  include <intrin.h>
#  define __builtin_popcount __popcnt
#endif

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <string.h>
#include "fast_rand.h"
#include <stdint.h>

void cbs_prepare_Xi(
        unsigned int *indices,
        int number_of_indices,
        unsigned int *Xi,
        int number_of_features
)
{
    for (int k = 0; k < number_of_indices; ++k) { 
        unsigned int chunk = indices[k] / 32;
        unsigned int pos = indices[k] % 32;
        Xi[chunk] |= (1U << pos);
        chunk = (indices[k] + number_of_features) / 32;
        pos = (indices[k] + number_of_features) % 32;
        Xi[chunk] &= ~(1U << pos);
    }
}

void cbs_restore_Xi(
        unsigned int *indices,
        int number_of_indices,
        unsigned int *Xi,
        int number_of_features
)
{
    for (int k = 0; k < number_of_indices; ++k) { 
        unsigned int chunk = indices[k] / 32;
        unsigned int pos = indices[k] % 32;
        Xi[chunk] &= ~(1U << pos);
        chunk = (indices[k] + number_of_features) / 32;
        pos = (indices[k] + number_of_features) % 32;
        Xi[chunk] |= (1U << pos);
    }
}

void cbs_pack_X(
        int *indptr,
        int *indices,
        int number_of_examples,
        int e,
        unsigned int *packed_X,
        int number_of_literals
)
{
    for (int k = 0; k < number_of_literals/2; ++k) {
    	packed_X[k] = 0U;
    }

    for (int k = number_of_literals/2; k < number_of_literals; ++k) {
    	packed_X[k] = ~0U;
    }

    for (int i = 0; i < 32; ++i) {
        if (e+i >= number_of_examples) {
            break;
        }

        for (int k = indptr[e+i]; k < indptr[e+i+1]; ++k) {
            packed_X[indices[k]] |= (1U << i);
            packed_X[indices[k] + number_of_literals/2] &= ~(1U << i);
        }
    }
}

void cbs_unpack_clause_output(
        int e,
        unsigned int *clause_output,
        unsigned int *clause_output_batch,
        int number_of_clauses
)
{
	for (int j = 0; j < number_of_clauses; ++j) {
		clause_output[j] = ((clause_output_batch[j] & (1U << (e % 32U))) > 0);
	}
}

void cbs_calculate_clause_outputs_update(
        unsigned int *literal_active,
        unsigned int *Xi,
        int number_of_clauses,
        int number_of_literals,
        unsigned int *clause_output,
        unsigned short *clause_bank_included,
        unsigned short *clause_bank_included_length
)
{
    for (int j = 0; j < number_of_clauses; ++j) {
        clause_output[j] = 1;
        for (int k = 0; k < clause_bank_included_length[j]; ++k) {
        	unsigned int clause_pos = j*number_of_literals*2 + k*2;
            unsigned int literal_chunk = clause_bank_included[clause_pos] / 32U;
            unsigned int literal_pos = clause_bank_included[clause_pos] % 32U;
            if (((Xi[literal_chunk] & (1U << literal_pos)) == 0) && (literal_active[literal_chunk] & (1U << literal_pos))) {
                clause_output[j] = 0;
                break;
            }
        }
    }
}

void cbs_calculate_clause_outputs_predict_packed_X(
        unsigned int *packed_X,
        int number_of_clauses,
        int number_of_literals,
        unsigned int *clause_output_batch,
        unsigned short *clause_bank_included,
        unsigned short *clause_bank_included_length
)
{
     for (int j = 0; j < number_of_clauses; ++j) {
         if (clause_bank_included_length[j] == 0) {
             clause_output_batch[j] = 0;
        } else {
             clause_output_batch[j] = ~0;
        }

        for (int k = 0; k < clause_bank_included_length[j]; ++k) {
        	unsigned int clause_pos = j*number_of_literals*2 + k*2;
        	clause_output_batch[j] &= packed_X[clause_bank_included[clause_pos]];
        }
    }
}

void cbs_calculate_clause_outputs_predict(
        unsigned int *Xi,
        int number_of_clauses,
        int number_of_literals,
        unsigned int *clause_output,
        unsigned short *clause_bank_included,
        unsigned short *clause_bank_included_length
)
{
    for (int j = 0; j < number_of_clauses; ++j) {
        if (clause_bank_included_length[j] == 0) {
            clause_output[j] = 0;
        } else {
            clause_output[j] = 1;
        }

        for (int k = 0; k < clause_bank_included_length[j]; ++k) {
            unsigned int clause_pos = j*number_of_literals*2 + k*2;
            unsigned int literal_chunk = clause_bank_included[clause_pos] / 32;
            unsigned int literal_pos = clause_bank_included[clause_pos] % 32;
            if ((Xi[literal_chunk] & (1U << literal_pos)) == 0) {
                clause_output[j] = 0;
                break;
            }
        }
    }
}

void cbs_type_i_feedback(
        float update_p,
        float s,
        int boost_true_positive_feedback,
        int max_included_literals,
        int absorbing,
        int feedback_rate_excluded_literals,
        int literal_insertion_state,
        int *clause_active,
        unsigned int *literal_active,
        unsigned int *Xi,
        int number_of_clauses,
        int number_of_literals,
        int number_of_states,
        unsigned short *clause_bank_included,
        unsigned short *clause_bank_included_length,
        unsigned short *clause_bank_excluded,
        unsigned short *clause_bank_excluded_length,
        unsigned short *clause_bank_unallocated,
        unsigned short *clause_bank_unallocated_length
)
{
	unsigned int number_of_ta_chunks = (number_of_literals-1)/32 + 1;

    for (int j = 0; j < number_of_clauses; ++j) {
        if ((((float)fast_rand())/((float)FAST_RAND_MAX) > update_p) || (!clause_active[j])) {
			continue;
		}

		int clause_pos_base = j*number_of_literals*2;

        int clause_output = 1;
        for (int k = 0; k < clause_bank_included_length[j]; ++k) {
        	unsigned int clause_pos = clause_pos_base + k*2;
            unsigned int literal_chunk = clause_bank_included[clause_pos] / 32;
            unsigned int literal_pos = clause_bank_included[clause_pos] % 32;
            if (((Xi[literal_chunk] & (1U << literal_pos)) == 0) && (literal_active[literal_chunk] & (1U << literal_pos))) {
                clause_output = 0;
                break;
            }
        }

        if (clause_output && (clause_bank_included_length[j] <= max_included_literals)) {
			int k = clause_bank_included_length[j];
			while (k--) {
				int clause_included_pos = clause_pos_base + k*2;
	            unsigned int literal_chunk = clause_bank_included[clause_included_pos] / 32;
	            unsigned int literal_pos = clause_bank_included[clause_included_pos] % 32;

                if ((literal_active[literal_chunk] & (1U << literal_pos)) == 0) {
                    continue;
                }

            	if ((Xi[literal_chunk] & (1U << literal_pos)) != 0) {
                   	if (clause_bank_included[clause_included_pos + 1] < number_of_states-1 && (boost_true_positive_feedback || (((float)fast_rand())/((float)FAST_RAND_MAX) > 1.0/s))) {
                        clause_bank_included[clause_included_pos + 1] += 1;
                	}
                } else if (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/s) {
                    clause_bank_included[clause_included_pos + 1] -= 1;
                    if (clause_bank_included[clause_included_pos + 1] < number_of_states / 2) {
                    	int clause_excluded_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
                        clause_bank_excluded[clause_excluded_pos] = clause_bank_included[clause_included_pos];
                        clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_included[clause_included_pos + 1];
                        clause_bank_excluded_length[j] += 1;

                        clause_bank_included_length[j] -= 1;
                        int clause_included_end_pos = clause_pos_base + clause_bank_included_length[j]*2;
                        clause_bank_included[clause_included_pos] = clause_bank_included[clause_included_end_pos];       
                        clause_bank_included[clause_included_pos + 1] = clause_bank_included[clause_included_end_pos + 1];
                    }
                }
            }

            if (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/feedback_rate_excluded_literals) {
                k = clause_bank_excluded_length[j];
    			while (k--) {
    				int clause_excluded_pos = clause_pos_base + k*2;
                	unsigned int literal_chunk = clause_bank_excluded[clause_excluded_pos] / 32;
                	unsigned int literal_pos = clause_bank_excluded[clause_excluded_pos] % 32;
    		
                    if ((literal_active[literal_chunk] & (1U << literal_pos)) == 0) {
                        continue;
                    }

                	if ((Xi[literal_chunk] & (1U << literal_pos)) != 0) {
    	               if (boost_true_positive_feedback || (((float)fast_rand())/((float)FAST_RAND_MAX) > 1.0/s)) {
                            clause_bank_excluded[clause_excluded_pos + 1] += feedback_rate_excluded_literals;
                            if (clause_bank_excluded[clause_excluded_pos + 1] >= number_of_states / 2) {
                                int clause_included_pos = clause_pos_base + clause_bank_included_length[j]*2;
    		                    clause_bank_included[clause_included_pos] = clause_bank_excluded[clause_excluded_pos];
    		                    clause_bank_included[clause_included_pos + 1] = clause_bank_excluded[clause_excluded_pos + 1];
    		                    clause_bank_included_length[j] += 1;

    		                    clause_bank_excluded_length[j] -= 1;
    		                    int clause_excluded_end_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
    		                    clause_bank_excluded[clause_excluded_pos] = clause_bank_excluded[clause_excluded_end_pos];
    		                    clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_excluded[clause_excluded_end_pos + 1];
                            }
                        }
                    } else if ((clause_bank_excluded[clause_excluded_pos + 1] > feedback_rate_excluded_literals) && (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/s)) {
                        clause_bank_excluded[clause_excluded_pos + 1] -= feedback_rate_excluded_literals;
                        
                        if ((int)clause_bank_excluded[clause_excluded_pos + 1] <= absorbing) {
                            if (clause_bank_unallocated_length[j] == 0 || literal_insertion_state == -1) {
                                clause_bank_excluded_length[j] -= 1;
                                int clause_excluded_end_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
                                clause_bank_excluded[clause_excluded_pos] = clause_bank_excluded[clause_excluded_end_pos];
                                clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_excluded[clause_excluded_end_pos + 1];
                            } else {
                                clause_bank_unallocated_length[j] -= 1;
                                int clause_unallocated_end_pos = j*number_of_literals + clause_bank_unallocated_length[j];
                                clause_bank_excluded[clause_excluded_pos] = clause_bank_unallocated[clause_unallocated_end_pos];
                                clause_bank_excluded[clause_excluded_pos + 1] = literal_insertion_state;
                            }
                        }
                    }
                }
            }
        } else {
        	int k = clause_bank_included_length[j];
			while (k--) {
				int clause_included_pos = clause_pos_base + k*2;
            	unsigned int literal_chunk = clause_bank_included[clause_included_pos] / 32;
            	unsigned int literal_pos = clause_bank_included[clause_included_pos] % 32;

                if ((literal_active[literal_chunk] & (1U << literal_pos)) == 0) {
                    continue;
                }

				if (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/s) {
                    clause_bank_included[clause_included_pos + 1] -= 1;
                    if (clause_bank_included[clause_included_pos + 1] < number_of_states / 2) {
                    	int clause_excluded_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
                        clause_bank_excluded[clause_excluded_pos] = clause_bank_included[clause_included_pos];
                        clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_included[clause_included_pos + 1];
                        clause_bank_excluded_length[j] += 1;

                        clause_bank_included_length[j] -= 1;
                        int clause_included_end_pos = clause_pos_base + clause_bank_included_length[j]*2;
                        clause_bank_included[clause_included_pos] = clause_bank_included[clause_included_end_pos];       
                        clause_bank_included[clause_included_pos + 1] = clause_bank_included[clause_included_end_pos + 1];
                    }
                }
            }
 			
            if (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/feedback_rate_excluded_literals) {
     			k = clause_bank_excluded_length[j];
    			while (k--) {
    				int clause_excluded_pos = clause_pos_base + k*2;
                	unsigned int literal_chunk = clause_bank_excluded[clause_excluded_pos] / 32;
                	unsigned int literal_pos = clause_bank_excluded[clause_excluded_pos] % 32;
    		
                    if ((literal_active[literal_chunk] & (1U << literal_pos)) == 0) {
                        continue;
                    }

                	if ((clause_bank_excluded[clause_excluded_pos + 1] > feedback_rate_excluded_literals) && (((float)fast_rand())/((float)FAST_RAND_MAX) <= 1.0/s)) {
                        clause_bank_excluded[clause_excluded_pos + 1] -= feedback_rate_excluded_literals;

                        if ((int)clause_bank_excluded[clause_excluded_pos + 1] <= absorbing) {
                           if (clause_bank_unallocated_length[j] == 0 || literal_insertion_state == -1) {
                                clause_bank_excluded_length[j] -= 1;
                                int clause_excluded_end_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
                                clause_bank_excluded[clause_excluded_pos] = clause_bank_excluded[clause_excluded_end_pos];
                                clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_excluded[clause_excluded_end_pos + 1];
                            } else {
                                clause_bank_unallocated_length[j] -= 1;
                                int clause_unallocated_end_pos = j*number_of_literals + clause_bank_unallocated_length[j];
                                clause_bank_excluded[clause_excluded_pos] = clause_bank_unallocated[clause_unallocated_end_pos];
                                clause_bank_excluded[clause_excluded_pos + 1] = literal_insertion_state;
                            }
                        }
                    }
                }
            }
    	}
    }
}

void cbs_type_ii_feedback(
        float update_p,
        int feedback_rate_excluded_literals,
        int *clause_active,
        unsigned int *literal_active,
        unsigned int *Xi,
        int number_of_clauses,
        int number_of_literals,
        int number_of_states,
        unsigned short *clause_bank_included,
        unsigned short *clause_bank_included_length,
        unsigned short *clause_bank_excluded,
        unsigned short *clause_bank_excluded_length
)
{
    for (int j = 0; j < number_of_clauses; ++j) {
    	if ((((float)fast_rand())/((float)FAST_RAND_MAX) > update_p) || (!clause_active[j])) {
			continue;
		}

        int clause_output = 1;
        for (int k = 0; k < clause_bank_included_length[j]; ++k) {
        	unsigned int clause_pos = j*number_of_literals*2 + k*2;
            unsigned int literal_chunk = clause_bank_included[clause_pos] / 32;
            unsigned int literal_pos = clause_bank_included[clause_pos] % 32;
            if (((Xi[literal_chunk] & (1U << literal_pos)) == 0) && (literal_active[literal_chunk] & (1U << literal_pos))) {
                clause_output = 0;
                break;
            }
        }

        if (clause_output == 0 || (((float)fast_rand())/((float)FAST_RAND_MAX) > 1.0/feedback_rate_excluded_literals)) {
            continue;
        }

        // Type II Feedback
	
		int clause_pos_base = j*number_of_literals*2;
		int k = clause_bank_excluded_length[j];
		while (k--) {
			int clause_excluded_pos = clause_pos_base + k*2;
            unsigned int literal_chunk = clause_bank_excluded[clause_excluded_pos] / 32;
            unsigned int literal_pos = clause_bank_excluded[clause_excluded_pos] % 32;
		
            if (((Xi[literal_chunk] & (1U << literal_pos)) == 0) && (literal_active[literal_chunk] & (1U << literal_pos))) {
                clause_bank_excluded[clause_excluded_pos + 1] += feedback_rate_excluded_literals;

                if (clause_bank_excluded[clause_excluded_pos + 1] >= number_of_states/2) {
                	int clause_included_pos = clause_pos_base + clause_bank_included_length[j]*2;
                    clause_bank_included[clause_included_pos] = clause_bank_excluded[clause_excluded_pos];
                    clause_bank_included[clause_included_pos + 1] = clause_bank_excluded[clause_excluded_pos + 1];
                    clause_bank_included_length[j] += 1;

                    clause_bank_excluded_length[j] -= 1;
                    int clause_excluded_end_pos = clause_pos_base + clause_bank_excluded_length[j]*2;
                    clause_bank_excluded[clause_excluded_pos] = clause_bank_excluded[clause_excluded_end_pos];
                    clause_bank_excluded[clause_excluded_pos + 1] = clause_bank_excluded[clause_excluded_end_pos + 1];
                }
			}
  		} 
	}
}

#include "fast_rand.h"

static uint64_t const multiplier = 6364136223846793005u;
static uint64_t mcg_state = 0xcafef00dd15ea5e5u;

void pcg32_seed(uint64_t seed) {
    mcg_state = seed;
}

uint32_t pcg32_fast() {
    uint64_t x = mcg_state;
    unsigned int count = (unsigned int)(x >> 61);	// 61 = 64 - 3
    mcg_state = x * multiplier;
    return (uint32_t)((x ^ x >> 22) >> (22 + count));	// 22 = 32 - 3 - 7
}


#include "fast_rand.h"

// Seed/state for the RNG.
static uint64_t xorshift_state[2] = {0xcafef00dbadc0ffeULL, 0xdeadbeef12345678ULL};


// Seeding function.
void xorshift128p_seed(uint64_t seed) {
    xorshift_state[0] = seed;
    xorshift_state[1] = ~seed;  // Just for some variety. You can use any other method you like.
}

// Fast XOR-Shift128+ RNG.
uint32_t xorshift128p_fast() {
    uint64_t s1 = xorshift_state[0];
    const uint64_t s0 = xorshift_state[1];
    xorshift_state[0] = s0;
    s1 ^= s1 << 23;
    xorshift_state[1] = s1 ^ s0 ^ (s1 >> 18) ^ (s0 >> 5);

    return (uint32_t)(xorshift_state[1] + s0);
}




/************************************************************/

static void *_cffi_types[] = {
/*  0 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1), // int()(int)
/*  1 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7), // int
/*  2 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  3 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1), // int()(unsigned int *, int, int, int)
/*  4 */ _CFFI_OP(_CFFI_OP_POINTER, 66), // unsigned int *
/*  5 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/*  6 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/*  7 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/*  8 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  9 */ _CFFI_OP(_CFFI_OP_FUNCTION, 66), // unsigned int()(void const *, void const *)
/* 10 */ _CFFI_OP(_CFFI_OP_POINTER, 308), // void const *
/* 11 */ _CFFI_OP(_CFFI_OP_NOOP, 10),
/* 12 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 13 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(FILE *, char const *, ...)
/* 14 */ _CFFI_OP(_CFFI_OP_POINTER, 304), // FILE *
/* 15 */ _CFFI_OP(_CFFI_OP_POINTER, 305), // char const *
/* 16 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 1),
/* 17 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(float, float, int, int, int, int, int, int *, unsigned int *, unsigned int *, int, int, int, unsigned short *, unsigned short *, unsigned short *, unsigned short *, unsigned short *, unsigned short *)
/* 18 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13), // float
/* 19 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 20 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 21 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 22 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 23 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 24 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 25 */ _CFFI_OP(_CFFI_OP_POINTER, 1), // int *
/* 26 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 27 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 28 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 29 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 30 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 31 */ _CFFI_OP(_CFFI_OP_POINTER, 306), // unsigned short *
/* 32 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 33 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 34 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 35 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 36 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 37 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 38 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(float, int, int *, unsigned int *, unsigned int *, int, int, int, unsigned short *, unsigned short *, unsigned short *, unsigned short *)
/* 39 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 40 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 41 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 42 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 43 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 44 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 45 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 46 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 47 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 48 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 49 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 50 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 51 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 52 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int *, int *, int, int, unsigned int *, int)
/* 53 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 54 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 55 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 56 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 57 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 58 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 59 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 60 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int *, int, unsigned int *, float, unsigned int *, unsigned int)
/* 61 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 62 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 63 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 64 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 65 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 66 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8), // unsigned int
/* 67 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 68 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int, int, unsigned int *, int)
/* 69 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 70 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 71 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 72 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 73 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 74 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int, unsigned int *, int *, int, int)
/* 75 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 76 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 77 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 78 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 79 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 80 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 81 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int, unsigned int *, int, int, unsigned int *, int *, int, int, int, int)
/* 82 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 83 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 84 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 85 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 86 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 87 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 88 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 89 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 90 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 91 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 92 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 93 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int, unsigned int *, int, int, unsigned int *, int *, int, int, unsigned int *, int *, int, int, int, int)
/* 94 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 95 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 96 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 97 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 98 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 99 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 100 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 101 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 102 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 103 */ _CFFI_OP(_CFFI_OP_NOOP, 25),
/* 104 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 105 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 106 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 107 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 108 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 109 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(int, unsigned int *, unsigned int *, int)
/* 110 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 111 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 112 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 113 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 114 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 115 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(uint64_t)
/* 116 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24), // uint64_t
/* 117 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 118 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, int, int, int, unsigned int *, unsigned int *)
/* 119 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 120 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 121 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 122 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 123 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 124 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 125 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 126 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 127 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, int, int, int, unsigned int *, unsigned int *, unsigned int *)
/* 128 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 129 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 130 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 131 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 132 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 133 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 134 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 135 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 136 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 137 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, int, int, unsigned int *)
/* 138 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 139 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 140 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 141 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 142 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 143 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 144 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, int, int, unsigned int *, unsigned int *)
/* 145 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 146 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 147 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 148 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 149 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 150 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 151 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 152 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, int, unsigned int *, unsigned short *, unsigned short *)
/* 153 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 154 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 155 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 156 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 157 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 158 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, unsigned int *, int)
/* 161 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 162 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 163 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 164 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 165 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 166 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, unsigned int *, unsigned int *, int, unsigned int *, unsigned int *, int, unsigned int *, int, int, int)
/* 167 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 168 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 169 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 170 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 171 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 172 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 173 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 174 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 175 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 176 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 177 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 178 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 179 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 180 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, int, unsigned int *, unsigned int *, int, unsigned int *, unsigned int *, int, unsigned int *, unsigned int *, int, unsigned int *, int, int, int, int, int)
/* 181 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 182 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 183 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 184 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 185 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 186 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 187 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 188 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 189 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 190 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 191 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 192 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 193 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 194 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 195 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 196 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 197 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 198 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 199 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, int, int, int, int, float, unsigned int *, unsigned int *, unsigned int *)
/* 200 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 201 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 202 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 203 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 204 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 205 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 206 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 207 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 208 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 209 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 210 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 211 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, int, int, int, int, int, int, int, int)
/* 212 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 213 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 214 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 215 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 216 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 217 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 218 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 219 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 220 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 221 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 222 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 223 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, int, int, unsigned int *, unsigned short *, unsigned short *)
/* 224 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 225 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 226 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 227 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 228 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 229 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 230 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 231 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 232 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, unsigned int *, int, int, int, int, float, float, unsigned int, unsigned int, unsigned int, unsigned int *, unsigned int *, unsigned int *)
/* 233 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 234 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 235 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 236 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 237 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 238 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 239 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 240 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 241 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 242 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 243 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 244 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 245 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 246 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 247 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 248 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 249 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, unsigned int *, int, int, int, unsigned int *, unsigned int *, unsigned int *, int)
/* 250 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 251 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 252 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 253 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 254 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 255 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 256 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 257 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 258 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 259 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 260 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 261 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, unsigned int *, int, int, unsigned int *, unsigned int *)
/* 262 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 263 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 264 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 265 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 266 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 267 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 268 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 269 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 270 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, unsigned int *, unsigned int *, int, int, int, int, int, float, float, unsigned int *, unsigned int *, unsigned int *, unsigned int)
/* 271 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 272 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 273 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 274 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 275 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 276 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 277 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 278 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 279 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 280 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 281 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 13),
/* 282 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 283 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 284 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 285 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 286 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 287 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int *, unsigned int *, unsigned int *, unsigned int *, int, int, int, unsigned int *)
/* 288 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 289 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 290 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 291 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 292 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 293 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 294 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 295 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 296 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 297 */ _CFFI_OP(_CFFI_OP_FUNCTION, 308), // void()(unsigned int const *, unsigned int, unsigned int, unsigned int, unsigned int *)
/* 298 */ _CFFI_OP(_CFFI_OP_POINTER, 66), // unsigned int const *
/* 299 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 300 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 301 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 302 */ _CFFI_OP(_CFFI_OP_NOOP, 4),
/* 303 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 304 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 0), // FILE
/* 305 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 2), // char
/* 306 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 6), // unsigned short
/* 307 */ _CFFI_OP(_CFFI_OP_POINTER, 13), // void(*)(FILE *, char const *, ...)
/* 308 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 0), // void
};

static void _cffi_d_cb_calculate_clause_outputs_incremental(unsigned int * x0, unsigned int * x1, unsigned int * x2, int x3, int x4, unsigned int * x5, unsigned int * x6)
{
  cb_calculate_clause_outputs_incremental(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_clause_outputs_incremental(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  unsigned int * x2;
  int x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "cb_calculate_clause_outputs_incremental", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_clause_outputs_incremental(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_clause_outputs_incremental _cffi_d_cb_calculate_clause_outputs_incremental
#endif

static void _cffi_d_cb_calculate_clause_outputs_incremental_batch(unsigned int * x0, unsigned int * x1, unsigned int * x2, int x3, int x4, int x5, unsigned int * x6, unsigned int * x7, unsigned int * x8, int x9)
{
  cb_calculate_clause_outputs_incremental_batch(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_clause_outputs_incremental_batch(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  unsigned int * x2;
  int x3;
  int x4;
  int x5;
  unsigned int * x6;
  unsigned int * x7;
  unsigned int * x8;
  int x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "cb_calculate_clause_outputs_incremental_batch", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_clause_outputs_incremental_batch(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_clause_outputs_incremental_batch _cffi_d_cb_calculate_clause_outputs_incremental_batch
#endif

static void _cffi_d_cb_calculate_clause_outputs_patchwise(unsigned int * x0, int x1, int x2, int x3, int x4, unsigned int * x5, unsigned int * x6)
{
  cb_calculate_clause_outputs_patchwise(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_clause_outputs_patchwise(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "cb_calculate_clause_outputs_patchwise", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_clause_outputs_patchwise(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_clause_outputs_patchwise _cffi_d_cb_calculate_clause_outputs_patchwise
#endif

static void _cffi_d_cb_calculate_clause_outputs_predict(unsigned int * x0, int x1, int x2, int x3, int x4, unsigned int * x5, unsigned int * x6)
{
  cb_calculate_clause_outputs_predict(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_clause_outputs_predict(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "cb_calculate_clause_outputs_predict", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_clause_outputs_predict(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_clause_outputs_predict _cffi_d_cb_calculate_clause_outputs_predict
#endif

static void _cffi_d_cb_calculate_clause_outputs_update(unsigned int * x0, int x1, int x2, int x3, int x4, unsigned int * x5, unsigned int * x6, unsigned int * x7)
{
  cb_calculate_clause_outputs_update(x0, x1, x2, x3, x4, x5, x6, x7);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_clause_outputs_update(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  unsigned int * x7;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;

  if (!PyArg_UnpackTuple(args, "cb_calculate_clause_outputs_update", 8, 8, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_clause_outputs_update(x0, x1, x2, x3, x4, x5, x6, x7); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_clause_outputs_update _cffi_d_cb_calculate_clause_outputs_update
#endif

static void _cffi_d_cb_calculate_literal_frequency(unsigned int * x0, int x1, int x2, int x3, unsigned int * x4, unsigned int * x5)
{
  cb_calculate_literal_frequency(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_calculate_literal_frequency(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  unsigned int * x4;
  unsigned int * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "cb_calculate_literal_frequency", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_calculate_literal_frequency(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_calculate_literal_frequency _cffi_d_cb_calculate_literal_frequency
#endif

static void _cffi_d_cb_get_literals(unsigned int const * x0, unsigned int x1, unsigned int x2, unsigned int x3, unsigned int * x4)
{
  cb_get_literals(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_get_literals(PyObject *self, PyObject *args)
{
  unsigned int const * x0;
  unsigned int x1;
  unsigned int x2;
  unsigned int x3;
  unsigned int * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "cb_get_literals", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(298), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(298), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, unsigned int);
  if (x1 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, unsigned int);
  if (x3 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_get_literals(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_get_literals _cffi_d_cb_get_literals
#endif

static void _cffi_d_cb_included_literals(unsigned int * x0, int x1, int x2, int x3, unsigned int * x4)
{
  cb_included_literals(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_included_literals(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  unsigned int * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "cb_included_literals", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_included_literals(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_included_literals _cffi_d_cb_included_literals
#endif

static void _cffi_d_cb_initialize_incremental_clause_calculation(unsigned int * x0, unsigned int * x1, unsigned int * x2, unsigned int * x3, int x4, int x5, int x6, unsigned int * x7)
{
  cb_initialize_incremental_clause_calculation(x0, x1, x2, x3, x4, x5, x6, x7);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_initialize_incremental_clause_calculation(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  unsigned int * x2;
  unsigned int * x3;
  int x4;
  int x5;
  int x6;
  unsigned int * x7;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;

  if (!PyArg_UnpackTuple(args, "cb_initialize_incremental_clause_calculation", 8, 8, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_initialize_incremental_clause_calculation(x0, x1, x2, x3, x4, x5, x6, x7); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_initialize_incremental_clause_calculation _cffi_d_cb_initialize_incremental_clause_calculation
#endif

static int _cffi_d_cb_number_of_include_actions(unsigned int * x0, int x1, int x2, int x3)
{
  return cb_number_of_include_actions(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_number_of_include_actions(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "cb_number_of_include_actions", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = cb_number_of_include_actions(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_cb_number_of_include_actions _cffi_d_cb_number_of_include_actions
#endif

static void _cffi_d_cb_type_i_feedback(unsigned int * x0, unsigned int * x1, unsigned int * x2, int x3, int x4, int x5, int x6, float x7, float x8, unsigned int x9, unsigned int x10, unsigned int x11, unsigned int * x12, unsigned int * x13, unsigned int * x14)
{
  cb_type_i_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_type_i_feedback(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  unsigned int * x2;
  int x3;
  int x4;
  int x5;
  int x6;
  float x7;
  float x8;
  unsigned int x9;
  unsigned int x10;
  unsigned int x11;
  unsigned int * x12;
  unsigned int * x13;
  unsigned int * x14;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;
  PyObject *arg12;
  PyObject *arg13;
  PyObject *arg14;

  if (!PyArg_UnpackTuple(args, "cb_type_i_feedback", 15, 15, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11, &arg12, &arg13, &arg14))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = (float)_cffi_to_c_float(arg7);
  if (x7 == (float)-1 && PyErr_Occurred())
    return NULL;

  x8 = (float)_cffi_to_c_float(arg8);
  if (x8 == (float)-1 && PyErr_Occurred())
    return NULL;

  x9 = _cffi_to_c_int(arg9, unsigned int);
  if (x9 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x10 = _cffi_to_c_int(arg10, unsigned int);
  if (x10 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  x11 = _cffi_to_c_int(arg11, unsigned int);
  if (x11 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg12, (char **)&x12);
  if (datasize != 0) {
    x12 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg12, (char **)&x12,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg13, (char **)&x13);
  if (datasize != 0) {
    x13 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg13, (char **)&x13,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg14, (char **)&x14);
  if (datasize != 0) {
    x14 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg14, (char **)&x14,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_type_i_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_type_i_feedback _cffi_d_cb_type_i_feedback
#endif

static void _cffi_d_cb_type_ii_feedback(unsigned int * x0, unsigned int * x1, int x2, int x3, int x4, int x5, float x6, unsigned int * x7, unsigned int * x8, unsigned int * x9)
{
  cb_type_ii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_type_ii_feedback(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  int x2;
  int x3;
  int x4;
  int x5;
  float x6;
  unsigned int * x7;
  unsigned int * x8;
  unsigned int * x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "cb_type_ii_feedback", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = (float)_cffi_to_c_float(arg6);
  if (x6 == (float)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg9, (char **)&x9);
  if (datasize != 0) {
    x9 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg9, (char **)&x9,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_type_ii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_type_ii_feedback _cffi_d_cb_type_ii_feedback
#endif

static void _cffi_d_cb_type_iii_feedback(unsigned int * x0, unsigned int * x1, unsigned int * x2, unsigned int * x3, int x4, int x5, int x6, int x7, int x8, float x9, float x10, unsigned int * x11, unsigned int * x12, unsigned int * x13, unsigned int x14)
{
  cb_type_iii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cb_type_iii_feedback(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  unsigned int * x2;
  unsigned int * x3;
  int x4;
  int x5;
  int x6;
  int x7;
  int x8;
  float x9;
  float x10;
  unsigned int * x11;
  unsigned int * x12;
  unsigned int * x13;
  unsigned int x14;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;
  PyObject *arg12;
  PyObject *arg13;
  PyObject *arg14;

  if (!PyArg_UnpackTuple(args, "cb_type_iii_feedback", 15, 15, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11, &arg12, &arg13, &arg14))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  x8 = _cffi_to_c_int(arg8, int);
  if (x8 == (int)-1 && PyErr_Occurred())
    return NULL;

  x9 = (float)_cffi_to_c_float(arg9);
  if (x9 == (float)-1 && PyErr_Occurred())
    return NULL;

  x10 = (float)_cffi_to_c_float(arg10);
  if (x10 == (float)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg11, (char **)&x11);
  if (datasize != 0) {
    x11 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg11, (char **)&x11,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg12, (char **)&x12);
  if (datasize != 0) {
    x12 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg12, (char **)&x12,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg13, (char **)&x13);
  if (datasize != 0) {
    x13 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg13, (char **)&x13,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x14 = _cffi_to_c_int(arg14, unsigned int);
  if (x14 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cb_type_iii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cb_type_iii_feedback _cffi_d_cb_type_iii_feedback
#endif

static void _cffi_d_cbs_calculate_clause_outputs_predict(unsigned int * x0, int x1, int x2, unsigned int * x3, unsigned short * x4, unsigned short * x5)
{
  cbs_calculate_clause_outputs_predict(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_calculate_clause_outputs_predict(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  unsigned int * x3;
  unsigned short * x4;
  unsigned short * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "cbs_calculate_clause_outputs_predict", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_calculate_clause_outputs_predict(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_calculate_clause_outputs_predict _cffi_d_cbs_calculate_clause_outputs_predict
#endif

static void _cffi_d_cbs_calculate_clause_outputs_predict_packed_X(unsigned int * x0, int x1, int x2, unsigned int * x3, unsigned short * x4, unsigned short * x5)
{
  cbs_calculate_clause_outputs_predict_packed_X(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_calculate_clause_outputs_predict_packed_X(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  int x2;
  unsigned int * x3;
  unsigned short * x4;
  unsigned short * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "cbs_calculate_clause_outputs_predict_packed_X", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_calculate_clause_outputs_predict_packed_X(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_calculate_clause_outputs_predict_packed_X _cffi_d_cbs_calculate_clause_outputs_predict_packed_X
#endif

static void _cffi_d_cbs_calculate_clause_outputs_update(unsigned int * x0, unsigned int * x1, int x2, int x3, unsigned int * x4, unsigned short * x5, unsigned short * x6)
{
  cbs_calculate_clause_outputs_update(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_calculate_clause_outputs_update(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  int x2;
  int x3;
  unsigned int * x4;
  unsigned short * x5;
  unsigned short * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "cbs_calculate_clause_outputs_update", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_calculate_clause_outputs_update(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_calculate_clause_outputs_update _cffi_d_cbs_calculate_clause_outputs_update
#endif

static void _cffi_d_cbs_pack_X(int * x0, int * x1, int x2, int x3, unsigned int * x4, int x5)
{
  cbs_pack_X(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_pack_X(PyObject *self, PyObject *args)
{
  int * x0;
  int * x1;
  int x2;
  int x3;
  unsigned int * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "cbs_pack_X", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_pack_X(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_pack_X _cffi_d_cbs_pack_X
#endif

static void _cffi_d_cbs_prepare_Xi(unsigned int * x0, int x1, unsigned int * x2, int x3)
{
  cbs_prepare_Xi(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_prepare_Xi(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  unsigned int * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "cbs_prepare_Xi", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_prepare_Xi(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_prepare_Xi _cffi_d_cbs_prepare_Xi
#endif

static void _cffi_d_cbs_restore_Xi(unsigned int * x0, int x1, unsigned int * x2, int x3)
{
  cbs_restore_Xi(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_restore_Xi(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  unsigned int * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "cbs_restore_Xi", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_restore_Xi(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_restore_Xi _cffi_d_cbs_restore_Xi
#endif

static void _cffi_d_cbs_type_i_feedback(float x0, float x1, int x2, int x3, int x4, int x5, int x6, int * x7, unsigned int * x8, unsigned int * x9, int x10, int x11, int x12, unsigned short * x13, unsigned short * x14, unsigned short * x15, unsigned short * x16, unsigned short * x17, unsigned short * x18)
{
  cbs_type_i_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, x15, x16, x17, x18);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_type_i_feedback(PyObject *self, PyObject *args)
{
  float x0;
  float x1;
  int x2;
  int x3;
  int x4;
  int x5;
  int x6;
  int * x7;
  unsigned int * x8;
  unsigned int * x9;
  int x10;
  int x11;
  int x12;
  unsigned short * x13;
  unsigned short * x14;
  unsigned short * x15;
  unsigned short * x16;
  unsigned short * x17;
  unsigned short * x18;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;
  PyObject *arg12;
  PyObject *arg13;
  PyObject *arg14;
  PyObject *arg15;
  PyObject *arg16;
  PyObject *arg17;
  PyObject *arg18;

  if (!PyArg_UnpackTuple(args, "cbs_type_i_feedback", 19, 19, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11, &arg12, &arg13, &arg14, &arg15, &arg16, &arg17, &arg18))
    return NULL;

  x0 = (float)_cffi_to_c_float(arg0);
  if (x0 == (float)-1 && PyErr_Occurred())
    return NULL;

  x1 = (float)_cffi_to_c_float(arg1);
  if (x1 == (float)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg9, (char **)&x9);
  if (datasize != 0) {
    x9 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg9, (char **)&x9,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x10 = _cffi_to_c_int(arg10, int);
  if (x10 == (int)-1 && PyErr_Occurred())
    return NULL;

  x11 = _cffi_to_c_int(arg11, int);
  if (x11 == (int)-1 && PyErr_Occurred())
    return NULL;

  x12 = _cffi_to_c_int(arg12, int);
  if (x12 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg13, (char **)&x13);
  if (datasize != 0) {
    x13 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg13, (char **)&x13,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg14, (char **)&x14);
  if (datasize != 0) {
    x14 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg14, (char **)&x14,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg15, (char **)&x15);
  if (datasize != 0) {
    x15 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg15, (char **)&x15,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg16, (char **)&x16);
  if (datasize != 0) {
    x16 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg16, (char **)&x16,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg17, (char **)&x17);
  if (datasize != 0) {
    x17 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg17, (char **)&x17,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg18, (char **)&x18);
  if (datasize != 0) {
    x18 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg18, (char **)&x18,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_type_i_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, x15, x16, x17, x18); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_type_i_feedback _cffi_d_cbs_type_i_feedback
#endif

static void _cffi_d_cbs_type_ii_feedback(float x0, int x1, int * x2, unsigned int * x3, unsigned int * x4, int x5, int x6, int x7, unsigned short * x8, unsigned short * x9, unsigned short * x10, unsigned short * x11)
{
  cbs_type_ii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_type_ii_feedback(PyObject *self, PyObject *args)
{
  float x0;
  int x1;
  int * x2;
  unsigned int * x3;
  unsigned int * x4;
  int x5;
  int x6;
  int x7;
  unsigned short * x8;
  unsigned short * x9;
  unsigned short * x10;
  unsigned short * x11;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;

  if (!PyArg_UnpackTuple(args, "cbs_type_ii_feedback", 12, 12, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11))
    return NULL;

  x0 = (float)_cffi_to_c_float(arg0);
  if (x0 == (float)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg9, (char **)&x9);
  if (datasize != 0) {
    x9 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg9, (char **)&x9,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg10, (char **)&x10);
  if (datasize != 0) {
    x10 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg10, (char **)&x10,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(31), arg11, (char **)&x11);
  if (datasize != 0) {
    x11 = ((size_t)datasize) <= 640 ? (unsigned short *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(31), arg11, (char **)&x11,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_type_ii_feedback(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_type_ii_feedback _cffi_d_cbs_type_ii_feedback
#endif

static void _cffi_d_cbs_unpack_clause_output(int x0, unsigned int * x1, unsigned int * x2, int x3)
{
  cbs_unpack_clause_output(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_cbs_unpack_clause_output(PyObject *self, PyObject *args)
{
  int x0;
  unsigned int * x1;
  unsigned int * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "cbs_unpack_clause_output", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { cbs_unpack_clause_output(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_cbs_unpack_clause_output _cffi_d_cbs_unpack_clause_output
#endif

static unsigned int _cffi_d_compareIndexedValues(void const * x0, void const * x1)
{
  return compareIndexedValues(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_compareIndexedValues(PyObject *self, PyObject *args)
{
  void const * x0;
  void const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "compareIndexedValues", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(10), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(10), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(10), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(10), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = compareIndexedValues(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_compareIndexedValues _cffi_d_compareIndexedValues
#endif

static unsigned int _cffi_d_compareints(void const * x0, void const * x1)
{
  return compareints(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_compareints(PyObject *self, PyObject *args)
{
  void const * x0;
  void const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "compareints", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(10), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(10), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(10), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(10), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = compareints(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_compareints _cffi_d_compareints
#endif

static int _cffi_d_generate_random(int x0)
{
  return generate_random(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_generate_random(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = generate_random(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_generate_random _cffi_d_generate_random
#endif

static void _cffi_const_myPrint(char *o)
{
  *(void(* *)(FILE *, char const *, ...))o = myPrint;
}

static void _cffi_d_pcg32_seed(uint64_t x0)
{
  pcg32_seed(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_pcg32_seed(PyObject *self, PyObject *arg0)
{
  uint64_t x0;

  x0 = _cffi_to_c_int(arg0, uint64_t);
  if (x0 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { pcg32_seed(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_pcg32_seed _cffi_d_pcg32_seed
#endif

static void _cffi_d_produce_example_by_clauses(int x0, unsigned int * x1, int x2, int x3, unsigned int * x4, int * x5, int x6, int x7, int x8, int x9)
{
  produce_example_by_clauses(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_produce_example_by_clauses(PyObject *self, PyObject *args)
{
  int x0;
  unsigned int * x1;
  int x2;
  int x3;
  unsigned int * x4;
  int * x5;
  int x6;
  int x7;
  int x8;
  int x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "produce_example_by_clauses", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  x8 = _cffi_to_c_int(arg8, int);
  if (x8 == (int)-1 && PyErr_Occurred())
    return NULL;

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { produce_example_by_clauses(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_produce_example_by_clauses _cffi_d_produce_example_by_clauses
#endif

static void _cffi_d_produce_example_by_combined_clauses(int x0, unsigned int * x1, int x2, int x3, unsigned int * x4, int * x5, int x6, int x7, unsigned int * x8, int * x9, int x10, int x11, int x12, int x13)
{
  produce_example_by_combined_clauses(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_produce_example_by_combined_clauses(PyObject *self, PyObject *args)
{
  int x0;
  unsigned int * x1;
  int x2;
  int x3;
  unsigned int * x4;
  int * x5;
  int x6;
  int x7;
  unsigned int * x8;
  int * x9;
  int x10;
  int x11;
  int x12;
  int x13;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;
  PyObject *arg12;
  PyObject *arg13;

  if (!PyArg_UnpackTuple(args, "produce_example_by_combined_clauses", 14, 14, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11, &arg12, &arg13))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg9, (char **)&x9);
  if (datasize != 0) {
    x9 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg9, (char **)&x9,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x10 = _cffi_to_c_int(arg10, int);
  if (x10 == (int)-1 && PyErr_Occurred())
    return NULL;

  x11 = _cffi_to_c_int(arg11, int);
  if (x11 == (int)-1 && PyErr_Occurred())
    return NULL;

  x12 = _cffi_to_c_int(arg12, int);
  if (x12 == (int)-1 && PyErr_Occurred())
    return NULL;

  x13 = _cffi_to_c_int(arg13, int);
  if (x13 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { produce_example_by_combined_clauses(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_produce_example_by_combined_clauses _cffi_d_produce_example_by_combined_clauses
#endif

static void _cffi_d_produce_example_by_documents(unsigned int * x0, int x1, unsigned int * x2, unsigned int * x3, int x4, unsigned int * x5, unsigned int * x6, int x7, unsigned int * x8, unsigned int * x9, int x10, unsigned int * x11, int x12, int x13, int x14, int x15, int x16)
{
  produce_example_by_documents(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, x15, x16);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_produce_example_by_documents(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  unsigned int * x2;
  unsigned int * x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  int x7;
  unsigned int * x8;
  unsigned int * x9;
  int x10;
  unsigned int * x11;
  int x12;
  int x13;
  int x14;
  int x15;
  int x16;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;
  PyObject *arg12;
  PyObject *arg13;
  PyObject *arg14;
  PyObject *arg15;
  PyObject *arg16;

  if (!PyArg_UnpackTuple(args, "produce_example_by_documents", 17, 17, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11, &arg12, &arg13, &arg14, &arg15, &arg16))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg9, (char **)&x9);
  if (datasize != 0) {
    x9 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg9, (char **)&x9,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x10 = _cffi_to_c_int(arg10, int);
  if (x10 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg11, (char **)&x11);
  if (datasize != 0) {
    x11 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg11, (char **)&x11,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x12 = _cffi_to_c_int(arg12, int);
  if (x12 == (int)-1 && PyErr_Occurred())
    return NULL;

  x13 = _cffi_to_c_int(arg13, int);
  if (x13 == (int)-1 && PyErr_Occurred())
    return NULL;

  x14 = _cffi_to_c_int(arg14, int);
  if (x14 == (int)-1 && PyErr_Occurred())
    return NULL;

  x15 = _cffi_to_c_int(arg15, int);
  if (x15 == (int)-1 && PyErr_Occurred())
    return NULL;

  x16 = _cffi_to_c_int(arg16, int);
  if (x16 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { produce_example_by_documents(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, x15, x16); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_produce_example_by_documents _cffi_d_produce_example_by_documents
#endif

static void _cffi_d_produce_example_by_features(int x0, unsigned int * x1, int * x2, int x3, int x4)
{
  produce_example_by_features(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_produce_example_by_features(PyObject *self, PyObject *args)
{
  int x0;
  unsigned int * x1;
  int * x2;
  int x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "produce_example_by_features", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { produce_example_by_features(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_produce_example_by_features _cffi_d_produce_example_by_features
#endif

static void _cffi_d_produce_example_per_target_by_document(unsigned int * x0, int x1, unsigned int * x2, unsigned int * x3, int x4, unsigned int * x5, unsigned int * x6, int x7, unsigned int * x8, int x9, int x10, int x11)
{
  produce_example_per_target_by_document(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_produce_example_per_target_by_document(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  int x1;
  unsigned int * x2;
  unsigned int * x3;
  int x4;
  unsigned int * x5;
  unsigned int * x6;
  int x7;
  unsigned int * x8;
  int x9;
  int x10;
  int x11;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;
  PyObject *arg10;
  PyObject *arg11;

  if (!PyArg_UnpackTuple(args, "produce_example_per_target_by_document", 12, 12, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9, &arg10, &arg11))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  x10 = _cffi_to_c_int(arg10, int);
  if (x10 == (int)-1 && PyErr_Occurred())
    return NULL;

  x11 = _cffi_to_c_int(arg11, int);
  if (x11 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { produce_example_per_target_by_document(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_produce_example_per_target_by_document _cffi_d_produce_example_per_target_by_document
#endif

static void _cffi_d_store_feature_to_X(int x0, int x1, unsigned int * x2, int x3)
{
  store_feature_to_X(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_store_feature_to_X(PyObject *self, PyObject *args)
{
  int x0;
  int x1;
  unsigned int * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "store_feature_to_X", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { store_feature_to_X(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_store_feature_to_X _cffi_d_store_feature_to_X
#endif

static void _cffi_d_tmu_encode(unsigned int * x0, unsigned int * x1, int x2, int x3, int x4, int x5, int x6, int x7, int x8, int x9)
{
  tmu_encode(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_tmu_encode(PyObject *self, PyObject *args)
{
  unsigned int * x0;
  unsigned int * x1;
  int x2;
  int x3;
  int x4;
  int x5;
  int x6;
  int x7;
  int x8;
  int x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "tmu_encode", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  x8 = _cffi_to_c_int(arg8, int);
  if (x8 == (int)-1 && PyErr_Occurred())
    return NULL;

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { tmu_encode(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_tmu_encode _cffi_d_tmu_encode
#endif

static void _cffi_d_wb_decrement(int * x0, int x1, unsigned int * x2, float x3, unsigned int * x4, unsigned int x5)
{
  wb_decrement(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_wb_decrement(PyObject *self, PyObject *args)
{
  int * x0;
  int x1;
  unsigned int * x2;
  float x3;
  unsigned int * x4;
  unsigned int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "wb_decrement", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = (float)_cffi_to_c_float(arg3);
  if (x3 == (float)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, unsigned int);
  if (x5 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { wb_decrement(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_wb_decrement _cffi_d_wb_decrement
#endif

static void _cffi_d_wb_increment(int * x0, int x1, unsigned int * x2, float x3, unsigned int * x4, unsigned int x5)
{
  wb_increment(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_wb_increment(PyObject *self, PyObject *args)
{
  int * x0;
  int x1;
  unsigned int * x2;
  float x3;
  unsigned int * x4;
  unsigned int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "wb_increment", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(25), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(25), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = (float)_cffi_to_c_float(arg3);
  if (x3 == (float)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(4), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(4), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, unsigned int);
  if (x5 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { wb_increment(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_wb_increment _cffi_d_wb_increment
#endif

static void _cffi_d_xorshift128p_seed(uint64_t x0)
{
  xorshift128p_seed(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_xorshift128p_seed(PyObject *self, PyObject *arg0)
{
  uint64_t x0;

  x0 = _cffi_to_c_int(arg0, uint64_t);
  if (x0 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { xorshift128p_seed(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_xorshift128p_seed _cffi_d_xorshift128p_seed
#endif

static int *_cffi_var_enable_printing(void)
{
  return &(enable_printing);
}

static const struct _cffi_global_s _cffi_globals[] = {
  { "cb_calculate_clause_outputs_incremental", (void *)_cffi_f_cb_calculate_clause_outputs_incremental, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 261), (void *)_cffi_d_cb_calculate_clause_outputs_incremental },
  { "cb_calculate_clause_outputs_incremental_batch", (void *)_cffi_f_cb_calculate_clause_outputs_incremental_batch, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 249), (void *)_cffi_d_cb_calculate_clause_outputs_incremental_batch },
  { "cb_calculate_clause_outputs_patchwise", (void *)_cffi_f_cb_calculate_clause_outputs_patchwise, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 118), (void *)_cffi_d_cb_calculate_clause_outputs_patchwise },
  { "cb_calculate_clause_outputs_predict", (void *)_cffi_f_cb_calculate_clause_outputs_predict, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 118), (void *)_cffi_d_cb_calculate_clause_outputs_predict },
  { "cb_calculate_clause_outputs_update", (void *)_cffi_f_cb_calculate_clause_outputs_update, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 127), (void *)_cffi_d_cb_calculate_clause_outputs_update },
  { "cb_calculate_literal_frequency", (void *)_cffi_f_cb_calculate_literal_frequency, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 144), (void *)_cffi_d_cb_calculate_literal_frequency },
  { "cb_get_literals", (void *)_cffi_f_cb_get_literals, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 297), (void *)_cffi_d_cb_get_literals },
  { "cb_included_literals", (void *)_cffi_f_cb_included_literals, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 137), (void *)_cffi_d_cb_included_literals },
  { "cb_initialize_incremental_clause_calculation", (void *)_cffi_f_cb_initialize_incremental_clause_calculation, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 287), (void *)_cffi_d_cb_initialize_incremental_clause_calculation },
  { "cb_number_of_include_actions", (void *)_cffi_f_cb_number_of_include_actions, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 3), (void *)_cffi_d_cb_number_of_include_actions },
  { "cb_type_i_feedback", (void *)_cffi_f_cb_type_i_feedback, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 232), (void *)_cffi_d_cb_type_i_feedback },
  { "cb_type_ii_feedback", (void *)_cffi_f_cb_type_ii_feedback, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 199), (void *)_cffi_d_cb_type_ii_feedback },
  { "cb_type_iii_feedback", (void *)_cffi_f_cb_type_iii_feedback, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 270), (void *)_cffi_d_cb_type_iii_feedback },
  { "cbs_calculate_clause_outputs_predict", (void *)_cffi_f_cbs_calculate_clause_outputs_predict, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 152), (void *)_cffi_d_cbs_calculate_clause_outputs_predict },
  { "cbs_calculate_clause_outputs_predict_packed_X", (void *)_cffi_f_cbs_calculate_clause_outputs_predict_packed_X, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 152), (void *)_cffi_d_cbs_calculate_clause_outputs_predict_packed_X },
  { "cbs_calculate_clause_outputs_update", (void *)_cffi_f_cbs_calculate_clause_outputs_update, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 223), (void *)_cffi_d_cbs_calculate_clause_outputs_update },
  { "cbs_pack_X", (void *)_cffi_f_cbs_pack_X, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 52), (void *)_cffi_d_cbs_pack_X },
  { "cbs_prepare_Xi", (void *)_cffi_f_cbs_prepare_Xi, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 160), (void *)_cffi_d_cbs_prepare_Xi },
  { "cbs_restore_Xi", (void *)_cffi_f_cbs_restore_Xi, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 160), (void *)_cffi_d_cbs_restore_Xi },
  { "cbs_type_i_feedback", (void *)_cffi_f_cbs_type_i_feedback, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 17), (void *)_cffi_d_cbs_type_i_feedback },
  { "cbs_type_ii_feedback", (void *)_cffi_f_cbs_type_ii_feedback, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 38), (void *)_cffi_d_cbs_type_ii_feedback },
  { "cbs_unpack_clause_output", (void *)_cffi_f_cbs_unpack_clause_output, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 109), (void *)_cffi_d_cbs_unpack_clause_output },
  { "compareIndexedValues", (void *)_cffi_f_compareIndexedValues, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 9), (void *)_cffi_d_compareIndexedValues },
  { "compareints", (void *)_cffi_f_compareints, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 9), (void *)_cffi_d_compareints },
  { "enable_printing", (void *)_cffi_var_enable_printing, _CFFI_OP(_CFFI_OP_GLOBAL_VAR_F, 1), (void *)0 },
  { "generate_random", (void *)_cffi_f_generate_random, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_O, 0), (void *)_cffi_d_generate_random },
  { "myPrint", (void *)_cffi_const_myPrint, _CFFI_OP(_CFFI_OP_CONSTANT, 307), (void *)0 },
  { "pcg32_seed", (void *)_cffi_f_pcg32_seed, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_O, 115), (void *)_cffi_d_pcg32_seed },
  { "produce_example_by_clauses", (void *)_cffi_f_produce_example_by_clauses, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 81), (void *)_cffi_d_produce_example_by_clauses },
  { "produce_example_by_combined_clauses", (void *)_cffi_f_produce_example_by_combined_clauses, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 93), (void *)_cffi_d_produce_example_by_combined_clauses },
  { "produce_example_by_documents", (void *)_cffi_f_produce_example_by_documents, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 180), (void *)_cffi_d_produce_example_by_documents },
  { "produce_example_by_features", (void *)_cffi_f_produce_example_by_features, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 74), (void *)_cffi_d_produce_example_by_features },
  { "produce_example_per_target_by_document", (void *)_cffi_f_produce_example_per_target_by_document, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 166), (void *)_cffi_d_produce_example_per_target_by_document },
  { "store_feature_to_X", (void *)_cffi_f_store_feature_to_X, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 68), (void *)_cffi_d_store_feature_to_X },
  { "tmu_encode", (void *)_cffi_f_tmu_encode, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 211), (void *)_cffi_d_tmu_encode },
  { "wb_decrement", (void *)_cffi_f_wb_decrement, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 60), (void *)_cffi_d_wb_decrement },
  { "wb_increment", (void *)_cffi_f_wb_increment, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_V, 60), (void *)_cffi_d_wb_increment },
  { "xorshift128p_seed", (void *)_cffi_f_xorshift128p_seed, _CFFI_OP(_CFFI_OP_CPYTHON_BLTN_O, 115), (void *)_cffi_d_xorshift128p_seed },
};

static const struct _cffi_struct_union_s _cffi_struct_unions[] = {
  { "_IO_FILE", 304, _CFFI_F_OPAQUE,
    (size_t)-1, -1, -1, 0 /* opaque */ },
};

static const struct _cffi_typename_s _cffi_typenames[] = {
  { "FILE", 304 },
};

static const struct _cffi_type_context_s _cffi_type_context = {
  _cffi_types,
  _cffi_globals,
  NULL,  /* no fields */
  _cffi_struct_unions,
  NULL,  /* no enums */
  _cffi_typenames,
  38,  /* num_globals */
  1,  /* num_struct_unions */
  0,  /* num_enums */
  1,  /* num_typenames */
  NULL,  /* no includes */
  309,  /* num_types */
  0,  /* flags */
};

#ifdef __GNUC__
#  pragma GCC visibility push(default)  /* for -fvisibility= */
#endif

#ifdef PYPY_VERSION
PyMODINIT_FUNC
_cffi_pypyinit_tmulib(const void *p[])
{
    p[0] = (const void *)0x2601;
    p[1] = &_cffi_type_context;
#if PY_MAJOR_VERSION >= 3
    return NULL;
#endif
}
#  ifdef _MSC_VER
     PyMODINIT_FUNC
#  if PY_MAJOR_VERSION >= 3
     PyInit_tmulib(void) { return NULL; }
#  else
     inittmulib(void) { }
#  endif
#  endif
#elif PY_MAJOR_VERSION >= 3
PyMODINIT_FUNC
PyInit_tmulib(void)
{
  return _cffi_init("tmu.tmulib", 0x2601, &_cffi_type_context);
}
#else
PyMODINIT_FUNC
inittmulib(void)
{
  _cffi_init("tmu.tmulib", 0x2601, &_cffi_type_context);
}
#endif

#ifdef __GNUC__
#  pragma GCC visibility pop
#endif
