import attr
from attr.validators import in_, instance_of
from torch import nn
from torch.nn import functional as F
from torchvision.models import alexnet, vgg11

from utils import AbstractConf


class SimpleNN(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int, simple_hidden_dim: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes
        self.hidden_dim = simple_hidden_dim

        self.layer1 = nn.Sequential(
            nn.Linear(self.input_channels * self.input_size * self.input_size, self.hidden_dim),
            nn.ELU(),
        )

        self.out = nn.Linear(self.hidden_dim, num_classes)

    def forward(self, x):
        x = x.view(x.size(0), -1)
        x = self.layer1(x)
        output = self.out(x)
        return output


class LeNet(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes

        self.features = nn.Sequential(
            nn.Conv2d(input_channels, 6, kernel_size=5, padding=2),
            nn.ReLU(),
            nn.MaxPool2d(kernel_size=2, stride=2),
            nn.Conv2d(6, 16, kernel_size=5),
            nn.ReLU(),
            nn.MaxPool2d(kernel_size=2, stride=2)
        )

        self.classifier = nn.Sequential(
            nn.Linear(16 * (input_size // 4 - 2) * (input_size // 4 - 2), 120),
            nn.ReLU(),
            nn.Linear(120, 84),
            nn.ReLU(),
            nn.Linear(84, num_classes)
        )

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        y = self.classifier(x)
        return y


class AlexNet(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes
        self.net = alexnet(num_classes=num_classes)

    def forward(self, x):
        y = self.net(x)
        return y


class VGG(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes
        self.net = vgg11(num_classes=num_classes)

    def forward(self, x):
        y = self.net(x)
        return y


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion * planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNetAll(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(ResNetAll, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512 * block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


def ResNet18(num_classes=10):
    return ResNetAll(BasicBlock, [2, 2, 2, 2], num_classes=num_classes)


class ResNet(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes
        self.net = ResNet18(num_classes=num_classes)

    def forward(self, x):
        y = self.net(x)
        return y

class PreActBlock(nn.Module):
    """
    Pre-activation version of the BasicBlock for Resnets.
    Arguments:
        in_planes (int): number of input planes.
        planes (int): number of output filters.
        stride (int): stride of convolution.
    """
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)

        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, 'shortcut') else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out += shortcut
        return out


class PreActBottleneck(nn.Module):
    """
    Pre-activation version of the original Bottleneck module for Resnets.
    Arguments:
        in_planes (int): number of input planes.
        planes (int): number of output filters.
        stride (int): stride of convolution.
    """
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBottleneck, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)

        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, 'shortcut') else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out = self.conv3(F.relu(self.bn3(out)))
        out += shortcut
        return out


class PreActResNet(nn.Module):
    """
    Pre-activation Resnet model
    """
    def __init__(self, block, num_blocks, num_classes=10):
        super(PreActResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.bn = nn.BatchNorm2d(512 * block.expansion)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.relu(self.bn(out))
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


def preact_resnet(name, num_classes=10, pretrained=False, device='cpu'):
    """
    Returns suitable Resnet model from its name.
    Arguments:
        name (str): name of resnet architecture.
        num_classes (int): number of target classes.
        pretrained (bool): whether to use a pretrained model.
        device (str or torch.device): device to work on.
    Returns:
        torch.nn.Module.
    """
    if name == 'preact-resnet18':
        return PreActResNet(PreActBlock, [2, 2, 2, 2], num_classes=num_classes)
    elif name == 'preact-resnet34':
        return PreActResNet(PreActBlock, [3, 4, 6, 3], num_classes=num_classes)
    elif name == 'preact-resnet50':
        return PreActResNet(PreActBottleneck, [3, 4, 6, 3], num_classes=num_classes)
    elif name == 'preact-resnet101':
        return PreActResNet(PreActBottleneck, [3, 4, 23, 3], num_classes=num_classes)
    raise ValueError('Only preact-resnet18, preact-resnet34, preact-resnet50 and preact-resnet101 are supported!')
    return

class PreActResNet18(nn.Module):
    def __init__(self, input_size: int, input_channels: int, num_classes: int):
        super().__init__()
        self.input_size = input_size
        self.input_channels = input_channels
        self.num_classes = num_classes
        self.net = PreActResNet(PreActBlock, [2, 2, 2, 2])

    def forward(self, x):
        y = self.net(x)
        return y

@attr.s
class ModelHpars(AbstractConf):
    OPTIONS = {"simple": SimpleNN, "lenet": LeNet, "alexnet": AlexNet, "vgg": VGG, "resnet": ResNet, "preactresnet":PreActResNet18}
    name = attr.ib(validator=in_(OPTIONS))
    input_size = attr.ib(validator=instance_of(int))
    input_channels = attr.ib(validator=instance_of(int))
    num_classes = attr.ib(validator=instance_of(int))
    simple_hidden_dim = attr.ib(default=256, validator=instance_of(int))
